/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QDir>
#include <QtCore/QFileInfoList>


#include <core_api/AppContext.h>
#include <gobjects/BioStruct3DObject.h>
#include <util_tasks/LoadDocumentTask.h>
#include <core_api/DocumentFormats.h>

#include "BioStruct3DObjectTests.h"
#include "DocumentModelTests.h"




namespace GB2 { 

#define	VALUE_ATTR			"value"
#define	OBJ_ATTR			"obj"
#define	ATOM_ID_ATTR		"atom-id"
#define	X_COORD_ATTR		"x"
#define	Y_COORD_ATTR		"y"
#define Z_COORD_ATTR		"z"



void GTest_BioStruct3DNumberOfAtoms::init(XMLTestFormat *tf, const QDomElement& el) {
	Q_UNUSED(tf);

	objContextName = el.attribute(OBJ_ATTR);
	if (objContextName.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
		return;
	}

	QString v = el.attribute(VALUE_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
		return;
	} 

	bool ok = false;
	numAtoms = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
	}

}

Task::ReportResult GTest_BioStruct3DNumberOfAtoms::report()
{

	GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL){
		stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
		return ReportResult_Finished;  
	}

	BioStruct3DObject * biostructObj = qobject_cast<BioStruct3DObject*>(obj);
	if(biostructObj==NULL){
		stateInfo.error = GTest::tr("can't cast to biostruct3d object from: %1").arg(obj->getGObjectName());
		return ReportResult_Finished;
	}

	
    
    
    int tmpNumAtoms = biostructObj->getBioStruct3D().getNumberOfAtoms();
	
	if(tmpNumAtoms != numAtoms){
		stateInfo.error = QString("number of atoms does not match: %1, expected %2 ").arg(tmpNumAtoms).arg(numAtoms);
	}

	return ReportResult_Finished;
}

///////////////////////////////////////////////////////////////////////////////////////////

void GTest_BioStruct3DNumberOfResidues::init(XMLTestFormat *tf, const QDomElement& el) {
	Q_UNUSED(tf);

	objContextName = el.attribute(OBJ_ATTR);
	if (objContextName.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
		return;
	}

	QString v = el.attribute(VALUE_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
		return;
	} 

	bool ok = false;
	numResidues = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
	}

}

Task::ReportResult GTest_BioStruct3DNumberOfResidues::report()
{

	GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL){
		stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
		return ReportResult_Finished;  
	}

	BioStruct3DObject * biostructObj = qobject_cast<BioStruct3DObject*>(obj);
	if(biostructObj==NULL){
		stateInfo.error = GTest::tr("can't cast to biostruct3d object from: %1").arg(obj->getGObjectName());
		return ReportResult_Finished;
	}

	int tmpNumResidues = biostructObj->getBioStruct3D().getNumberOfResidues();

	if(tmpNumResidues != numResidues){
		stateInfo.error = QString("number of amino acid residues does not match: %1, expected %2 ").arg(tmpNumResidues).arg(numResidues);
	}

	return ReportResult_Finished;
}

///////////////////////////////////////////////////////////////////////////////////////////

void GTest_BioStruct3DNumberOfChains::init(XMLTestFormat *tf, const QDomElement& el) {
	Q_UNUSED(tf);

	objContextName = el.attribute(OBJ_ATTR);
	if (objContextName.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
		return;
	}

	QString v = el.attribute(VALUE_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
		return;
	} 

	bool ok = false;
	numChains = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, int required").arg(VALUE_ATTR);
	}

}

Task::ReportResult GTest_BioStruct3DNumberOfChains::report()
{

	GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL){
		stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
		return ReportResult_Finished;  
	}

	BioStruct3DObject * biostructObj = qobject_cast<BioStruct3DObject*>(obj);
	if(biostructObj==NULL){
		stateInfo.error = GTest::tr("can't cast to biostruct3d object from: %1").arg(obj->getGObjectName());
		return ReportResult_Finished;
	}

	int tmpNumChains = biostructObj->getBioStruct3D().moleculeMap.size();

	if(tmpNumChains != numChains) {
		stateInfo.error = QString("number of polymer chains does not match: %1, expected %2 ").arg(tmpNumChains).arg(numChains);
	}

	return ReportResult_Finished;
}

///////////////////////////////////////////////////////////////////////////////////////////

void GTest_BioStruct3DAtomCoordinates::init(XMLTestFormat *tf, const QDomElement& el) {
	Q_UNUSED(tf);

	objContextName = el.attribute(OBJ_ATTR);
	if (objContextName.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
		return;
	}
	
	// atom id
	QString v = el.attribute(ATOM_ID_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(ATOM_ID_ATTR);
		return;
	} 
	bool ok = false;
	atomId = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, int required").arg(ATOM_ID_ATTR);
	}


	// x coordinate
	v = el.attribute(X_COORD_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(X_COORD_ATTR);
		return;
	} 
	ok = false;
	x = v.toDouble(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, double required").arg(X_COORD_ATTR);
	}
	
	// y coordinate
	v = el.attribute(Y_COORD_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(Y_COORD_ATTR);
		return;
	} 
	ok = false;
	y = v.toDouble(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, double required").arg(Y_COORD_ATTR);
	}

	// z coordinate
	v = el.attribute(Z_COORD_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(Z_COORD_ATTR);
		return;
	} 
	ok = false;
	z = v.toDouble(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, double required").arg(Z_COORD_ATTR);
	}

	
}

Task::ReportResult GTest_BioStruct3DAtomCoordinates::report()
{

	GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL){
		stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
		return ReportResult_Finished;  
	}

	BioStruct3DObject * biostructObj = qobject_cast<BioStruct3DObject*>(obj);
	if(biostructObj==NULL){
		stateInfo.error = GTest::tr("can't cast to biostruct3d object from: %1").arg(obj->getGObjectName());
		return ReportResult_Finished;
	}

	
	SharedAtom atom = biostructObj->getBioStruct3D().getAtomById(atomId);
    
    if (atom == NULL) {
        stateInfo.error = QString("atom with index = %1 not found").arg(atomId);
        return ReportResult_Finished;
    }
	
    Vector3D coords(x,y,z);
	Vector3D tmpCoords = atom->coord3d;
	if (coords != tmpCoords) {
		stateInfo.error = QString("atom coords not match: (%1,%2,%3)").arg(tmpCoords.x).arg(tmpCoords.y).arg(tmpCoords.z) + 
		QString(", expected (%1,%2,%3) ").arg(x).arg(y).arg(z);
	}
	return ReportResult_Finished;

	
	
}

///////////////////////////////////////////////////////////////////////////////////////////

void GTest_BioStruct3DAtomSequenceId::init(XMLTestFormat *tf, const QDomElement& el) {
	Q_UNUSED(tf);

	objContextName = el.attribute(OBJ_ATTR);
	if (objContextName.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
		return;
	}

	// atom id
	QString v = el.attribute(ATOM_ID_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(ATOM_ID_ATTR);
		return;
	} 
	bool ok = false;
	atomId = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, int required").arg(ATOM_ID_ATTR);
	}

	//sequence id
	v = el.attribute(VALUE_ATTR);
	if (v.isEmpty()) {
		stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
		return;
	} 
	ok = false;
	seqId = v.toInt(&ok);
	if (!ok) {
		stateInfo.error = GTest::tr("invalid value type %1, int required").arg(VALUE_ATTR);
	}


	
}


Task::ReportResult GTest_BioStruct3DAtomSequenceId::report()
{

	GObject *obj = getContext<GObject>(this, objContextName);
	if(obj==NULL){
		stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
		return ReportResult_Finished;  
	}

	BioStruct3DObject * biostructObj = qobject_cast<BioStruct3DObject*>(obj);
	if(biostructObj==NULL){
		stateInfo.error = GTest::tr("can't cast to biostruct3d object from: %1").arg(obj->getGObjectName());
		return ReportResult_Finished;
	}

	SharedAtom atom = biostructObj->getBioStruct3D().getAtomById(atomId);
    if (atom == NULL) {
        stateInfo.error = QString("atom with index = %1 not found").arg(atomId);
        return ReportResult_Finished;
    }

			
	int tmpId = atom->residueIndex;
					
	if (seqId != tmpId) {
		stateInfo.error = QString("atom with id=%1 sequenceId does not match: %2, expected %3").arg(atomId).arg(tmpId).arg(seqId);
	}

    return ReportResult_Finished;


}

///////////////////////////////////////////////////////////////////////////////////////////

#define DIR_NAME_ENV "DIR_WITH_PDB_FILES"

void GTest_PDBFormatStressTest::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);
    Q_UNUSED(el);
    
    quint32 mask = ~TaskFlag_StopOnSubtaskError;  
    flags = flags & mask;
    QString dirName = getEnv()->getVar(DIR_NAME_ENV);
    
    QDir dir(dirName);
    if (!dir.exists()) {
        stateInfo.error = GTest::tr("Cannot_find_the_directory %1").arg(dirName);
        return;
    }

    dir.setFilter(QDir::Files);
    QFileInfoList fileList  = dir.entryInfoList(); 

    if (fileList.empty()) {
        stateInfo.error = GTest::tr("Directory %1 is_empty").arg(dirName);
        return;
    }
    
    IOAdapterId         ioId(BaseIOAdapters::LOCAL_FILE);
    IOAdapterFactory*   iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioId);
    DocumentFormatId    format = BaseDocumentFormats::PLAIN_PDB;
    
    for (int i = 0; i < fileList.size(); ++i) {
        QFileInfo fileInfo = fileList.at(i);
        LoadDocumentTask* task = new LoadDocumentTask(format, fileInfo.absoluteFilePath(), iof);
        addSubTask(task);
        fileNames.insert(task, fileInfo.fileName());
    }

}


Task::ReportResult GTest_PDBFormatStressTest::report()
{

    foreach (Task* task, getSubtasks()) {
        if (task->hasErrors()) {
            stateInfo.error += fileNames.value(task) + "(" + task->getError() + ");   ";
        }
    }
    
    return ReportResult_Finished;

}

QList<Task*> GTest_PDBFormatStressTest::onSubTaskFinished( Task* subTask )
{
    subTask->cleanup();
    QList<Task*> lst;
    return lst;
}
///////////////////////////////////////////////////////////////////////////////////////////

QList<XMLTestFactory*> BioStruct3DObjectTests::createTestFactories()
{
	QList<XMLTestFactory*> res;
	res.append(GTest_BioStruct3DNumberOfAtoms::createFactory());
	res.append(GTest_BioStruct3DNumberOfChains::createFactory());
	res.append(GTest_BioStruct3DNumberOfResidues::createFactory());
	res.append(GTest_BioStruct3DAtomCoordinates::createFactory());
	res.append(GTest_BioStruct3DAtomSequenceId::createFactory());
    res.append(GTest_PDBFormatStressTest::createFactory());
    
	return res;

}


} //namespace
