/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocumentModelTests.h"
#include "DNASequenceObjectTests.h"
#include "gobjects/MAlignmentObject.h"
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/DocumentModel.h>
#include <core_api/GObject.h>

#include <util_tasks/LoadDocumentTask.h>
#include <gobjects/DNASequenceObject.h>

namespace GB2 {

/* TRANSLATOR GB2::GTest */

#define DOC_ATTR   "doc"
#define VALUE_ATTR  "value"
#define START_ATTR  "seqstart"
#define OBJ_ATTR    "obj"
#define SEQNAME_ATTR  "seqname"

void GTest_DNASequenceSize::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    QString v = el.attribute(VALUE_ATTR);
    if (v.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    bool ok = false;
    seqSize = v.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
    }

}

Task::ReportResult GTest_DNASequenceSize::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    DNASequenceObject * mySequence = qobject_cast<DNASequenceObject*>(obj);
    if(mySequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
    int tempLength=mySequence->getDNASequence().length();
    if(tempLength != seqSize){
        stateInfo.error = QString("sequence size not matched: %1, expected %2 ").arg(tempLength).arg(seqSize);
    }
    return ReportResult_Finished;
}

void GTest_DNASequenceAlphabet::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    alphabetId = el.attribute(VALUE_ATTR);
    if (alphabetId.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    
}

Task::ReportResult GTest_DNASequenceAlphabet::report() {
    GObject *obj = getContext<GObject>(this,objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    DNASequenceObject * mySequence = qobject_cast<DNASequenceObject*>(obj);
    if(mySequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(name);
        return ReportResult_Finished;
    }

    DNAAlphabet *tempAlphabet = mySequence->getAlphabet();
    assert(tempAlphabet!=NULL);

    if (tempAlphabet->getId() != alphabetId){   
        stateInfo.error = QString("Alphabet not matched: %1, expected %2 ").arg(tempAlphabet->getId()).arg(alphabetId);
    }

    return ReportResult_Finished;
}

void GTest_DNASequencePart::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    subseq = el.attribute(VALUE_ATTR).toAscii();
    if (subseq.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    
	QString p = el.attribute(START_ATTR);
    if (p.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(START_ATTR);
        return;
    } 
    bool ok = false;
    startPos= p.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(START_ATTR);
    }
}

Task::ReportResult GTest_DNASequencePart::report() {
    GObject *obj = getContext<GObject>(this,objContextName);
    if (obj==NULL) {
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    DNASequenceObject * objSequence = qobject_cast<DNASequenceObject*>(obj);
    if (objSequence==NULL) {
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
    if (objSequence->getSequence().length() < startPos + subseq.length()) {
        stateInfo.error = QString("sequence size is less that region end: size=%1, region-end=%").arg(objSequence->getSequence().size(), startPos + subseq.length());
        return ReportResult_Finished;
    }
    QByteArray objSubSeq = objSequence->getSequence().mid(startPos, subseq.length());
    if (!objSequence->getAlphabet()->isCaseSensitive()) {
        subseq = subseq.toUpper();
    }
	if (objSubSeq != subseq){	
	    stateInfo.error = QString("region not matched: %1, expected %2").arg(objSubSeq.constData()).arg(subseq.constData());
	    return ReportResult_Finished;
	}
    return ReportResult_Finished;
}
void GTest_DNASequenceAlphabetType::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

	QString v = el.attribute(VALUE_ATTR);
    if (v.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
	if (v == "DNAAlphabet_RAW"){
		alphabetType = DNAAlphabet_RAW;
		return;
	}
	if (v == "DNAAlphabet_NUCL"){
		alphabetType = DNAAlphabet_NUCL;
		return;
	}
	if (v == "DNAAlphabet_AMINO"){
		alphabetType = DNAAlphabet_AMINO;
		return;
	}
	stateInfo.error = GTest::tr("alphabetType not set %1").arg(VALUE_ATTR);
    return;
	
}
Task::ReportResult GTest_DNASequenceAlphabetType::report() {
    GObject *obj = getContext<GObject>(this,objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

	DNASequenceObject * mySequence = qobject_cast<DNASequenceObject*>(obj);
	if(mySequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
	DNAAlphabet *tempAlphabet = mySequence->getAlphabet();
	if( tempAlphabet->getType() != alphabetType){	
		stateInfo.error = QString("Alphabet type not matched: %1, expected %2").arg(tempAlphabet->getType()).arg(alphabetType);
	}
    return ReportResult_Finished;
}
void GTest_DNASequenceAlphabetId::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

	alpId = el.attribute(VALUE_ATTR);
    if (alpId.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    return;
	
}
Task::ReportResult GTest_DNASequenceAlphabetId::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

	DNASequenceObject * mySequence = qobject_cast<DNASequenceObject*>(obj);
	if(mySequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
	DNAAlphabet *tempAlphabet = mySequence->getAlphabet();
	if( tempAlphabet->getId() != alpId){	
		stateInfo.error = QString("Alphabet id not matched: %1 expected %2").arg(tempAlphabet->getId()).arg(alpId);
	}
    return ReportResult_Finished;
}

//----------------------------------------------------------
void GTest_DNACompeareSequencesNamesInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareSequencesNamesInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    DNASequenceObject * mySequence;
    DNASequenceObject * mySequence2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType()== GObjectTypes::DNA_SEQUENCE)&&(obj2->getGObjectType()== GObjectTypes::DNA_SEQUENCE)){
            mySequence = qobject_cast<DNASequenceObject*>(obj);
            if(mySequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            mySequence2 = qobject_cast<DNASequenceObject*>(obj2);
            if(mySequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            if(mySequence->getGObjectName()!=mySequence2->getGObjectName()){
                stateInfo.error = GTest::tr("Name of object in position %1 not matched").arg(i);
                return ReportResult_Finished;
            }
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//----------------------------------------------------------
void GTest_DNACompeareSequencesInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareSequencesInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    DNASequenceObject * mySequence;
    DNASequenceObject * mySequence2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType()== GObjectTypes::DNA_SEQUENCE)&&(obj2->getGObjectType()== GObjectTypes::DNA_SEQUENCE)){
            mySequence = qobject_cast<DNASequenceObject*>(obj);
            if(mySequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            mySequence2 = qobject_cast<DNASequenceObject*>(obj2);
            if(mySequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            if(mySequence->getSequence()!=mySequence2->getSequence()){
                stateInfo.error = GTest::tr("Sequences of object in position %1 not matched").arg(i);
                return ReportResult_Finished;
            }
        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//----------------------------------------------------------
void GTest_DNACompeareSequencesAlphabetsInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareSequencesAlphabetsInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    DNASequenceObject * mySequence;
    DNASequenceObject * mySequence2;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if((obj->getGObjectType()== GObjectTypes::DNA_SEQUENCE)&&(obj2->getGObjectType()== GObjectTypes::DNA_SEQUENCE)){
            mySequence = qobject_cast<DNASequenceObject*>(obj);
            if(mySequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            mySequence2 = qobject_cast<DNASequenceObject*>(obj2);
            if(mySequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
            if(mySequence->getAlphabet()!=mySequence2->getAlphabet()){
                stateInfo.error = GTest::tr("Alphabets of object in position %1 not matched").arg(i);
                return ReportResult_Finished;
            }

        }

    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//-----------------------------------------------------------------------------
void GTest_DNAMulSequenceAlphabetId::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

	alpId = el.attribute(VALUE_ATTR);
    if (alpId.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    return;
	
}
Task::ReportResult GTest_DNAMulSequenceAlphabetId::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

	MAlignmentObject* myMSequence= qobject_cast<MAlignmentObject*>(obj);
	if(myMSequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
    DNAAlphabet *tempAlphabet = myMSequence->getMAlignment().alphabet;
	if( tempAlphabet->getId() != alpId){	
		stateInfo.error = QString("Alphabet id not matched: %1 expected %2").arg(tempAlphabet->getId()).arg(alpId);
	}
    return ReportResult_Finished;
}

void GTest_DNAMulSequenceSize::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    QString v = el.attribute(VALUE_ATTR);
    if (v.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    bool ok = false;
    seqSize = v.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
    }

}

Task::ReportResult GTest_DNAMulSequenceSize::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    MAlignmentObject* myMSequence= qobject_cast<MAlignmentObject*>(obj);
    if(myMSequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
    int tempLength=myMSequence->getMAlignment().getLength();
    if(tempLength != seqSize){
        stateInfo.error = QString("sequence size not matched: %1, expected %2 ").arg(tempLength).arg(seqSize);
    }
    return ReportResult_Finished;
}
void GTest_DNAMulSequencePart::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    subseq = el.attribute(VALUE_ATTR).toAscii();
    if (subseq.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    
	QString p = el.attribute(START_ATTR);
    if (p.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(START_ATTR);
        return;
    } 
    bool ok = false;
    startPos= p.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(START_ATTR);
    }

    seqName = el.attribute(SEQNAME_ATTR);
    if (seqName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(SEQNAME_ATTR);
        return;
    } 

}

Task::ReportResult GTest_DNAMulSequencePart::report() {
    GObject *obj = getContext<GObject>(this,objContextName);
    if (obj==NULL) {
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    MAlignmentObject* myMSequence= qobject_cast<MAlignmentObject*>(obj);
    if(myMSequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
  
    if (myMSequence->getMAlignment().getLength() < startPos + subseq.length()) {
        stateInfo.error = QString("sequence size is less that region end: size=%1, region-end=%").arg(myMSequence->getMAlignment().getLength(), startPos + subseq.length());
        return ReportResult_Finished;
    }
    
    if (!myMSequence->getMAlignment().alphabet->isCaseSensitive()) {
        subseq = subseq.toUpper();
    }
    bool ok_flag=false;
    foreach(MAlignmentItem myItem , myMSequence->getMAlignment().alignedSeqs){
        if(myItem.name==seqName){
            ok_flag=true;
            QByteArray objSubSeq = myItem.sequence.mid(startPos, subseq.length());
	        if (objSubSeq != subseq){	
	            stateInfo.error = QString("region not matched: %1, expected %2").arg(objSubSeq.constData()).arg(subseq.constData());
	            return ReportResult_Finished;
            }
        }   
    }
    if(!ok_flag){
        stateInfo.error = QString("no Sequence name: %1").arg(seqName);
    }
    return ReportResult_Finished;

}
void GTest_DNASequencInMulSequence::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }

    QString v = el.attribute(VALUE_ATTR);
    if (v.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
    bool ok = false;
    seqInMSeq = v.toInt(&ok);
    if (!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
    }

}

Task::ReportResult GTest_DNASequencInMulSequence::report() {
    GObject *obj = getContext<GObject>(this, objContextName);
    if(obj==NULL){
        stateInfo.error = GTest::tr("wrong value: %1").arg(OBJ_ATTR);
        return ReportResult_Finished;  
    }

    MAlignmentObject* myMSequence= qobject_cast<MAlignmentObject*>(obj);
    if(myMSequence==NULL){
        stateInfo.error = GTest::tr("can't cast to sequence from: %1").arg(obj->getGObjectName());
        return ReportResult_Finished;
    }
    int tempSize=myMSequence->getMAlignment().getNumSequences();
    if(tempSize != seqInMSeq){
        stateInfo.error = QString("numbers of Sequence not match: %1, expected %2 ").arg(tempSize).arg(seqInMSeq);
    }
    return ReportResult_Finished;
}
//----------------------------------------------------------
void GTest_DNACompeareMulSequencesInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareMulSequencesInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    MAlignmentObject * myMSequence = 0;
    MAlignmentObject * myMSequence2 = 0;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if(obj->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence = qobject_cast<MAlignmentObject*>(obj);
            if(myMSequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
        if(obj2->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence2 = qobject_cast<MAlignmentObject*>(obj2);
            if(myMSequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
////////////////////////////////////////
     if (myMSequence->getMAlignment().getLength() != myMSequence2->getMAlignment().getLength()) {
        stateInfo.error = QString("sequences size not matched: size1=%1, size2=%").arg(myMSequence->getMAlignment().getLength(), myMSequence2->getMAlignment().getLength());
        return ReportResult_Finished;
    }
    QList <MAlignmentItem> myQList1 = myMSequence->getMAlignment().alignedSeqs;
    QList <MAlignmentItem> myQList2 = myMSequence2->getMAlignment().alignedSeqs;

    for(int n=0;(n!=myQList1.size())||(n!=myQList2.size());n++){
        MAlignmentItem myItem1=myQList1.at(i);
        MAlignmentItem myItem2=myQList2.at(i);
        if(myItem1.name != myItem2.name){
            stateInfo.error = QString("names of regions in position %1 not matched: %2, expected %3").arg(n).arg(myItem2.name).arg(myItem1.name);
	        return ReportResult_Finished;
        }
    }
 //////////////////////////////////////////////////////////      
    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}


//----------------------------------------------------------
void GTest_DNACompeareMulSequencesNamesInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareMulSequencesNamesInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    MAlignmentObject * myMSequence = 0;
    MAlignmentObject * myMSequence2 = 0;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if(obj->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence = qobject_cast<MAlignmentObject*>(obj);
            if(myMSequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
        if(obj2->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence2 = qobject_cast<MAlignmentObject*>(obj2);
            if(myMSequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
////////////////////////////////////////
     if (myMSequence->getMAlignment().getLength() != myMSequence2->getMAlignment().getLength()) {
        stateInfo.error = QString("sequences size not matched: size1=%1, size2=%").arg(myMSequence->getMAlignment().getLength(), myMSequence2->getMAlignment().getLength());
        return ReportResult_Finished;
    }
    QList <MAlignmentItem> myQList1 = myMSequence->getMAlignment().alignedSeqs;
    QList <MAlignmentItem> myQList2 = myMSequence2->getMAlignment().alignedSeqs;

    for(int n=0;(n!=myQList1.size())||(n!=myQList2.size());n++){
        MAlignmentItem myItem1=myQList1.at(i);
        MAlignmentItem myItem2=myQList2.at(i);
        if(myItem1.name != myItem2.name){
            stateInfo.error = QString("names of regions in position %1 not matched: %2, expected %3").arg(n).arg(myItem2.name).arg(myItem1.name);
	        return ReportResult_Finished;
        }
    }
 //////////////////////////////////////////////////////////      
    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//----------------------------------------------------------
void GTest_DNACompeareMulSequencesAlphabetIdInTwoObjects::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DOC_ATTR);
        return;
    }

    secondDocContextName = el.attribute(VALUE_ATTR);
    if (secondDocContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(VALUE_ATTR);
        return;
    } 
}

Task::ReportResult GTest_DNACompeareMulSequencesAlphabetIdInTwoObjects::report() {
    Document* doc = getContext<Document>(this, docContextName);
    if (doc == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(docContextName);
        return ReportResult_Finished;
    }
    Document* doc2 = getContext<Document>(this, secondDocContextName);
    if (doc2 == NULL) {
        stateInfo.error = GTest::tr("document not found %1").arg(secondDocContextName);
        return ReportResult_Finished;
    }

    const QList<GObject*>& objs = doc->getObjects();
    const QList<GObject*>& objs2 = doc2->getObjects();
    GObject*obj=NULL;
    GObject*obj2=NULL;
    MAlignmentObject * myMSequence = 0;
    MAlignmentObject * myMSequence2 = 0;

    for(int i=0;(i!=objs.size())||(i!=objs2.size());i++){
        obj = objs.at(i);
        obj2 = objs2.at(i);
        
        if(obj->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence = qobject_cast<MAlignmentObject*>(obj);
            if(myMSequence==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
        if(obj2->getGObjectType()== GObjectTypes::MULTIPLE_ALIGNMENT){
            myMSequence2 = qobject_cast<MAlignmentObject*>(obj2);
            if(myMSequence2==NULL){
                stateInfo.error = GTest::tr("can't cast to sequence from: %1 in position %2").arg(obj2->getGObjectName()).arg(i);
                return ReportResult_Finished;
            }
        }
////////////////////////////////////////
        DNAAlphabet *tempAlphabet = myMSequence->getMAlignment().alphabet;
        DNAAlphabet *tempAlphabet2 = myMSequence2->getMAlignment().alphabet;
        if (tempAlphabet->getId() != tempAlphabet2->getId()) {
            stateInfo.error = QString("sequences alphabets not matched: alphabet1=%1, alphabet2=%").arg(tempAlphabet->getId(),tempAlphabet2->getId());
            return ReportResult_Finished;
        }
//////////////////////////////////////////////////////////      
    }
    
    if(obj!=objs.last() ){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }
    if(obj2!=objs2.last()){
        stateInfo.error = GTest::tr("number of objects in document not matches: %1").arg(obj2->getGObjectName());
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}

//-----------------------------------------------------------------------------
QList<XMLTestFactory*> DNASequenceObjectTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_DNASequenceSize::createFactory());
    res.append(GTest_DNASequenceAlphabet::createFactory());
    res.append(GTest_DNASequencePart::createFactory());
	res.append(GTest_DNASequenceAlphabetType::createFactory());
    res.append(GTest_DNASequenceAlphabetId::createFactory());
    res.append(GTest_DNAMulSequenceAlphabetId::createFactory());
    res.append(GTest_DNAMulSequenceSize::createFactory());
    res.append(GTest_DNAMulSequencePart::createFactory());
    res.append(GTest_DNASequencInMulSequence::createFactory());
    res.append(GTest_DNACompeareSequencesNamesInTwoObjects::createFactory());
    res.append(GTest_DNACompeareSequencesInTwoObjects::createFactory());
    res.append(GTest_DNACompeareSequencesAlphabetsInTwoObjects::createFactory());
    res.append(GTest_DNACompeareMulSequencesInTwoObjects::createFactory());
    res.append(GTest_DNACompeareMulSequencesNamesInTwoObjects::createFactory());
    res.append(GTest_DNACompeareMulSequencesAlphabetIdInTwoObjects::createFactory());
    return res;
}

}//namespace
