/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "FindDialog.h"

#include <util_algorithm/FindAlgorithmTask.h>


#include <core_api/DNATranslation.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/AppContext.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>

#include <selection/DNASequenceSelection.h>

#include <util_gui/CreateAnnotationDialog.h>
#include <util_gui/CreateAnnotationWidgetController.h>

#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_tasks/CreateAnnotationTask.h>
#include <util_text/TextUtils.h>

#include <assert.h>

#include <QtCore/QFileInfo>
#include <QtGui/QMessageBox>
#include <QtGui/QListWidgetItem>

namespace GB2 {

/* TRANSLATOR GB2::FindDialog */ 

class FRListItem : public QListWidgetItem {
public:
    FRListItem(const FindAlgorithmResult& r);
    FindAlgorithmResult res;

    virtual bool operator< ( const QListWidgetItem & other ) const;
};

static FRListItem* findItem(const FindAlgorithmResult& r, QListWidget* lv) {
    for (int i=0, n = lv->count(); i<n ; i++) {
        FRListItem* item = (FRListItem*)lv->item(i);
        if (r == item->res) {
            return item;
        }
    }
    return NULL;
}

bool FindDialog::runDialog(ADVSequenceObjectContext* ctx) {
    FindDialog d(ctx);
    d.exec();
    return true;
}

FindDialog::FindDialog(ADVSequenceObjectContext* context) {
    setupUi(this);

    ctx = context;
    prevAlgorithm = 0;
    prevMatch = 100;
    task = NULL;
    
    connectGUI();
    updateState();
    if (context->getComplementTT() == NULL) {
        rbDirect->setChecked(true);
    }
    
    sbMatch->setMinimum(30);

    int seqLen = context->getSequenceLen();
    sbRangeStart->setMinimum(1);
    sbRangeStart->setMaximum(seqLen);
    
    sbCurrentPos->setMinimum(1);
    sbCurrentPos->setMaximum(seqLen);
    
    sbRangeEnd->setMinimum(1);
    sbRangeEnd->setMaximum(seqLen);

    sbRangeStart->setValue(initialSelection.isEmpty() ? 1 : initialSelection.startPos + 1);
    sbRangeEnd->setValue(initialSelection.isEmpty() ? seqLen : initialSelection.endPos());

    
    leFind->setFocus();
    lbResult->setSortingEnabled(true);

    timer = new QTimer(this);

    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_onTaskFinished(Task*)));
    connect(timer, SIGNAL(timeout()), SLOT(sl_onTimer()));
}

void FindDialog::connectGUI() {
    //buttons
    connect(pbSaveAnnotations, SIGNAL(clicked()), SLOT(sl_onSaveAnnotations()));
    connect(pbClearList, SIGNAL(clicked()), SLOT(sl_onClearList()));
    connect(pbRemoveOverlaps, SIGNAL(clicked()), SLOT(sl_onRemoveOverlaps()));
    connect(pbFind, SIGNAL(clicked()), SLOT(sl_onFindNext()));
    connect(pbFindAll, SIGNAL(clicked()), SLOT(sl_onFindAll()));
    connect(pbClose, SIGNAL(clicked()), SLOT(sl_onClose()));
    
    //search line edit
    connect(leFind, SIGNAL(textEdited(const QString&)), SLOT(sl_onSearchPatternChanged(const QString&)));

    //radio button groups
    connect(rbSequence, SIGNAL(clicked()), SLOT(sl_onSequenceTypeChanged()));
    connect(rbTranslation, SIGNAL(clicked()), SLOT(sl_onSequenceTypeChanged()));
    connect(rbBoth, SIGNAL(clicked()), SLOT(sl_onStrandChanged()));
    connect(rbDirect, SIGNAL(clicked()), SLOT(sl_onStrandChanged()));
    connect(rbComplement, SIGNAL(clicked()), SLOT(sl_onStrandChanged()));
    connect(rbMismatchAlg, SIGNAL(clicked()), SLOT(sl_onAlgorithmChanged()));
    connect(rbInsDelAlg, SIGNAL(clicked()), SLOT(sl_onAlgorithmChanged()));

    //match percent spin
    connect(sbMatch, SIGNAL(valueChanged(int)), SLOT(sl_onMatchPercentChanged(int)));

    //connect position selectors
    connect(sbRangeStart, SIGNAL(valueChanged(int)), SLOT(sl_onRangeStartChanged(int)));
    connect(sbCurrentPos, SIGNAL(valueChanged(int)), SLOT(sl_onCurrentPosChanged(int)));
    connect(sbRangeEnd, SIGNAL(valueChanged(int)), SLOT(sl_onRangeEndChanged(int)));

    //results list
    connect(lbResult, SIGNAL(itemActivated(QListWidgetItem*)), SLOT(sl_onResultActivated(QListWidgetItem*)));
    connect(lbResult, SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)), SLOT(sl_currentResultChanged(QListWidgetItem*, QListWidgetItem*)));

    //range buttons
    connect(pbRangeToSelection, SIGNAL(clicked()), SLOT(sl_onRangeToSelection()));
    connect(pbRangeToSeq, SIGNAL(clicked()), SLOT(sl_onRangeToSequence()));

    lbResult->installEventFilter(this);
}


void FindDialog::updateState() {
    bool hasInitialSelection = initialSelection.len > 0;
    bool hasActiveTask = task!=NULL;
    bool hasAmino = ctx->getAminoTT()!=NULL;
    bool hasCompl = ctx->getComplementTT()!=NULL;

    bool hasResults = lbResult->count() > 0;
    bool hasPattern = !leFind->text().isEmpty();
    
    sbMatch->setEnabled(hasPattern);
    pbFind->setEnabled(hasPattern);
    pbFindAll->setEnabled(hasPattern);

    leFind->setEnabled(!hasActiveTask);
    pbFind->setEnabled(!hasActiveTask);
    pbFindAll->setEnabled(!hasActiveTask);
    pbSaveAnnotations->setEnabled(!hasActiveTask && hasResults);
    pbClearList->setEnabled(!hasActiveTask &&  hasResults);
    pbRemoveOverlaps->setEnabled(lbResult->count() > 2);
    pbClose->setText(hasActiveTask ? tr("cancel_button") : tr("close_button"));  

    rbSequence->setEnabled(!hasActiveTask);
    rbTranslation->setEnabled(!hasActiveTask && hasAmino);
    rbBoth->setEnabled(!hasActiveTask && hasCompl);
    rbDirect->setEnabled(!hasActiveTask);
    rbComplement->setEnabled(!hasActiveTask && hasCompl);
    rbMismatchAlg->setEnabled(!hasActiveTask);
    rbInsDelAlg->setEnabled(!hasActiveTask);

    sbRangeStart->setEnabled(!hasActiveTask);
    sbCurrentPos->setEnabled(!hasActiveTask);
    sbRangeEnd->setEnabled(!hasActiveTask);
    pbRangeToSelection->setEnabled(!hasActiveTask && hasInitialSelection);
    pbRangeToSeq->setEnabled(!hasActiveTask);

    updateStatus();
}

void FindDialog::updateStatus() {
    QString message;
    if (task != NULL) {
        message = tr("progress_%1%_current_pos_%2_").arg(task->getProgress()).arg(task->getCurrentPos());
    }
    message += tr("%1_results_found.").arg(lbResult->count());
    statusBar->setText(message);
}

bool FindDialog::eventFilter(QObject *obj, QEvent *ev) {
    if (obj == lbResult && ev->type() == QEvent::KeyPress) {
        QKeyEvent* ke = (QKeyEvent*)ev;
        if (ke->key() == Qt::Key_Space) {
            FRListItem* item = (FRListItem*)lbResult->currentItem();
            if (item != NULL) {
                sl_onResultActivated(item);
            }
        }
    }
    return false;
}

void FindDialog::sl_onSaveAnnotations() {
    if (lbResult->count() == 0) {
        return;
    }
    CreateAnnotationModel m;
    m.sequenceObjectRef = ctx->getSequenceObject();
    m.hideLocation = true;
    CreateAnnotationDialog d(&m);
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    assert(m.annotationObject!=NULL);
    const QString& name = m.data->name;
    QList<SharedAnnotationData> list;
    for (int i=0, n = lbResult->count(); i<n; ++i) {
        FRListItem* item = (FRListItem* )lbResult->item(i);
        list.append(item->res.toAnnotation(name));
    }

    CreateAnnotationsTask* t = new CreateAnnotationsTask(m.annotationObject, m.groupName, list);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void FindDialog::sl_onClearList() {
    lbResult->clear();
    updateState();
}

void FindDialog::sl_onFindNext() {
    //TODO: check if the same search params are used after find-all -> use cached results?
    bool ok = checkState(true);
    if (!ok) {
        return;
    }
    
    if (sbCurrentPos->value() >= sbRangeEnd->value()) {
        int res = QMessageBox::question(this, tr("question_caption"), tr("restart_q"), QMessageBox::Yes, QMessageBox::No);
        if (res != QMessageBox::Yes) {
            return;
        }
        sbCurrentPos->setValue(sbRangeStart->value());
    }

    savePrevSettings();
    runTask(true);
}

void FindDialog::sl_onFindAll() {
    //TODO: check if the same search params are used after find-all -> use cached results?
    bool ok = checkState(false);
    if (!ok) {
        return;
    }
    
    sbCurrentPos->setValue(sbRangeStart->value());

    savePrevSettings();
    runTask(false);
}

void FindDialog::reject() {
    if (task!=NULL) {
        task->cancel();
        return;
    }
    QDialog::reject();
}

void FindDialog::sl_onRangeStartChanged(int v) {
    if (v > sbCurrentPos->value()) {
        sbCurrentPos->setValue(v);
    }
}

void FindDialog::sl_onCurrentPosChanged(int v) {
    if (v > sbRangeEnd->value()) {
        sbRangeEnd->setValue(v);
    }
    if (v < sbRangeStart->value()) {
        sbRangeStart->setValue(v);
    }
}

void FindDialog::sl_onRangeEndChanged(int v) {
    if (v < sbCurrentPos->value()) {
        sbCurrentPos->setValue(v);
    }
}


void FindDialog::sl_onClose() {
    reject();
}

void FindDialog::tunePercentBox() {
    int patternLen = qMax(1, leFind->text().length());
    int p = sbMatch->value();
    int step = qMax(1, 100 / patternLen);
    sbMatch->setSingleStep(step);
    int diff = p % step;
    if (diff == 0 || p == 100) {
        return;
    }
    int newVal = p;
    if (diff > step /2) {
        newVal=qMin(100, newVal + (step - diff));
    } else {
        newVal-= diff;
    }
    if (newVal < sbMatch->minimum()) {
        newVal+= step;
    }
    assert(newVal <= 100);
    sbMatch->setValue(newVal);

}

//line ed
void FindDialog::sl_onSearchPatternChanged(const QString&) {
    if (leFind->text().length() > getCompleteSearchRegion().len) {
        sl_onRangeToSequence();
    }
    tunePercentBox();
    updateState();
}

// groups
void FindDialog::sl_onSequenceTypeChanged() {
}

void FindDialog::sl_onStrandChanged() {
}

void FindDialog::sl_onAlgorithmChanged() {
}

//spin box
void FindDialog::sl_onMatchPercentChanged(int) {
    tunePercentBox();
}



bool FindDialog::checkState(bool forSingleShot) {
    QString pattern = leFind->text();
    if (pattern.isEmpty()) {
        QMessageBox::critical(this, tr("error"), tr("search_pattern_is_empty_text"));
        return false;
    }

    int maxErr = getMaxErr();
    int minMatch = pattern.length() - maxErr;
    assert(minMatch > 0);
    if (minMatch > getCompleteSearchRegion().len) {
        QMessageBox::critical(this, tr("error"), tr("pattern_is_too_long"));
        return false;
    }
    
    //check pattern's alphabet
    DNAAlphabet* al = ctx->getAlphabet();
    if (!al->isCaseSensitive()) {
        QString oldPattern = pattern;
        pattern = pattern.toUpper();
        if (pattern!=oldPattern) { // make visible the logic we use to user
            leFind->setText(pattern); 
        }
    }

    bool isTranslation = rbTranslation->isChecked();
    if (isTranslation) {
        DNATranslation* t = ctx->getAminoTT();
        assert(t!=NULL);
        al = t->getDstAlphabet();
    }
    bool alphabetIsOk = TextUtils::fits(al->getMap(), pattern.toLocal8Bit().data(), pattern.size());
    if (!alphabetIsOk) {
        int res = QMessageBox::warning(this, tr("warning"), tr("search_pattern_alphabet_error_continue_q"), QMessageBox::Yes, QMessageBox::No);
        if (res == QMessageBox::No) {
            return false;
        }
    }

    if (lbResult->count() > 0) {
        if (forSingleShot) {
            bool settingsTheSame = checkPrevSettings();
            if (!settingsTheSame) {
                int res = QMessageBox::warning(this, tr("warning"), tr("settings_changed_clean_res_q"), QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel);
                if (res == QMessageBox::Cancel) {
                    return false;
                }
                if (res == QMessageBox::Yes) {
                    lbResult->clear();
                    sbCurrentPos->setValue(sbRangeStart->value());
                }
            }
        } else {
            int res = QMessageBox::warning(this, tr("warning"), tr("results_list_not_empty_ask_clear"), QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel);
            if (res == QMessageBox::Cancel) {
                return false;
            }
            if (res == QMessageBox::Yes) {
                lbResult->clear();
            }
        }
    }

    return true;
}


bool FindDialog::checkPrevSettings() {
    if (prevSearchString != leFind->text()) {
        return false;
    }
    int match = sbMatch->value();
    if (match != prevMatch) {
        return false;
    }
    int alg = match == 100 ? 0 : rbMismatchAlg->isChecked() ? 1 : 2;
    if (prevAlgorithm != alg) {
        return false;
    }
    return true;
}

void FindDialog::savePrevSettings() {
    prevSearchString = leFind->text();
    prevMatch = sbMatch->value();
    prevAlgorithm = prevMatch == 100 ? 0 : rbMismatchAlg->isChecked() ? 1 : 2;
}

int FindDialog::getMaxErr() const {
    return int((float)(1 - float(sbMatch->value()) / 100) * leFind->text().length());
}

LRegion FindDialog::getCompleteSearchRegion() const {
    return LRegion(sbRangeStart->value()-1, sbRangeEnd->value() - sbRangeStart->value() + 1);
}

void FindDialog::runTask(bool singleShot) {
    assert(task == NULL);
    
    FindAlgorithmTaskSettings s;
    s.sequence = ctx->getSequenceData();
    s.pattern = leFind->text().toLocal8Bit();
    s.strand = rbBoth->isChecked() ? FindAlgorithmStrand_Both : (rbDirect->isChecked() ? FindAlgorithmStrand_Direct : FindAlgorithmStrand_Complement);
    s.complementTT = ctx->getComplementTT();
    if (s.complementTT == NULL && s.strand!=FindAlgorithmStrand_Direct) {
        assert(0);
        s.strand = FindAlgorithmStrand_Both;
    }
    s.proteinTT = rbTranslation->isChecked() ? ctx->getAminoTT() : NULL;
    s.singleShot = singleShot;
    
    s.maxErr = getMaxErr();
    
    s.insDelAlg = rbInsDelAlg->isChecked();

    //setup search region
    s.searchRegion = getCompleteSearchRegion();

    if (singleShot) { //TODO: loosing complementary strand here!
        int newStartPos = sbCurrentPos->value(); //visual val is +1 to the last used current
        s.searchRegion.len-=(newStartPos - s.searchRegion.startPos);
        s.searchRegion.startPos = newStartPos;
    }
    
    task = new FindAlgorithmTask(s);
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
    updateState();
    timer->start(400);
}

void FindDialog::sl_onTaskFinished(Task* t) {
    if (t != task || t->getState()!= Task::State_Finished) {
        return;
    }
    //todo: show message if task was canceled?
    importResults();
    task = NULL;
    //TODO: show report window if not a singlShot?
    updateState();
    timer->stop();
}

void FindDialog::sl_onTimer() {
    importResults();
}

void FindDialog::importResults() {
    if (task == NULL) {
        return;
    }
    
    int currentPos = task->getCurrentPos();
    sbCurrentPos->setValue(currentPos+1);

    QList<FindAlgorithmResult> newResults = task->popResults();
    if (!newResults.empty()) {
        FRListItem* item = NULL;
        foreach(const FindAlgorithmResult& r, newResults) {
            item = findItem(r, lbResult);
            if (item==NULL) {
                item = new FRListItem(r);
                lbResult->addItem(item);
            }
        }

        if (task->getSettings().singleShot) {
            item->setSelected(true);
            lbResult->scrollToItem(item);
            sl_onResultActivated(item);
        }
        lbResult->setFocus();
    }
    updateStatus();
}

void FindDialog::sl_onResultActivated(QListWidgetItem* i) {
    assert(i);
    FRListItem* item = (FRListItem*)i;
    DNASequenceSelection* sel = ctx->getSequenceSelection();
    sel->clear();
    sel->addRegion(item->res.region);
    sbCurrentPos->setValue(item->res.region.startPos+1);
    //TODO: add complement info to selection!!
}

void FindDialog::sl_currentResultChanged(QListWidgetItem* current, QListWidgetItem* prev) {
    Q_UNUSED(prev);
    if (current==NULL) {
        return;
    }
    FRListItem* item = (FRListItem*)current;
    sbCurrentPos->setValue(item->res.region.startPos+1);
}

void FindDialog::sl_onRangeToSelection() {
    assert(initialSelection.len!=0);
    sbRangeStart->setValue(initialSelection.startPos + 1);
    sbCurrentPos->setValue(sbRangeStart->value());
    sbRangeEnd->setValue(initialSelection.endPos());
}


void FindDialog::sl_onRangeToSequence() {
    sbRangeStart->setValue(1);
    sbCurrentPos->setValue(sbRangeStart->value());
    sbRangeEnd->setValue(ctx->getSequenceLen());
}

#define MAX_OVERLAP_K 0.5F

void FindDialog::sl_onRemoveOverlaps() {
    int nBefore = lbResult->count();
    for (int i = 0, n = lbResult->count(); i < n; i++) {
        FRListItem* ri = (FRListItem*)lbResult->item(i);
        for (int j=i+1; j < n; j++) {
            FRListItem* rj = (FRListItem*)lbResult->item(j);
            assert(rj->res.region.startPos >= ri->res.region.startPos);

            if (rj->res.complement != ri->res.complement) {
                continue;
            }
            if (rj->res.translation != ri->res.translation) {
                continue;
            }
            if (rj->res.translation) {
                bool complement =rj->res.complement ;
                int framej = complement ? rj->res.region.endPos() % 3 : rj->res.region.startPos % 3;
                int framei = complement ? ri->res.region.endPos() % 3 : ri->res.region.startPos % 3;
                if (framei != framej) {
                    continue;
                }
            }
            LRegion r = rj->res.region.intersect(ri->res.region);
            if (r.len > 0 && r.len >= MAX_OVERLAP_K * ri->res.region.len) {
                if (ri->res.err > ri->res.err) {
                    delete ri;
                    i--;
                    break;
                } else {
                    j--;
                    delete rj;
                }
                n--;
            } else {
                break;
            }
        }
    }
    
    int removed = nBefore - lbResult->count();
    
    QString message= tr("%1 overlaps filtered, %2 results left.").arg(removed).arg(lbResult->count());
    statusBar->setText(message);
}


//////////////////////////////////////////////////////////////////////////
/// list

FRListItem::FRListItem(const FindAlgorithmResult& r) : res(r) 
{
    QString yes = FindDialog::tr("yes");
    QString no = FindDialog::tr("no");
    setText(FindDialog::tr("[%1 %2]  translation:%3 complement:%4")
        .arg(res.region.startPos+1) //user sees sequence from [1, end]
        .arg(res.region.endPos())
        .arg(res.translation ? yes : no)
        .arg(res.complement ? yes : no));
}

bool FRListItem::operator< ( const QListWidgetItem & other ) const {
    const FRListItem& o = (const FRListItem &)other;
    if (o.res.region.startPos == res.region.startPos) {
        if (o.res.region.endPos() == res.region.endPos()) {
            return this > &other;
        }
        return o.res.region.endPos() > res.region.endPos();
    }
    return o.res.region.startPos > res.region.startPos;
}

}//namespace
