/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DialogUtils.h"

#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <core_api/Settings.h>

#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtGui/QMessageBox>

namespace GB2 {

void DialogUtils::showProjectIsLockedWarning(QWidget* p) {
	QMessageBox::critical(p, tr("project_is_locked_title"), tr("project_is_locked_message"), QMessageBox::Ok, QMessageBox::NoButton);
}

QString DialogUtils::getDocumentExtension(const QString& url) {
    QFileInfo f(url);
    QString ext = f.suffix();
    if (ext == "gz") {
        QString completeSuffix = f.completeSuffix();
        ext = completeSuffix.split(".").first();
        assert(!ext.isEmpty());
    }
    return ext;
}

QString DialogUtils::ensureFileExt(const QString& url, QStringList typeExt) {
    if (typeExt.contains(getDocumentExtension(url))) {
        return url;
    } else {
        return url + "." + typeExt.first();
    }
}

QString DialogUtils::prepareFileName(const QString& url, int count, QStringList typeExt) {
    return prepareFileName(url, QString("%1").arg(count, 3, 10, QChar('0')), typeExt);
}

QString DialogUtils::prepareFileName(const QString& url, const QString& baseSuffix, QStringList typeExt) {    
    QFileInfo fi(url);
    QStringList sfs = fi.completeSuffix().split(".");
    QString ext;
    foreach(QString s, sfs) {
        if (typeExt.contains(s)) {
            ext = s;
            break;
        }
    }
    if (ext.isEmpty()) {
        return url + QString(".%1.%2").arg(baseSuffix).arg(typeExt.first());
    }
    QString base = fi.baseName();
    while(!sfs.isEmpty()) {
        QString nextS = sfs.takeFirst();
        if (nextS == ext) {
            break;
        }
        base += "." + nextS;
    }
    QString result = QString("%1/%2.%3.%4").arg(fi.dir().path()).arg(base).arg(baseSuffix).arg(ext);
    if (!sfs.isEmpty()) {
        result.append(".").append(sfs.join("."));
    }

    return result;
}

QString DialogUtils::prepareFileFilter(const QString& name, const QStringList& exts, bool any, const QStringList& extra) {
    QString line = name + " (";
    foreach(QString ext, exts) {
        line+=" *."+ext;
    }
    foreach(QString ext, exts) {
        foreach(QString s, extra) {
            line+=" *."+ext+s;
        }
    }
    line+=")";
    if (any) {
        line += "\n" + tr("All files") + " (*.*)";
    }
    return line;
}

QString DialogUtils::prepareDocumentsFileFilter(const DocumentFormatId& fid, bool any, const QStringList& extra) {
    DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
    QString result = prepareFileFilter(df->getFormatName(), df->getSupportedDocumentFileExtensions(), any, extra);
    return result;
}

QString DialogUtils::prepareDocumentsFileFilter(bool any, const QStringList& extra) {
    DocumentFormatRegistry* fr = AppContext::getDocumentFormatRegistry();
    QList<DocumentFormatId> ids = fr->getRegisteredFormats();
    QStringList result;
    if (any) { 
        result.append(tr("All files") + " (*.*)");
    }
    foreach(DocumentFormatId id , ids) {
        DocumentFormat* df = fr->getFormatById(id);
        result.append(prepareFileFilter(df->getFormatName(), df->getSupportedDocumentFileExtensions(), false, extra));
    }
    return result.join("\n");
}

QString DialogUtils::prepareDocumentsFileFilter(const DocumentFormatConstraints& c, bool any) {
    QString result;

    if (any) {
        result+=tr("All files") + " (*.*)";
    }

    QList<DocumentFormatId> ids = AppContext::getDocumentFormatRegistry()->getRegisteredFormats();
    foreach(const DocumentFormatId& id, ids) {
        DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(id);
        if (df->checkConstraints(c)) {
            if (!result.isEmpty()) {
                result.append("\n");
            }
            result += prepareDocumentsFileFilter(id, false);
        }
    }

    return result;
}

QString DialogUtils::prepareDocumentsFileFilterByObjType(const GObjectType& t, bool any) {
    DocumentFormatConstraints c;
    c.supportedObjectTypes.append(t);;
    return prepareDocumentsFileFilter(c, any);
}


#define SETTINGS_ROOT QString("gui/")

QString DialogUtils::getLastOpenFileDir(const QString& toolType, const QString& defaultVal) {
    QString key = SETTINGS_ROOT + (toolType.isEmpty() ? "" : toolType + "/") + "lastDir";
    QString res = AppContext::getSettings()->getValue(key, defaultVal).toString();
    return res;
}

void DialogUtils::setLastOpenFileDir(const QString& ld, const QString& toolType) {
    QString key = SETTINGS_ROOT + (toolType.isEmpty() ? "" : toolType + "/") + "lastDir";
    AppContext::getSettings()->setValue(key, ld);
}


LastOpenDirHelper::LastOpenDirHelper(const QString& d, const QString& defaultVal) {
    domain = d;
    dir = DialogUtils::getLastOpenFileDir(domain, defaultVal);
}

LastOpenDirHelper::~LastOpenDirHelper() {
    saveURLDir2LastOpenDir();
}

void LastOpenDirHelper::saveURLDir2LastOpenDir() {
    if (!url.isEmpty()) {
        QString newDir = QFileInfo(url).absoluteDir().absolutePath();
        if (dir != newDir) {
            DialogUtils::setLastOpenFileDir(newDir, domain);
        }
    }
}

} // namespace
