/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotatedDNAViewFactory.h"
#include "AnnotatedDNAView.h"
#include "AnnotatedDNAViewTasks.h"
#include "AnnotatedDNAViewState.h"
#include "ADVConstants.h"

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>

#include <selection/SelectionUtils.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/GObjectUtils.h>

namespace GB2 {
/* TRANSLATOR GB2::AnnotatedDNAView */

static LogCategory log(ULOG_CAT_ADV);

const GObjectViewFactoryId AnnotatedDNAViewFactory::ID(ANNOTATED_DNA_VIEW_FACTORY_ID);

AnnotatedDNAViewFactory::AnnotatedDNAViewFactory() 
: GObjectViewFactory(ID, tr("Sequence view"))
{
}

bool AnnotatedDNAViewFactory::canCreateView(const MultiGSelection& multiSelection) {
    //return true if
    //1. selection has DNA sequence object
    //2. selection has document can contain DNA sequence object
    //3. selection has any object with SEQUENCE relation to DNA Sequence that is in project
    
	//1 & 2
    bool hasSequenceDocuments = !SelectionUtils::findDocumentsWithObjects(GObjectTypes::DNA_SEQUENCE, &multiSelection, true, true).isEmpty();
    if (hasSequenceDocuments) {
        return true;
    }
    
    //3
    QSet<GObject*> selectedObjects = SelectionUtils::findObjects("", &multiSelection);
    //TODO: process related objects with unloaded sequence object too!!
    QList<GObject*> objectsWithSeqRelation = GObjectUtils::selectObjectsWithRelation(selectedObjects.toList(), GObjectType::null,
                                                    GObjectRelationRole::SEQUENCE, false);
    return !objectsWithSeqRelation.isEmpty();
}


Task* AnnotatedDNAViewFactory::createViewTask(const MultiGSelection& multiSelection, bool single) {
	QSet<GObject*> sequenceObjects = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, &multiSelection);
    QSet<GObject*> selectedObjects = SelectionUtils::findObjects("", &multiSelection);
    QList<GObject*> objectsWithSequenceRelation = GObjectUtils::selectObjectsWithRelation(selectedObjects.toList(), GObjectType::null,
                                                    GObjectRelationRole::SEQUENCE, false);
    sequenceObjects.unite(QSet<GObject*>::fromList(objectsWithSequenceRelation));
    
	
    QList<OpenAnnotatedDNAViewTask*> resTasks;

    if (!sequenceObjects.isEmpty()) {
        resTasks.append(new OpenAnnotatedDNAViewTask(sequenceObjects.toList()));
    }

    //BUG:424: limit number of opened views

    QSet<Document*> docsWithSequences = SelectionUtils::findDocumentsWithObjects(GObjectTypes::DNA_SEQUENCE, &multiSelection, true, false);
	foreach(Document* doc, docsWithSequences) {
		if (doc->isLoaded()) {
			QList<GObject*> docObjs = doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE);
            if (sequenceObjects.intersect(QSet<GObject*>::fromList(docObjs)).isEmpty()) {
                resTasks.append(new OpenAnnotatedDNAViewTask(docObjs));
            }
		} else {
			resTasks.append(new OpenAnnotatedDNAViewTask(doc));
		}
	}

    if (resTasks.isEmpty()) {
		return NULL;
	}
	if (resTasks.size() == 1 || single) {
		return resTasks.first();
	}
	Task* result = new Task(tr("open_multiple_views"), TaskFlags_NR_DWF);
	foreach(Task* t, resTasks) {
		result->addSubTask(t);
	}
	return result;
}

bool AnnotatedDNAViewFactory::isStateInSelection(const MultiGSelection& multiSelection, const QVariantMap& stateData) {
	AnnotatedDNAViewState state(stateData);
	if (!state.isValid()) {
		return false;
	}
	QList<GObjectReference> refs = state.getSequenceObjects();
    assert(!refs.isEmpty());
    foreach (const GObjectReference& ref, refs) {
	    Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
	    if (doc == NULL) { //todo: accept to use invalid state removal routines of ObjectViewTask ???
    		return false;
    	}
        //check that document is in selection
	    QList<Document*> selectedDocs = SelectionUtils::getSelectedDocs(multiSelection);
	    bool docIsSelected = selectedDocs.contains(doc);

        //check that object is in selection
	    QList<GObject*> selectedObjects = SelectionUtils::getSelectedObjects(multiSelection);
	    GObject* obj = doc->findGObjectByName(ref.objName);
        bool objIsSelected = obj!=NULL && selectedObjects.contains(obj);

        //check that object associated with sequence object is in selection
        bool refIsSelected = false;
        foreach (const GObject* selObject, selectedObjects) {
            GObjectReference selRef(selObject);
            if (ref == selRef) {
                refIsSelected = true;
                break;
            }
        }
        if (!docIsSelected && !objIsSelected && !refIsSelected) {
            return false;
        }
    }

	return true;
}

Task* AnnotatedDNAViewFactory::createViewTask(const QString& viewName, const QVariantMap& stateData) {
	return new OpenSavedAnnotatedDNAViewTask(viewName, stateData);
}




} // namespace

