/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_PAN_VIEW_H_
#define _GB2_PAN_VIEW_H_

#include "GSequenceLineViewAnnotated.h"

#include <QtGui/QFont>
#include <QtGui/QAction>
#include <QtGui/QScrollBar>

namespace GB2 {

class PanViewRenderArea;
class GScrollBar;
class GObjectView;
class PVRowsManager;
class AnnotationTableObject;
class Annotation;

class GB2_COREAPI_EXPORT PanView : public GSequenceLineViewAnnotated {
	Q_OBJECT

public:
	PanView(QWidget* p, ADVSequenceObjectContext* ctx);
	~PanView();
	
    const LRegion& getFrameRange() const {return frameView->getVisibleRange();}
	
	virtual QAction* getZoomInAction() const {return zoomInAction;}

	virtual QAction* getZoomOutAction() const {return zoomOutAction;}

    virtual QAction* getZoomToSelectionAction() const {return zoomToSelectionAction;}

    virtual QAction* getZoomToSequenceAction() const {return zoomToSequenceAction;}

	// [0..seqLen)
	virtual void setVisibleRange(const LRegion& reg, bool signal = true);

	int getSeqLen() const {return seqLen;}

	PVRowsManager* getRowsManager() const {return rowsManager;}

protected:
	virtual int getSingleStep() const {return qMax(1, visibleRange.len / 10);}
	virtual int getPageStep() const {return qMax(1, visibleRange.len / 5);}
	virtual void onVisibleRangeChanged(bool signal = true);
	virtual void pack();
	
	virtual void registerAnnotations(const QList<Annotation*>& l);
	virtual void unregisterAnnotations(const QList<Annotation*>& l);
	virtual void ensureVisible(Annotation* a, int locationIdx);

private slots:
	void sl_zoomInAction();
	void sl_zoomOutAction();
    void sl_zoomToSelection();
    void sl_zoomToSequence();
    
	void sl_onRowBarMoved(int);

	void sl_onRangeChangeRequest(int start, int end);
	
    virtual void sl_onDNASelectionChanged(LRegionsSelection* s, const QList<LRegion>& added, const QList<LRegion>& removed);
    virtual void sl_onAnnotationSettingsChanged(const QStringList& changedSettings);

private:
	void setSelection(const LRegion& r);
	void centerRow(int row);
	int  calculateNumRowBarSteps() const;
	
	void updateActions();
	void updateRows();
	void updateRowBar();

	LRegion				frameRange;
	int					minNuclsPerScreen;
	
	QAction*			zoomInAction;
	QAction*			zoomOutAction;
    QAction*			zoomToSelectionAction;
    QAction*			zoomToSequenceAction;

	PVRowsManager*	    rowsManager;
	QScrollBar*			rowBar;

};


class PanViewRenderArea : public GSequenceLineViewAnnotatedRenderArea {
public:
	PanViewRenderArea(PanView* d);

	bool isRowVisible(int row) const { return getRowLine(row) >= 0; }
	int getRowLine(int i) const;
	int getNumVisibleRows() const {return getFirstRowLine() + 1;}
	int getRowLinesOffset() const {return rowLinesOffset;}
	void setRowLinesOffset(int r);

	virtual LRegion getAnnotationYRange(Annotation* a, const LRegion& r, const AnnotationSettings* as) const;

    bool updateNumVisibleRows();

protected:
	virtual void drawAll(QPaintDevice* pd);
	virtual void drawAnnotations(QPainter& p);

private:
	int getSelectionLine() const {return numLines - 1;}
	int getRulerLine() const {return numLines - 2;}
	int getFirstRowLine()const {return numLines - 3;}
	
	PanView* getPanView() const {return (PanView*)view;}
	
	int getLineY(int line) const {
		return cachedView->height() - ((numLines - line) * lineHeight);
	}

	void drawRuler(QPainter& p);
	void drawSequenceSelection(QPainter& p);

	PanView*		panView;
	QFont			rulerFont;
	int				lineHeight;     // single line height
	int				numLines;	    // number of visible lines
	int				rowLinesOffset; // row number on the first row line
};


}//namespace;

#endif
