/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SequenceWalkerTask.h"

#include <core_api/DNATranslation.h>
#include <util_text/TextUtils.h>

namespace GB2 {

SequenceWalkerTask::SequenceWalkerTask(const SequenceWalkerConfig& c, SequenceWalkerCallback* cb, const QString& name, TaskFlags tf) 
: Task(name, tf | TaskFlags(c.parallel ? TaskFlag_None : TaskFlag_SerialSubtasks | TaskFlag_StopOnSubtaskError)), config(c), callback(cb)
{
    assert(cb);
    swSubTasks = prepareSubtasks();

	int count = config.parallel ? qMin(swSubTasks.count(), config.maxThreads) : swSubTasks.count();
	for(int i=0; i < count; i++){
		runningSubTasks << swSubTasks.takeFirst();
	}
    foreach(SequenceWalkerSubtask* sub, runningSubTasks) {
        addSubTask(sub);
    }
}

QList<SequenceWalkerSubtask*> SequenceWalkerTask::prepareSubtasks() {
    QList<SequenceWalkerSubtask*> res;
    
    LRegion seqRange(0, config.seqSize);
    if (config.aminoTrans == NULL ) {
        //try walk in direct in complement strands
        QList<LRegion> chunks = splitRange(seqRange, config.chunkSize, config.overlapSize, config.exOverlapSize, 1, false);
        QList<SequenceWalkerSubtask*> directTasks  = createSubs(chunks, false, false);
        res+=directTasks;
        if (config.complTrans!=NULL) {
            QList<SequenceWalkerSubtask*> complTasks = createSubs(chunks, true, false);
            res+=complTasks;
        }
    } else {
        // try walk in 3 direct and 3 complement translations
        for (int i=0; i<3; i++) {
            LRegion strandRange(seqRange.startPos + i, seqRange.len - i);
            QList<LRegion> chunks = splitRange(strandRange, config.chunkSize, config.overlapSize, config.exOverlapSize, 3, false);
            QList<SequenceWalkerSubtask*> directTasks = createSubs(chunks, false, true);
            res+=directTasks;
        }
        if (config.complTrans!=NULL) {
            for (int i=0; i<3; i++) {
                LRegion strandRange(seqRange.startPos, seqRange.len - i);
                QList<LRegion> chunks = splitRange(strandRange, config.chunkSize, config.overlapSize, config.exOverlapSize, 3, true);
                QList<SequenceWalkerSubtask*> complTasks = createSubs(chunks, true, true);
                res+=complTasks;
            }
        }
    }
    return res;
}

QList<SequenceWalkerSubtask*> SequenceWalkerTask::createSubs(const QList<LRegion>& chunks, bool doCompl, bool doAmino) {
    QList<SequenceWalkerSubtask*> res;
    foreach(const LRegion& chunk, chunks) {
        SequenceWalkerSubtask* t = new SequenceWalkerSubtask(this, chunk, config.seq + chunk.startPos, chunk.len, doCompl, doAmino);
        res.append(t);
    }
    return res;
}

QList<LRegion> SequenceWalkerTask::splitRange(const LRegion& range, int chunkSize, int overlapSize, int exOverlapSize, int normK, bool revertNorm) {
    QList<LRegion> res;
        int halfOverlap = (int) (overlapSize/2.0 + 0.5);
    int stepSize = chunkSize + halfOverlap;
	int segSize = chunkSize + overlapSize;
    if (stepSize > (range.len / 2)) { //the overlap is too large -> avoid splitting
        res.append(range);
        return res;
    }
    
    if (stepSize%normK !=0) {
        stepSize+=normK - stepSize%normK;
    }
	if (segSize%normK !=0) {
        segSize+=normK - segSize%normK;
    }
    
    if (revertNorm) {
        for (int currentEnd = range.endPos(), startPos = range.startPos; currentEnd > startPos;) {
			
			int currentSize = qMin(segSize, currentEnd - startPos);
			bool endCut = false;
			if (currentEnd - currentSize - startPos <= exOverlapSize) {
                currentSize = currentEnd - startPos;
				endCut = true;
            }
			if(currentSize >= halfOverlap)
			{
				LRegion r(currentEnd-currentSize, currentSize);
				res.append(r);
			}
			currentEnd-=stepSize;
			if(endCut) break;
        }
    } else {
        for (int startPos=range.startPos, end = range.endPos(); startPos <= end;) {
			
			int currentSize = qMin(startPos+segSize, end) - startPos;
			bool endCut = false;
			if (end - (startPos + currentSize) <= exOverlapSize) {
                currentSize = end-startPos;
				endCut = true;
            }
			if(currentSize >= halfOverlap)
			{
				LRegion r(startPos, currentSize);
				res.append(r);
			}
			startPos+=stepSize;
			if(endCut) break;
        }
    }
    return res;
}
QList<Task*> SequenceWalkerTask::onSubTaskFinished(Task* subTask)
{
	Q_UNUSED(subTask);
	QList<Task*> res;
	if(!swSubTasks.isEmpty())
		if(runningSubTasks.contains((SequenceWalkerSubtask*)subTask)){
			//assert(runningSubTasks.count()<=config.maxThreadCount);
			//runningSubTasks.removeOne((SequenceWalkerSubtask*)subTask);
			SequenceWalkerSubtask *sub(swSubTasks.takeFirst());
			runningSubTasks << sub;
			res << sub;
			//assert(runningSubTasks.count()<=config.maxThreadCount);
		}
		
	return res;
}
//////////////////////////////////////////////////////////////////////////
// subtask
SequenceWalkerSubtask::SequenceWalkerSubtask(SequenceWalkerTask* _t, const LRegion& glob, const char* _seq, int _len, bool _doCompl, bool _doAmino)
: Task(tr("sequence_walker_subtask"), TaskFlag_DeleteWhenFinished), 
t(_t), globalRegion(glob), localSeq(_seq), originalLocalSeq(_seq), 
localLen(_len), originalLocalLen(_len), doCompl(_doCompl), doAmino(_doAmino)
{
    tpm = Task::Progress_Manual;
}

const char* SequenceWalkerSubtask::getRegionSequence() {
    if (needLocalRegionProcessing()) {
        prepareLocalRegion();
    }
    return localSeq;
}

int SequenceWalkerSubtask::getRegionSequenceLen() {
    if (needLocalRegionProcessing()) {
        prepareLocalRegion();
    }
    return localLen;
}

void SequenceWalkerSubtask::prepareLocalRegion() {
    assert(doAmino || doCompl);

    QByteArray res(localSeq, localLen);
    if (doCompl) {
        //do complement;
        assert(t->getConfig().complTrans!=NULL);
        const QByteArray& complementMap = t->getConfig().complTrans->getOne2OneMapper();
        TextUtils::translate(complementMap, res.data(), res.length());
        TextUtils::reverse(res.data(), res.length());
    }
    if (doAmino) {
        assert(t->getConfig().aminoTrans!=NULL && t->getConfig().aminoTrans->isThree2One());
        t->getConfig().aminoTrans->translate(res.data(), res.length(), res.data(), res.length());
        int newLen = res.length()/3;
        res.resize(newLen);
    }
    processedSeqImage = res;
    localLen = processedSeqImage.size();
    localSeq = processedSeqImage.constData();
}

void SequenceWalkerSubtask::run() {
    assert(!t->hasErrors());
    t->getCallback()->onRegion(this, stateInfo);
}


} //namespace
