/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ExportImageDialog.h"
#include "BioStruct3DGLWidget.h"

#include <core_api/Log.h>
#include <util_gui/DialogUtils.h>

#include <QtGui/QImageWriter>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

namespace GB2 {

static LogCategory log(ULOG_CAT_PLUGIN_BIOSTRUCT_3D);

ExportImageDialog::ExportImageDialog( BioStruct3DGLWidget* widget ) : QDialog(widget), glWidget(widget)
{
    setupUi(this);
    setupFormats();
    setupComponents();    
}

#define IMAGE_DIR "image"
void ExportImageDialog::setupComponents()
{
    widthSpinBox->setValue(glWidget->width());
    heightSpinBox->setValue(glWidget->height());
    
    foreach (QString formatName, supportedFormats) {
        formatsBox->addItem(formatName);
    }

    //todo: add counter to the name to avoid overwrites
    LastOpenDirHelper lod(IMAGE_DIR);
    fileNameEdit->setText(QFileInfo(lod.dir + "image." + formatsBox->currentText()).absoluteFilePath());

    connect(browseFileButton, SIGNAL(clicked()), this, SLOT(sl_onBrowseButtonClick()));
    connect(formatsBox, SIGNAL(currentIndexChanged(const QString&)), this, SLOT(sl_onFormatsBoxItemChanged(const QString&)));

}

void ExportImageDialog::setupFormats() {
    supportedFormats.append("png");
    supportedFormats.append("jpg");
    supportedFormats.append("tiff");
}

void ExportImageDialog::accept() 
{
    QCursor cursor = this->cursor();
    QString fileName(fileNameEdit->text());
    if (fileName.isEmpty()) {
        QMessageBox::warning(this, tr("Error"),tr("The filename is empty!"));
        return;
    }
//     QString format = getFileExtension(fileName);
//     if (!supportedFormats.contains(format)) {
//         QMessageBox::error(this, tr("Error"),tr(QString("Format \"%1\" is not supported!").arg(format)),QMessageBox::Ok);
//         return;
//     }
//     
    log.info(tr("Saving image to '%1'").arg(fileName));
    setCursor(Qt::WaitCursor);
    QPixmap image = glWidget->renderPixmap(widthSpinBox->value(), heightSpinBox->value());
    image.save(fileName, qPrintable(formatsBox->currentText()));
    setCursor(cursor);
    QDialog::accept();
}


void ExportImageDialog::sl_onBrowseButtonClick() {
    QString fileFormats;
    QMap<QString,QString>::const_iterator iter;
    foreach (QString formatName, supportedFormats) {
        fileFormats += formatName.toUpper() + " format ( *." + formatName + ");;"; 
    }
    LastOpenDirHelper lod(IMAGE_DIR);
    lod.url = QFileDialog::getSaveFileName(this, tr("Save image to..."), lod, fileFormats);
    if (lod.url.isEmpty()) {
        return;
    }
    fileNameEdit->setText(lod.url);
    QString formatSuffix = QFileInfo(lod.url).suffix();
    Q_ASSERT(supportedFormats.contains(formatSuffix));
    int index = formatsBox->findText(formatSuffix);
    formatsBox->setCurrentIndex(index);
}

void ExportImageDialog::sl_onFormatsBoxItemChanged( const QString& text )
{
    QString format("");
    foreach (QString formatName, supportedFormats) {
        if (formatName == text) {
            format = text;
            break;
        }
    } 
    Q_ASSERT(!format.isEmpty());
    QString fileName = QFileInfo(fileNameEdit->text()).absoluteFilePath();
    fileName = DialogUtils::ensureFileExt(fileName, QStringList(format));
    fileNameEdit->setText(fileName);
}

} // namespace