/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNAExportPlugin.h"

#include "ExportDialogController.h"
#include "DNAExportTask.h"

#include <core_api/MainWindow.h>
#include <core_api/ProjectView.h>
#include <core_api/AppContext.h>
#include <core_api/MainWindow.h>
#include <core_api/DocumentFormats.h>
#include <core_api/DNATranslation.h>

#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_text/TextUtils.h>

#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>

#include <selection/GObjectSelection.h>
#include <selection/DocumentSelection.h>
#include <selection/SelectionUtils.h>
#include <selection/AnnotationSelection.h>
#include <selection/DNASequenceSelection.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>


#include <QtGui/QMenu>
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>

namespace GB2 {

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    DNAExportPlugin* plug = new DNAExportPlugin();
    return plug;
}

DNAExportPlugin::DNAExportPlugin() : Plugin(tr("dna_export_name"), tr("dna_export_desc")) {
    services.push_back(new DNAExportService(this));
}

//////////////////////////////////////////////////////////////////////////
// Service
DNAExportService::DNAExportService(Plugin* p) 
: Service(p, Service_DNAExport, tr("dna_export_service_name"), tr("dna_export_service_desc"), QList<ServiceType>() << Service_ProjectView)
{
    exportObjectsToFastaAction = NULL;
    exportObjectsToClustalAction = NULL;
    viewContext = NULL;
}

void DNAExportService::serviceStateChangedCallback(ServiceState oldState, bool enabledStateChanged) {
    Q_UNUSED(oldState);

    if (!enabledStateChanged) {
        return;
    }
    if (isEnabled()) {
        exportObjectsToFastaAction = new QAction(tr("export_objects_to_fasta"), this);
        connect(exportObjectsToFastaAction, SIGNAL(triggered()), SLOT(sl_saveObjectsToFasta()));

        exportObjectsToClustalAction = new QAction(tr("export_objects_to_clustal"), this);
        connect(exportObjectsToClustalAction, SIGNAL(triggered()), SLOT(sl_saveObjectsToClustal()));

        ProjectView* pv = AppContext::getProjectView();
        assert(pv!=NULL);
        connect(pv, SIGNAL(si_onDocTreePopupMenuRequested(QMenu&)), SLOT(sl_addToProjectViewMenu(QMenu&)));
        viewContext = new DNAExportViewContext(this);
        viewContext->init();
    } else {
        ProjectView* pv = AppContext::getProjectView();
        assert(pv!=NULL);
        pv->disconnect(this);
        delete exportObjectsToFastaAction;
        exportObjectsToFastaAction = NULL;
        delete exportObjectsToClustalAction;
        exportObjectsToClustalAction = NULL;
        delete viewContext;
        viewContext = NULL;
    }
}

void DNAExportService::sl_addToProjectViewMenu(QMenu& m) {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv);
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, pv->getGObjectSelection());
    QSet<GObject*> set2 = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, pv->getDocumentSelection());
    set+=set2;
    if (set.isEmpty()) {
        return;
    }
    QMenu* sub = new QMenu(tr("export_submenu"));
    sub->addAction(exportObjectsToFastaAction);
    sub->addAction(exportObjectsToClustalAction);
    QAction* beforeAction = GUIUtils::findActionAfter(m.actions(), ACTION_PROJECT__ADD_MENU);
    m.insertMenu(beforeAction, sub);
}

static bool hasComplementForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findComplementTT(so) == NULL) {
            return false;
        } 
    }
    return true;
}

static bool hasAminoForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findAminoTT(so, false) == NULL) {
            return false;
        } 
    }
    return true;
}

void DNAExportService::sl_saveObjectsToFasta() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv);
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, pv->getGObjectSelection());
    //todo: sort by name
    if (set.isEmpty()) {
        QMessageBox::critical(NULL, tr("error"), tr("no_sequences_selected"));
        return;
    }
    bool allowMerge = set.size() > 1;
    bool allowComplement = hasComplementForAll(set);
    bool allowTranslate = hasAminoForAll(set);
        
    ExportDialogController d(allowMerge, allowComplement, allowTranslate, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.allAminoStrands = d.translateAllFrames;
    s.strand = d.strand;
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        assert(so!=NULL);
        s.names.append(so->getGObjectName());
        s.alphabets.append(so->getAlphabet());
        s.sequences.append(so->getSequence());
        s.complTranslations.append(GObjectUtils::findComplementTT(so));
        s.aminoTranslations.append(d.translate ? GObjectUtils::findAminoTT(so, false) : NULL);
    }
    
    Task* t = new DNAExportSequenceTask(s);
//    t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void DNAExportService::sl_saveObjectsToClustal() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, pv->getGObjectSelection());
    if (set.isEmpty()) {
        QMessageBox::critical(NULL, tr("error"), tr("no_sequences_selected"));
        return;
    }
    bool setIsOk = true;
    Q_UNUSED(setIsOk);
    MAlignment ma;
    foreach(GObject* obj, set) {
        DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
        const DNASequence seq = dnaObj->getSequence();
        if (ma.alphabet == NULL) {
            ma.alphabet = dnaObj->getAlphabet();
        } else if (ma.alphabet->getType() != dnaObj->getAlphabet()->getType()) {
            QMessageBox::critical(NULL, tr("error"), tr("different_types_of_alphabets"));
            return;
        } else if (seq.length() > MAX_ALI_LEN) {
            QMessageBox::critical(NULL, tr("error"), tr("sequence_size_is_too_large_for_alignment"));
            return;
        }
        ma.alignedSeqs.append(MAlignmentItem(dnaObj->getGObjectName(), seq.seq));
    }

    ma.normalizeModel();
    
    QString filter = DialogUtils::prepareDocumentsFileFilter(BaseDocumentFormats::CLUSTAL_ALN, false);
    LastOpenDirHelper lod;
    lod.url = QFileDialog::getSaveFileName(NULL, tr("select_file_title"), lod, filter);
    if (lod.url.isEmpty()) {
        return;
    }

    Task* t = new DNAExportAlignmentTask(ma, lod.url);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

//////////////////////////////////////////////////////////////////////////
// DNAExportViewContext

DNAExportViewContext::DNAExportViewContext(QObject* p) 
: GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)
{
}

void DNAExportViewContext::initViewContext(GObjectView* v) {

    GObjectViewAction *a1 = new GObjectViewAction(this, v, tr("export_sequence_selection"));
    connect(a1, SIGNAL(triggered()), SLOT(sl_saveSequenceToFasta()));

    GObjectViewAction *a2 = new GObjectViewAction(this, v, tr("export_annotation_selection"));
    connect(a2, SIGNAL(triggered()), SLOT(sl_saveAnnotationsToFasta()));

    addViewAction(a1);
    addViewAction(a2);
}

void DNAExportViewContext::buildMenu(GObjectView* v, QMenu* m) {
    QMenu* sub = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);
    QList<GObjectViewAction*> list = getViewActions(v);
    assert(!list.isEmpty());
    foreach(GObjectViewAction* a, list) {
        sub->addAction(a);
    }
}


void DNAExportViewContext::sl_saveAnnotationsToFasta() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);
    AnnotationSelection* as = av->getAnnotationsSelection();
    AnnotationGroupSelection* ags = av->getAnnotationsGroupSelection();

    QSet<Annotation*> annotations;

    const QList<AnnotationSelectionData>& aData = as->getSelection();
    foreach(const AnnotationSelectionData& ad, aData) {
        annotations.insert(ad.annotation);
    }

    const QList<AnnotationGroup*>& groups =  ags->getSelection();
    foreach(AnnotationGroup* g, groups) {
        g->findAllAnnotationsInGroupSubTree(annotations);
    }

    if (annotations.isEmpty()) {
        QMessageBox::warning(av->getWidget(), tr("warning"), tr("no_annotation_selected"));
        return;
    }
    bool merge = false;
    bool allowComplement = false;
    bool allowTranslation = true;
    ExportDialogController d(merge, allowComplement, allowTranslation, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    QHash<QString, TriState> usedNames;
    foreach(Annotation* a, annotations) {
        const QString& name = a->getAnnotationName();
        usedNames[name] = usedNames.contains(name) ? TriState_Yes : TriState_No;
    }
    
    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.strand = d.strand;
    s.allAminoStrands = d.translateAllFrames;
    foreach(Annotation* a, annotations) {
        ADVSequenceObjectContext* seqCtx = av->getSequenceContext(a->getGObject());
        if (seqCtx == NULL) {
            continue;
        } 
        DNATranslation* complTT = a->isOnComplementStrand() ? seqCtx->getComplementTT() : NULL;
        DNATranslation* aminoTT = d.translate ? seqCtx->getAminoTT() : NULL;
        LRegion seqReg(0, seqCtx->getSequenceLen());
        const QByteArray& sequence = seqCtx->getSequenceData();

        const QList<LRegion>& location = a->getLocation();
        bool multi = location.size() > 1;
        for (int i=0; i < location.size(); i++) {
            QString prefix = a->getAnnotationName();
            if (multi) {
                prefix+=QString(" part %1 of %2").arg(QString::number(i+1)).arg(QString::number(location.size()));
            }
            QString name = prefix;
            for (int j=0; usedNames.contains(name) && usedNames.value(name) == TriState_Yes; j++) {
                name = prefix + "|" + QString::number(j);
            }
            usedNames[name] = TriState_Yes;
            s.names.append(name);

            LRegion reg = location[i].intersect(seqReg);
            QByteArray partSeq = sequence.mid(reg.startPos, reg.len);//TODO: mid() creates a copy -> optimize!!
            if (complTT!=NULL) {
                complTT->translate(partSeq.data(), partSeq.length());
                TextUtils::reverse(partSeq.data(), partSeq.length()); //todo: do it not in the main thread, but in task!!
            }
            s.sequences.append(partSeq);
            s.alphabets.append(seqCtx->getAlphabet());
            s.complTranslations.append(complTT);
            s.aminoTranslations.append(aminoTT);
        }
    }
    assert(s.names.size() > 0);
    Task* t = new DNAExportSequenceTask(s);
    //t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void DNAExportViewContext::sl_saveSequenceToFasta() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);
    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();
    DNASequenceSelection* sel  = NULL;
    if (seqCtx!=NULL) {
        //TODO: support multi-export..
        sel = seqCtx->getSequenceSelection();
    }
    if (sel == NULL || sel->isEmpty()) {
        QMessageBox::warning(av->getWidget(), tr("warning"), tr("no_sequence_selected"));
        return;
    }

    const QList<LRegion>& regions =  sel->getSelectedRegions();
    bool merge = regions.size() > 1;
    bool complement = seqCtx->getComplementTT()!=NULL;
    bool amino = seqCtx->getAminoTT()!=NULL;
    ExportDialogController d(merge, complement, amino, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    const QByteArray& sequence = seqCtx->getSequenceData();
    DNAAlphabet* al = seqCtx->getAlphabet();
    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.strand = d.strand;
    s.allAminoStrands = d.translateAllFrames;
    foreach(const LRegion& r, regions) {
        QString prefix = QString("region [%1 %2]").arg(QString::number(r.startPos+1)).arg(QString::number(r.endPos()));
        QString name = prefix;
        for (int i=0; s.names.contains(name); i++) {
            name = prefix + "|" + QString::number(i);
        }
        s.names.append(name);
        s.alphabets.append(al);
        s.sequences.append(QByteArray(sequence.constData() + r.startPos, r.len)); //todo: optimize to avoid copying!!
        s.complTranslations.append(seqCtx->getComplementTT());
        s.aminoTranslations.append(d.translate ? seqCtx->getAminoTT() : NULL);
    }
    Task* t = new DNAExportSequenceTask(s);
    //t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}



}//namespace
