/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RepeatFinderDiagonal.h"
#include <stdlib.h>

namespace GB2 {

RepeatFinderDiagonalBase::RepeatFinderDiagonalBase(const char* seqX, quint32 sizeX, const char* seqY, quint32 sizeY, DNAAlphabet* al,  quint32 w, quint32 k, quint32 threads)
:RepeatFinder(seqX, sizeX, seqY, sizeY, al, w, k) , active(FALSE)
{
	nThreads = threads;
	s_done = 0;
	
	WINDOW_SIZE_1 = WINDOW_SIZE - 1;
	SIZE_X_1 = SIZE_X - 1;
	SIZE_Y_1 = SIZE_Y - 1;

	// final results (arrays sizes should grow dynamically in future version)
	resultX = (int *) malloc(1000 * 4);
	resultY = (int *) malloc(1000 * 4);
	resultL = (int *) malloc(1000 * 4);
	resultsCapacity = 1000;
	nResults = 0;
	
	quint32 MIN_SIZE = SIZE_X < SIZE_Y ? SIZE_X : SIZE_Y;
        Q_UNUSED(MIN_SIZE);
	
	START_DIAG = SIZE_X - HALF_WINDOW_SIZE_L;
	END_DIAG = reflective ? 1 : HALF_WINDOW_SIZE_L - SIZE_Y;
	
	//qDebug("START_DIAG=%d, END_DIAG=%d\n", START_DIAG, END_DIAG);
	
	plot_s = reflective ? SIZE_X * (quint64) (SIZE_X_1) / 2 :	SIZE_X * (quint64) SIZE_Y;
	plot_s -= WINDOW_SIZE * (WINDOW_SIZE_1) / (1 + reflective);
	report_s = plot_s / (nThreads * 100);
	if (plot_s < 50000 * (quint64) 100000) {
		report_s = report_s * 10;
	} else if (plot_s < 100000 * (quint64) 100000) {
		report_s = report_s * 5;
	}	
	
	//qDebug("S %I64d, reportS %I64d\n", plot_s, report_s);
	//qDebug(qDebug("Calc inited! size1: %d size2: %d w: %d k: %d , reflective: %s\n",
	//	SIZE_X, SIZE_Y, WINDOW_SIZE, K, reflective ? "YES" :	"NO");
}





RepeatResult* RepeatFinderDiagonalBase::getResult(quint32 index){
	if (index >= nResults) {
		return NULL;
	} else {
		edge.x = resultX[index];
		edge.y = resultY[index];
		edge.l = resultL[index];
		return &edge;
	}
}


quint32 RepeatFinderDiagonalBase::getNumResults(){
	return nResults;
}


RepeatFinderDiagonalBase::~RepeatFinderDiagonalBase(){
	active = FALSE;
	waitThreadsStop();
	if (resultsCapacity > 0) {
		free(resultX);
		free(resultY);
		free(resultL);
		nResults = 0;
		resultsCapacity = 0;
	}
	qDeleteAll(calcThreads);
}

void RepeatFinderDiagonalBase::startCalculation(){
	active = TRUE;
	stateMutex.lock();
//	qDebug("Calculation started size1: %d size2: %d w: %d k: %d threads: %d\n",
//		SIZE_X, SIZE_Y, WINDOW_SIZE, K, nThreads);
	activeThreads = nThreads;
	if (reflective) {
		addToResults(0, 0, SIZE_X);
	}
	logSizeDone(0);
	for (int i = 0; i < nThreads; i++) {
		QThread* t = newCalcThread(i);
		calcThreads.append(t);
		t->start(QThread::LowPriority);
	}
	stateMutex.unlock();
}

void RepeatFinderDiagonalBase::terminateCalculation(){
	//we should not terminate locking thread!
	stateMutex.lock();
	active = FALSE;
	waitThreadsStop();
	activeThreads = 0;
	stateMutex.unlock();
}

void RepeatFinderDiagonalBase::ensureResultsCapacity(int n){
	if (resultsCapacity < nResults + n) {
		quint32 newCapacity = resultsCapacity * 2;
		if (newCapacity < nResults + n) {
			newCapacity = nResults + n;
		}
		quint32 nBytes = newCapacity * 4;
		//printf("reallocate from %d to %d bytes:%d\r\n", resultsCapacity, newCapacity, nBytes);			
		resultX = (int *) realloc(resultX, nBytes);
		resultY = (int *) realloc(resultY, nBytes);
		resultL = (int *) realloc(resultL, nBytes);
		resultsCapacity = newCapacity;
	}
}

// adds single result to global results 
void RepeatFinderDiagonalBase::addToResults(int x, int y, int len){
	resultsMutex.lock();
	ensureResultsCapacity(1);
	resultX[nResults] = x + 1;
	resultY[nResults] = y + 1;
	resultL[nResults] = len;
	nResults++;
	resultsMutex.unlock();
}

// adds single result to global results 
void RepeatFinderDiagonalBase::addToResults(int* x, int* y, int* len, int n){
	resultsMutex.lock();
	ensureResultsCapacity(n * (reflective ? 2 : 1));
	
	for (int i = 0; i < n; i++) {
		resultX[nResults] = x[i] + 1;
		resultY[nResults] = y[i] + 1;
		resultL[nResults] = len[i];
		nResults++;
	}
	if (reflective) {
		for (int i = 0; i < n; i++) {
			resultX[nResults] = y[i] + 1;
			resultY[nResults] = x[i] + 1;
			resultL[nResults] = len[i];
			nResults++;
		}
	}
	resultsMutex.unlock();
}


void RepeatFinderDiagonalBase::logSizeDone(quint64 d_s){
	resultsMutex.lock();
	s_done += d_s;
	quint64 percent_done = (((quint64) 100) * s_done) / plot_s;
	
//	printf("DPpercent done is %i\n", int(100.0f*float(s_done)/float(plot_s)));

	if (percent_done <= 100) {
		// it's possible percent_done > 100 (plohoe okruglenie)
		if (resultsListener!= NULL) {
			resultsListener->percentDone((int) percent_done);
		}
	}
	resultsMutex.unlock();
}

void RepeatFinderDiagonalBase::threadFinished(){
	resultsMutex.lock();
	activeThreads--;
//	qDebug("thread finished, active left: %d\n", activeThreads);
	if (activeThreads == 0 && resultsListener != NULL) {
		resultsListener->calculationFinished();
	}
	active = FALSE;
	resultsMutex.unlock();
}

void RepeatFinderDiagonalBase::waitThreadsStop(){
	foreach(QThread* t, calcThreads) {
		t->wait();
	}
}
// --------------------- RepeatFinderDiagonalWK realization ------------------ //

RepeatFinderDiagonalWK::RepeatFinderDiagonalWK(const char* seqX, quint32 sizeX, const char* seqY, quint32 sizeY, DNAAlphabet* al,  quint32 w, quint32 k, quint32 threads)
: RepeatFinderDiagonalBase(seqX, sizeX, seqY, sizeY, al, w, k, threads)
{
//	qDebug("RepeatFinderDiagonalWK created\n");
	SAFE_SIZE_X = SIZE_X - WINDOW_SIZE; // we should not calculate diagonals near the border
	SAFE_SIZE_Y = SIZE_Y - WINDOW_SIZE; // we should not calculate diagonals near the border;
}



QThread* RepeatFinderDiagonalWK::newCalcThread(quint32 threadNum){
	return new RepeatFinderWKThread(this, threadNum);
}

//------------ RepeatFinderWKThread methods -------------//

RepeatFinderDiagonalWK::RepeatFinderWKThread::RepeatFinderWKThread(RepeatFinderDiagonalWK* owner, quint32 threadNum)
: owner(owner), num(threadNum), dataX(owner->seqX), dataY(owner->seqY), unknownChar(owner->unknownChar){
	// pre-calculation matches for one diagonal (arrays sizes should grow dynamically in future version)
	int max = owner->SIZE_X > owner->SIZE_Y ? owner->SIZE_X : owner->SIZE_Y;
	matchesX = (int *) malloc((max / owner->HALF_WINDOW_SIZE_L) * 4);
	matchesY = (int *) malloc((max / owner->HALF_WINDOW_SIZE_L) * 4);
	//number of mismatches in WINDOW_SIZE/2 pre-calculation
	matchesC = (int *) malloc((max / owner->HALF_WINDOW_SIZE_L) * 4);
	nMatches = 0;
	
	int min = owner->SIZE_X < owner->SIZE_Y ? owner->SIZE_X : owner->SIZE_Y;
	int maxEdgesOnDiagonal = min / (owner->WINDOW_SIZE * 2);
	d_resultX = (int *) malloc(maxEdgesOnDiagonal * 4);
	d_resultY = (int *) malloc(maxEdgesOnDiagonal * 4);
	d_resultL = (int *) malloc(maxEdgesOnDiagonal * 4);
	d_nResults = 0;
}

RepeatFinderDiagonalWK::RepeatFinderWKThread::~RepeatFinderWKThread(){
	if (matchesX != NULL) {
		free(matchesX);
		free(matchesY);
		free(matchesC);
	}
	
	if (d_resultX != NULL) {
		free(d_resultX);
		free(d_resultY);
		free(d_resultL);
	}
}
//int DEBUG_N_MATCHES = 0;

void RepeatFinderDiagonalWK::RepeatFinderWKThread::run(){
	//	DEBUG_N_MATCHES =0;
//	qDebug("thread started: %d\n", num);
	bool& active = owner->active;
	quint64 dS = 0;
	const quint32 SIZE_X = owner->SIZE_X;
	const quint32 SIZE_Y = owner->SIZE_Y;
    //const quint32 SAFE_SIZE_X = owner->SAFE_SIZE_X;
    //const quint32 SAFE_SIZE_Y = owner->SAFE_SIZE_Y;
    const quint64 report_s = owner->report_s;
	
	const int start_diag = owner->START_DIAG;
	const int end_diag = owner->END_DIAG;
	const int nThreads = owner->nThreads;
	int x, y, d = start_diag - num;
	while (d >= end_diag && active) {
		x = d > 0 ? d : 0;
		y = d > 0 ? 0 : -d;
		//(d>0 && ((y=0) || (x=d))) || ((x=0) || (y=-d));
		processDiagonal3(x, y);
		// TODO: OPTIMIZE		
		dS += d > 0 ? (SIZE_Y < SIZE_X - x ? SIZE_Y : SIZE_X - x) :
		(SIZE_X < SIZE_Y - y ? SIZE_X : SIZE_Y - y);
		if (dS >= report_s) {
			owner->logSizeDone(dS);
			msleep(5);
			dS = 0;
		}
		d -= nThreads;
	}
//	qDebug("thread finishing: %d\n", num);
	//printf("matches %d\n", DEBUG_N_MATCHES);
	owner->threadFinished();
}




void RepeatFinderDiagonalWK::RepeatFinderWKThread::processDiagonal(int x, int y){
	const int step = owner->WINDOW_SIZE;
	const int C = owner->C;
	const char* xseq = dataX + x + step - 1;
	const char* yseq = dataY + y + step - 1;
	
	const char* xseqMax = dataX + owner->SIZE_X;
	const char* yseqMax = dataY + owner->SIZE_Y;
	
	while (xseq < xseqMax && yseq < yseqMax) {
		int c = 0;
		for (int i = 0; i < step; i++, xseq--, yseq--) {
			if (*xseq != *yseq) {
				c++;
				if (c > C) {
					goto a;
				}
			}
		}
		// edge found!
		xseq += step;//going to start position
		yseq += step;
		matchesX[nMatches] = xseq - dataX;
		matchesY[nMatches] = yseq - dataY;
		matchesC[nMatches] = c; // number_of_not_matches
		nMatches++;
		//		DEBUG_N_MATCHES++;
a:
		xseq += step;
		yseq += step;
	}
	
	if (nMatches > 0) {
		processMatches();// checks all matches for this diagonal
		nMatches = 0;
		if (d_nResults > 0) {
			addLocalResultsToGlobal();
		}
	}
}

void RepeatFinderDiagonalWK::RepeatFinderWKThread::processDiagonal3(int x, int y){
	const int step = owner->WINDOW_SIZE;
	const int C = owner->C;
	const char* xseq = dataX + x + step - 1;
	const char* yseq = dataY + y + step - 1;
	char unknown = unknownChar;
	
	const char* xseqMax = dataX + owner->SIZE_X;
	const char* yseqMax = dataY + owner->SIZE_Y;
	//char* xseqStart = xseq-step;
	while (xseq < xseqMax && yseq < yseqMax) {
		int c = 0;
//		xseqStart = xseq - step;
        for (const char* xseqStart = xseq - step;xseq > xseqStart && (c+=(*xseq!=*yseq || *xseq==unknown))<=C ; xseq--, yseq--){}
		if (c > C) {
	//		xseqStart  = xseq;
			xseq+= step;
			yseq+= step;
		} else {
			// edge found!
			xseq+= step;//going to start position
	//		xseqStart = xseq;
			yseq+= step;
			matchesX[nMatches] = xseq - dataX;
			matchesY[nMatches] = yseq - dataY;
			matchesC[nMatches] = c; // number_of_not_matches
			nMatches++;
			//DEBUG_N_MATCHES++;
			xseq+= step;
			yseq+= step;
		}
	}
	
	if (nMatches > 0) {
		processMatches();// checks all matches for this diagonal
		nMatches = 0;
		if (d_nResults > 0) {
			addLocalResultsToGlobal();
		}
	}
}



void RepeatFinderDiagonalWK::RepeatFinderWKThread::processDiagonal2(int x, int y){
	const int step = owner->WINDOW_SIZE;
	const int C = owner->C;
	const char* xseqMax = dataX + owner->SIZE_X;
	const char* yseqMax = dataY + owner->SIZE_Y;
	const int n4 = step / 4;
	const int n1 = step % 4;
	const int n4step = step - n1;
	
	
	const char* xseq = dataX + x + step - 1;
	const char* yseq = dataY + y + step - 1;
	
	uint r;
	char* c0=(char*)&r;
	char* c1=c0+1;
	char* c2=c0+2;
	char* c3=c0+3;
	
	while (xseq < xseqMax && yseq < yseqMax) {
		int c = 0;
		int* xx = (int*)(xseq-3);
		int* yy = (int*)(yseq-3);
		int i;
		for (i = 0; i < n4; i++, xx--, yy--) {
			r = (*xx) ^ (*yy);
			if ((c+=(*c0!=0)+(*c1!=0)+(*c2!=0)+(*c3!=0)) > C) {
				xseq=((const char*)xx);
				yseq=((const char*)yy);
				goto a;
			}
		}
		xseq-=n4step;
		yseq-=n4step;
		if (n1 > 0) {
			for (i = 0; i < n1; i++, xseq--, yseq--) {
				if (*xseq != *yseq) {
					c++;
					if (c > C) {
						goto a;
					}
				}
			}
		}
		
		// edge found!
		xseq += step;//going to start position
		yseq += step;
		matchesX[nMatches] = xseq - dataX;
		matchesY[nMatches] = yseq - dataY;
		matchesC[nMatches] = c; // number_of_mismatches
		nMatches++;
a : 
		xseq += step;
		yseq += step;
	}
	
	if (nMatches > 0) {
		processMatches();// checks all matches for this diagonal
		nMatches = 0;
		if (d_nResults > 0) {
			addLocalResultsToGlobal();
		}
	}
}


void RepeatFinderDiagonalWK::RepeatFinderWKThread::processMatches(){
	const char* c1 = dataX;
	const char* c2 = dataY;
	const int WINDOW_SIZE = owner->WINDOW_SIZE;
	const int SIZE_X_1 = owner->SIZE_X_1;
	const int SIZE_Y_1 = owner->SIZE_Y_1;
	const int K = owner->K;

	//printf("processing matches: %d\n",nMatches);
	int lastWStartX = -1;
	char unknown = unknownChar;
	for (int i = 0; i < nMatches; i++) {
		int x = matchesX[i];
		if (x <= lastWStartX) {
			//this match is combined with previous into one edge
			continue;
		}
		int y = matchesY[i];
		int weight = WINDOW_SIZE - matchesC[i];
		
		int startXEdge = x - WINDOW_SIZE + 1;
		int maxAllowedUnmatchX = x + WINDOW_SIZE - 1;

		while (x < SIZE_X_1 && y < SIZE_Y_1) {
			//go forward
			x++;
			y++;
			weight += ( (c1[x] == c2[y] && c1[x]!=unknown)? 1 : 0) +
				(c1[x - WINDOW_SIZE] == c2[y - WINDOW_SIZE] ? -1 : 0);
			if (weight >= K) {
				if (startXEdge == -1) {
					// new edge
					startXEdge = x - WINDOW_SIZE + 1;
					//printf("start of the edge found: x:%d y:%d startEdge:%d, weight:%d\r\n", x, y, startXEdge, weight);
				}	// else old edge is growing
			} else {
				// we have no active edge
				if (startXEdge != -1) {
					//end of the edge
					//printf("end of the window found: x:%d y:%d startXEdge:%d\r\n", x , y, startXEdge);
					int len = x - startXEdge; // startXEdge is inclusive, x not inclusive
					addToLocalResults(startXEdge, y - len, len);
					startXEdge = -1;
				}
				if (x > maxAllowedUnmatchX) {
					// should not go anymore
					break;
				}
			}
		}
		if (startXEdge != -1) {
			// ok we found edge
			//printf("end of the window on complete: x:%d y:%d startXEdge:%d\r\n", x , y, startXEdge);
			int len = x - startXEdge + 1; // both variable is inclusive  here
			addToLocalResults(startXEdge, y - len + 1, len);
		}
		lastWStartX = x - (WINDOW_SIZE - 1);
	}
}

void RepeatFinderDiagonalWK::RepeatFinderWKThread::addToLocalResults(int x, int y, int len){
	//printf("add to local: %d %d %d\n", x, y , len);
	//proverka na skleivalie 
	if (d_nResults > 0 &&
		(d_resultX[d_nResults - 1] + d_resultL[d_nResults - 1] >= x)) {
		//skleivanie
		d_resultL[d_nResults - 1] = x + len - d_resultX[d_nResults - 1];
	} else {
		// new edge
		d_resultX[d_nResults] = x;
		d_resultY[d_nResults] = y;
		d_resultL[d_nResults] = len;
		d_nResults++;
	}
}

//int results=0;

void RepeatFinderDiagonalWK::RepeatFinderWKThread::addLocalResultsToGlobal(){
	//	results+=d_nResults;
	//	printf(">>%d\n",results);
	owner->addToResults(d_resultX, d_resultY, d_resultL, d_nResults);
	d_nResults = 0;
}
/*
void RepeatFinderDiagonalWK::startCalculation2(){
	active = TRUE;
//	qDebug("Calculation started size1: %d size2: %d w: %d k: %d threads: %d\n",
//		SIZE_X, SIZE_Y, WINDOW_SIZE, K, nThreads);
	if (reflective) {
		addToResults(0, 0, SIZE_X);
	}
	logSizeDone(0);
	RepeatFinderWKThread* t = (RepeatFinderWKThread*)newCalcThread(0);
	t->run();
	delete t;
}*/

}//namespace
