/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TestRunnerTask.h"

#include <core_api/AppContext.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <test_framework/GTest.h>

namespace GB2
{

TestRunnerTask::TestRunnerTask(const QList<GTestState*>& tests, const GTestEnvironment* _env, int testSizeToRun)
: Task( tr("test_runner"), TaskFlags_NR_DWF),  env(_env)
{
    tpm = Progress_Manual;

    sizeToRun = testSizeToRun;
    assert(!tests.isEmpty());
    awaitingTests=tests;
    foreach(GTestState *t,awaitingTests){
        t->clearState();
    }
    for (int i=0; !awaitingTests.isEmpty() && i < sizeToRun; i++){
        GTestState *t=awaitingTests.takeFirst();
        addSubTask(new LoadTestTask(t));
    }
}


QList<Task*> TestRunnerTask::onSubTaskFinished(Task* subTask) {
    stateInfo.progress = 100*getSubtasks().size()/(getSubtasks().size() + 2*awaitingTests.size());
    QList<Task*> res;
    if(isCanceled()){
        return res;
    }
    LoadTestTask* loader = qobject_cast<LoadTestTask*>(subTask);
    if (loader == NULL) {
        GTest* test = qobject_cast<GTest*>(subTask);
        assert(test);
        test->cleanup();
        GTestState* testState = stateByTest.value(test);
        assert(testState!=NULL);
        assert(testState->isNew());
        if (test->hasErrors()) {
            testState->setFailed(test->getStateInfo().error);
        } else {
            testState->setPassed();
        }
        if(!awaitingTests.isEmpty()){
            GTestState *t=awaitingTests.takeFirst();
            res.append(new LoadTestTask(t));
        }
    } else {
        GTestState* testState = loader->testState;
        if (loader->hasErrors()) {
            testState->setFailed(loader->getStateInfo().error);
        }  else  {
            GTestFormatId id = testState->getTestRef()->getFormatId();
            GTestFormat* tf = AppContext::getTestFramework()->getTestFormatRegistry()->findFormat(id);
            if (tf == NULL) {
                testState->setFailed(tr("test_format_not_supported_%1").arg(id));
            } else {
                QString err;
                GTest* test = tf->createTest(testState->getTestRef()->getShortName(), NULL, env, loader->testData, err);
                if (test == NULL) {
                    testState->setFailed(err);
                } else {
                    stateByTest[test] = testState;
                    res.append(test);
                }
            }
        }
    }
    return res;
}


LoadTestTask::LoadTestTask(GTestState* _testState)  
: Task( tr("test_loader_for_%1").arg(_testState->getTestRef()->getShortName()), TaskFlag_DeleteWhenFinished), testState(_testState)
{
}

void LoadTestTask::run() {
    const QString& url = testState->getTestRef()->getURL();
    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        stateInfo.error= tr("cant_open_file: %1").arg(url);
        return;
    }
    testData = f.readAll();
    f.close();
}


} //namespace
