/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TestViewController.h"
#include "TestRunnerPlugin.h"
#include "TestRunnerTask.h"
#include "TestViewReporter.h"

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QFormLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QFormLayout>
#include <QtGui/QLineEdit>
#include <QtGui/QMenu>
#include <QtGui/QToolBar>


/* TRANSLATOR GB2::TestViewController */

//todo: remember splitter geom

namespace GB2 {

#define SETTINGS_ROOT QString("test_runner/view/")

#define ICON_FAILD_DIR QIcon(":/plugins/test_runner/images/folder_faild.png")           
#define ICON_SUCCES_DIR QIcon(":/plugins/test_runner/images/folder_ok.png")
#define ICON_NOTRUN_DIR QIcon(":/plugins/test_runner/images/folder.png")
#define ICON_FAILD_TEST QIcon(":/plugins/test_runner/images/test_faild.png")
#define ICON_SUCCES_TEST QIcon(":/plugins/test_runner/images/test_ok.png")
#define ICON_NOTRUN_TEST QIcon(":/plugins/test_runner/images/test.png")

TestViewController::TestViewController(TestRunnerService* s) : MWMDIWindow(tr("window_name")), service(s)
{
    task=NULL;
    reporterForm=NULL;
    setupUi(this);
    tree->setContextMenuPolicy(Qt::CustomContextMenu);
    tree->setColumnWidth(0, AppContext::getSettings()->getValue(SETTINGS_ROOT + "treeColWidth", 400).toInt());
    tree->setSelectionMode(QAbstractItemView::ExtendedSelection);

    splitter->setStretchFactor(0, 10);
    splitter->setStretchFactor(1, 3);

    addTestSuiteAction = new QAction(tr("add_test_suite"), this);
    connect(addTestSuiteAction, SIGNAL(triggered()), SLOT(sl_addTestSuiteAction()));

    removeTestSuiteAction = new QAction(tr("remove_test_suite"), this);
    connect(removeTestSuiteAction, SIGNAL(triggered()), SLOT(sl_removeTestSuiteAction()));

    runAllSuitesAction = new QAction(tr("run_all_suites"), this);
    connect(runAllSuitesAction, SIGNAL(triggered()), SLOT(sl_runAllSuitesAction()));

    runSelectedSuitesAction = new QAction(tr("run_selected_suites"), this);
    QKeySequence runAllKS(Qt::CTRL+Qt::Key_R);
    runSelectedSuitesAction->setShortcut(runAllKS);
    connect(runSelectedSuitesAction, SIGNAL(triggered()), SLOT(sl_runSelectedSuitesAction()));

    stopSuitesActions = new QAction(tr("stop_suites"), this);
    connect(stopSuitesActions, SIGNAL(triggered()), SLOT(sl_stopSuitesActions()));

    setEnvAction = new QAction(tr("set_env_action"), this);
    connect(setEnvAction, SIGNAL(triggered()), SLOT(sl_setEnvAction()));

    report = new QAction(tr("report"), this);
    connect(report, SIGNAL(triggered()), SLOT(sl_report()));

    connect(s, SIGNAL(si_testSuiteAdded(GTestSuite*)), SLOT(sl_suiteAdded(GTestSuite*)));
    connect(s, SIGNAL(si_testSuiteRemoved(GTestSuite*)), SLOT(sl_suiteRemoved(GTestSuite*)));

    connect(tree, SIGNAL(itemSelectionChanged()),SLOT(sl_treeItemSelectionChanged()));
    connect(tree, SIGNAL(customContextMenuRequested(const QPoint&)),SLOT(sl_treeCustomContextMenuRequested(const QPoint&)));

    connect(tree, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)) , SLOT(sl_treeDoubleClicked(QTreeWidgetItem*, int)));


    const QList<GTestSuite*> suites =  s->getTestSuites();
    foreach(GTestSuite* ts, suites) {
        addTestSuite(ts);
    }
    updateState();
}

bool TestViewController::onCloseEvent() {
    AppContext::getSettings()->setValue(SETTINGS_ROOT + "treeColWidth", tree->columnWidth(0));

    if(task!=NULL)task->cancel();
    return true;
}

void TestViewController::setupMDIToolbar(QToolBar* tb) {
    tb->addAction(addTestSuiteAction);
    tb->addAction(removeTestSuiteAction);
    tb->addAction(setEnvAction);
    tb->addAction(runAllSuitesAction);
    tb->addAction(runSelectedSuitesAction);
    tb->addAction(stopSuitesActions);
    tb->addAction(report);
}

void TestViewController::setupViewMenu(QMenu* m) {
    m->addAction(addTestSuiteAction);
    m->addAction(removeTestSuiteAction);
    m->addAction(setEnvAction);
    m->addAction(runAllSuitesAction);
    m->addAction(runSelectedSuitesAction);
    m->addAction(stopSuitesActions);
    m->addAction(report);
}

TVTSItem* TestViewController::findTestSuiteItem(GTestSuite* ts) const {
    for (int i=0, n=tree->topLevelItemCount(); i<n; i++) {
        TVItem* item = (TVItem*)tree->topLevelItem(i);
        assert(item->type == TVItem_TestSuite);
        TVTSItem* tsi = (TVTSItem*)item;
        if (tsi->ts == ts) {
            return tsi;
        }
    }
    return NULL;
}

TVTestItem* TestViewController::findTestViewItem(GTestRef* testRef) const {
    for (int i=0, n=tree->topLevelItemCount(); i<n; i++) {
        TVItem* item = (TVItem*)tree->topLevelItem(i);
        assert(item->type == TVItem_TestSuite);
        TVTSItem* sItem = (TVTSItem*)item;
        TVTestItem* rezult = findTestViewItemRecursive(testRef,sItem);
        if(rezult)return rezult;
    }
    return NULL;
}
TVTestItem* TestViewController::findTestViewItemRecursive(GTestRef* testRef,TVItem* sItem) const {
    for(int j = 0, m = sItem->childCount(); j<m; j++) {
        TVItem* item = (TVItem*)sItem->child(j);
        if(item->isTest()){
            TVTestItem* tItem  = (TVTestItem*)item;
            if (tItem->testState->getTestRef() == testRef) {
                return tItem;
            }
        }
        else{
            assert(item->isSuite());
            TVTestItem* rezult = findTestViewItemRecursive(testRef,(TVTSItem*)item);
            if(rezult)return rezult;
        }
    }
    return NULL;
}

TVTSItem* TestViewController::getFolder(TVItem* element,const QString* firstDirName)const{
    for(int j = 0, m = element->childCount(); j<m; j++) {
        TVItem* item = (TVItem*)element->child(j);
        if(item->isSuite()){
            TVTSItem *ditem= (TVTSItem *)item;
            if(ditem->name==firstDirName){
                return ditem;
            }
        }
    }
    return NULL;
}
    
void TestViewController::killAllChildForms(){
    
    if(reporterForm!=NULL){
        AppContext::getMainWindow()->getMDIManager()->closeMDIWindow((MWMDIWindow*) reporterForm);
        //reporterForm = NULL;
        assert(reporterForm == NULL);
    }
    return;
}
//<-----------------------------------------------------------------------------------
void TestViewController::addTestSuite(GTestSuite* ts) {
    TVTSItem* tsi = new TVTSItem(ts);
    foreach(GTestRef* t, ts->getTests()) {

        QString firstDirName=t->getShortName().section('/', 0 , 0 );//find first directory name
        if(t->getShortName()==firstDirName){
            addTest(tsi, t);
        }
        else{
            TVTSItem* curDir=getFolder(tsi,&firstDirName);
            QString otherPath=t->getShortName().section('/', 1 );//find other path
            if(curDir){//find if dir already exist
                addFolderTests(curDir,t,&otherPath);  
            }
            else{
                TVTSItem* newDir=new TVTSItem(firstDirName);
                tsi->addChild(newDir);
                addFolderTests(newDir,t,&otherPath);
            }
        }
    }
    tsi->updateVisual();
    tree->addTopLevelItem(tsi);
}

void TestViewController::addFolderTests(TVTSItem* tsi, GTestRef* testRef,const QString* curPath) {
    
    QString firstDirName=curPath->section('/', 0 , 0 );
    if(*curPath==firstDirName){
        addTest(tsi, testRef);
    }
    else{
        TVTSItem* curDir=getFolder(tsi,&firstDirName);
        QString otherPath=curPath->section('/', 1 );//find other path
        if(curDir){//find if dir already exist
            addFolderTests(curDir,testRef,&otherPath);  
        }
        else{
            TVTSItem* newDir=new TVTSItem(firstDirName);
            tsi->addChild(newDir);
            addFolderTests(newDir,testRef,&otherPath);
        }
    }
    tsi->updateVisual();
}

void TestViewController::addTest(TVTSItem* tsi, GTestRef* testRef) {
    GTestState* testState = new GTestState(testRef);
    connect(testState, SIGNAL(si_stateChanged(GTestState*)), SLOT(sl_testStateChanged(GTestState*)));
    TVTestItem* ti = new TVTestItem(testState);
    tsi->addChild(ti);
}

void TestViewController::sl_suiteAdded(GTestSuite* ts) {
    addTestSuite(ts);
    updateState();
}

void TestViewController::sl_suiteRemoved(GTestSuite* ts) {
    TVTSItem* item = findTestSuiteItem(ts);
    assert(item);
    delete item;
    updateState();
}

void TestViewController::sl_treeCustomContextMenuRequested(const QPoint & pos) {
    Q_UNUSED(pos);

    QMenu* menu = new QMenu(tree);    
    menu->addAction(addTestSuiteAction);
    menu->addAction(removeTestSuiteAction);
    menu->addAction(runAllSuitesAction);
    menu->addAction(runSelectedSuitesAction);
    menu->addAction(stopSuitesActions);
    menu->addAction(setEnvAction);
    menu->exec(QCursor::pos());
    menu->addAction(report);
}

void TestViewController::sl_treeItemSelectionChanged() {
    updateState();
}

QList<TVTSItem*> TestViewController::getSelectedSuiteItems() const {
    QList<TVTSItem*> res;
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    foreach(QTreeWidgetItem* i, items) {
        TVItem* item = (TVItem*)i;
        if (item->isSuite()) {
            res.append((TVTSItem*)item);
        }
    }
    return res;
}

QList<TVTestItem*> TestViewController::getSelectedTestItems() const {
    QList<TVTestItem*> res;
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    foreach(QTreeWidgetItem* i, items) {
        TVItem* item = (TVItem*)i;
        if (item->isTest()) {
            res.append((TVTestItem*)item);
        }
    }
    return res;
}

bool  TestViewController::allSuitesIsInRoot(const QList<TVTSItem*> suites) const {
    if(suites.isEmpty())return false;
    bool rezult=true;
    foreach(TVTSItem* si, suites){
        if(si->ts==NULL)rezult=false;
    }
    return rezult;
}

void TestViewController::updateState() {
    QList<TVTSItem*> suites = getSelectedSuiteItems();
    QList<TVTestItem*> tests = getSelectedTestItems();

    removeTestSuiteAction->setEnabled(allSuitesIsInRoot(suites) && task==NULL && tests.isEmpty());
    setEnvAction->setEnabled(task==NULL);
    runAllSuitesAction->setEnabled(!service->getTestSuites().isEmpty() && task==NULL);
    report->setEnabled(!service->getTestSuites().isEmpty() && task==NULL);
    runSelectedSuitesAction->setEnabled((!suites.isEmpty() || !tests.isEmpty()) && task==NULL );
    stopSuitesActions->setEnabled( task!=NULL);

    if (tree->currentItem()!=NULL) {
        TVItem* i = (TVItem*)tree->currentItem();
        contextInfoEdit->setText(i->getRichDesc());
    }
}


void TestViewController::sl_addTestSuiteAction() {
    QString dir = AppContext::getSettings()->getValue(SETTINGS_ROOT + "lastDir", QString()).toString();
    QStringList fileNames=QFileDialog::getOpenFileNames(this, tr("select_ts_caption"),dir);
    foreach(QString file,fileNames){
        if (file.isEmpty()) {
            return;
        }
        QFileInfo fi(file);
        dir = fi.absoluteDir().absolutePath();
        AppContext::getSettings()->setValue(SETTINGS_ROOT + "lastDir", dir);
        
        QString url = fi.absoluteFilePath();
        if (service->findTestSuiteByURL(url)!=NULL) {
            QMessageBox::critical(this, tr("error"), tr("test_suite_already_added"));
            //return;
        }
        else{
            QString err;
            GTestSuite* ts = GTestSuite::readTestSuite(url, err);
            if (ts == NULL) {
                assert(!err.isEmpty());
                QMessageBox::critical(this, tr("error"), tr("error_reading_ts %1").arg(err));
                //return;
            }
            else{
                service->addTestSuite(ts);
            }
        }
    }
}

void TestViewController::sl_removeTestSuiteAction() {
    TVItem* item = (TVItem*)tree->currentItem();
    assert(item && item->isSuite());
    TVTSItem* sItem = (TVTSItem*)item;
    service->removeTestSuite(sItem->ts);

}

void TestViewController::sl_runAllSuitesAction() {

    assert(task==NULL);
    GTestEnvironment* env = service->getEnv();
    if (env->containsEmptyVars()) {
        QMessageBox::critical(this, tr("error"), tr("env_vars_not_set"));
        return;
    }
    QList<GTestState*> testsToRun;
    for (int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        TVItem* item = (TVItem*)tree->topLevelItem(i);
        assert(item->isSuite());
        bool runAll=true;
        testsToRun+= getSubTestToRun(item,runAll);
    }
    if (!testsToRun.isEmpty()) {
        createAndRunTask(testsToRun);
    }
}

void TestViewController::sl_runSelectedSuitesAction() {

    assert(task==NULL);
    GTestEnvironment* env = service->getEnv();
    if (env->containsEmptyVars()) {
        QMessageBox::critical(this, tr("error"), tr("env_vars_not_set"));
        return;
    }
    QList<GTestState*> testsToRun;
    for (int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        TVItem* item = (TVItem*)tree->topLevelItem(i);
        assert(item->isSuite());
        bool runAll=false;
        if(item->isSelected())runAll=true;
        testsToRun+= getSubTestToRun(item,runAll);
    }
    if (!testsToRun.isEmpty()) {
        createAndRunTask(testsToRun);
    }
}

QList<GTestState*> TestViewController::getSubTestToRun(TVItem* sItem,bool runAll)const{   
    QList<GTestState*> testsToRun;
    for(int j = 0, m = sItem->childCount(); j<m; j++) {
        TVItem* item = (TVItem*)sItem->child(j);
        if(item->isTest()){
            TVTestItem* tItem  = (TVTestItem*)item;
            if(tItem->isSelected() || runAll){
                testsToRun.append(tItem->testState);
            }
        }
        
        else{
            assert(item->isSuite());
            TVTSItem* tItem2  = (TVTSItem*)item;
            if(tItem2->isSelected()){
                testsToRun+= getSubTestToRun(tItem2,true);
            }
            else{
                testsToRun+= getSubTestToRun(tItem2,runAll);
            }
        }
    }
return testsToRun;
}

void TestViewController::sl_stopSuitesActions() { 

    task->cancel();
}

void TestViewController::createAndRunTask(const QList<GTestState*>& testsToRun) {
    bool ok;
    QTime* temp=new QTime;
    *temp=QTime::currentTime();
    startRunTime = temp;
    int numberTestsToRun= service->getEnv()->getVar("NUM_THREADS").toInt(&ok);
    if(!ok || numberTestsToRun <= 0 ){numberTestsToRun=5;}
    TestRunnerTask* ttask = new TestRunnerTask(testsToRun, service->getEnv(),numberTestsToRun);

    task = ttask;
    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_taskStateChanged(Task*)));

    AppContext::getTaskScheduler()->registerTopLevelTask(ttask);
    updateState();
}

void TestViewController::sl_taskStateChanged(Task* t) {

    if(t != task)return;
    if (!t->isFinished())return;
    task = NULL;
    AppContext::getTaskScheduler()->disconnect(this);
    QTime* temptime=new QTime;
    *temptime=QTime::currentTime();
    endRunTime=temptime;
    int ttime=endRunTime->second() + (60*(endRunTime->minute ()+(60*(endRunTime->hour()))));
    time=ttime-(startRunTime->second() + (60*(startRunTime->minute ()+(60*(startRunTime->hour())))));
    delete endRunTime;delete startRunTime;
    endRunTime=NULL;startRunTime=NULL;
    updateState();
}

void TestViewController::sl_setEnvAction() {
    GTestEnvironment* env = service->getEnv();
    const QMap<QString, QString>& vars = env->getVars();
    if (vars.isEmpty()) {
        QMessageBox::information(this, tr("info"), tr("no_vars_to_set"));
        return;
    }

    //todo: create custom utility class for properties like this
    QDialog d(this);
    d.setMinimumWidth(400);
    d.setWindowTitle(tr("env_mb_title"));
    QVBoxLayout* vl = new QVBoxLayout();
    d.setLayout(vl);

    QFormLayout* fl = new QFormLayout();
    vl->addLayout(fl);

    QMap<QString, QLineEdit*> valsByName;
    foreach(const QString& name, vars.keys()) {
        QString val = vars.value(name);
        QLineEdit* le = new QLineEdit(val, &d);
        valsByName[name] = le;
        fl->addRow(name + ":", le);
    }
    vl->addItem(new QSpacerItem(10, 10, QSizePolicy::Minimum, QSizePolicy::MinimumExpanding));
    QHBoxLayout* hl = new QHBoxLayout();
    vl->addLayout(hl);
    QDialogButtonBox* dbb = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);
    d.connect(dbb, SIGNAL(accepted()), SLOT(accept()));        
    d.connect(dbb, SIGNAL(rejected()), SLOT(reject()));
    vl->addWidget(dbb);
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }

    //save info from dialog to model
    foreach(const QString& name, vars.keys()) {
        QLineEdit* le = valsByName.value(name);
        assert(le);
        QString val = le->text();
        env->setVar(name, val);
    }

}

void TestViewController::sl_report(){
//create report form

    if (reporterForm!=NULL){
        AppContext::getMainWindow()->getMDIManager()->closeMDIWindow((MWMDIWindow*) reporterForm);
        assert(reporterForm == NULL);
    }
    reporterForm= new TestViewReporter(this,tree,time);
    AppContext::getMainWindow()->getMDIManager()->addMDIWindow(reporterForm); 
    AppContext::getMainWindow()->getMDIManager()->activateWindow(reporterForm);
}

void TestViewController::sl_treeDoubleClicked(QTreeWidgetItem* i, int col) {
    Q_UNUSED(col); Q_UNUSED(i);
    /*    TVItem* item = (TVItem*)i;
    TVTSItem* sItem = item!=NULL && item->isSuite() ? (TVTSItem*)item : NULL;
    TVTestItem* tItem = item!=NULL && item->isTest() ? (TVTestItem*)item : NULL;
    //TODO:*/
}

void TestViewController::sl_testStateChanged(GTestState* ts) {
    TVTestItem* tItem = findTestViewItem(ts->getTestRef());
    assert(tItem);
    tItem->updateVisual();
    TVTSItem* temp_parent=(TVTSItem*)tItem->parent();
    temp_parent->updateVisual();
}

//////////////////////////////////////////////////////////////////////////
// tree items;

TVTSItem::TVTSItem(GTestSuite* _ts) : TVItem(TVItem_TestSuite), ts(_ts){
    name = "/";
    updateVisual();
}

TVTSItem::TVTSItem(const QString& _name): TVItem(TVItem_TestSuite), name(_name) {
    ts = NULL;
    updateVisual();
}


void TVTSItem::updateVisual() {

    QString text0;
    if(!ts){
        text0 = name;
    }
    else{ text0 = ts->getName();
        setToolTip(0, ts->getURL());
    }
    setText(0, text0);
    
int total = 0;
    int passed = 0;
    int failed = 0;

    int tpassed=0;
    int tfailed=0;
    int tnone=0;
    int ttotal = 0;

    for(int i = 0; i < childCount(); i++) {        
        TVItem* item = (TVItem*)child(i);
        if(item->isTest()){
            total++;
            TVTestItem* tItem  = (TVTestItem*)item;
            GTestState* testState = tItem->testState;
            if (testState->isFailed()) {
            failed++;
            } else if (testState->isPassed()) {
             passed++;
            } 
        }
        else{
            assert(item->isSuite());
            TVTSItem* tItem  = (TVTSItem*)item;
            tItem->getTestsState(&tpassed,&tfailed,&tnone);
        }
    }

    tpassed=tpassed+passed;
    tfailed=tfailed+failed;
    tnone=tnone+(total - passed - failed);
    ttotal=tpassed + tfailed + tnone;

    QString text1;
    /*text1+="T:"+QString::number(total)+" [";
    text1+=" P:"+QString::number(passed);
    text1+=" F:"+QString::number(failed);
    text1+=" N:"+QString::number(total - passed - failed);
    text1+=" ]";*/

    text1+="T:"+QString::number(ttotal)+" [";
    text1+=" P:"+QString::number(tpassed);
    text1+=" F:"+QString::number(tfailed);
    text1+=" N:"+QString::number(tnone);
    text1+=" ]";    
    setText(1, text1);  
    //add icon
    if(tfailed){
    this->setIcon (0, ICON_FAILD_DIR);
    }
    else{
        if(tnone){
            this->setIcon (0, ICON_NOTRUN_DIR);
        }
        else{
            this->setIcon (0, ICON_SUCCES_DIR);
        }
    }
    if(this->parent() != NULL){
        ((TVTSItem*)parent())->updateVisual();
    }
}

QString TVTSItem::getRichDesc() const {
    //todo:
    if(ts!=NULL) return ts->getName();       
    else return getUrl();
}

void TVTSItem::getTestsState(int * rPassed,int * rFailed,int * rNone){    
    int total = 0;
    int passed = 0;
    int failed = 0;
    for(int i = 0; i < childCount(); i++) {
        TVItem* item = (TVItem*)child(i);
        if(item->isTest()){
            total++;
            TVTestItem* tItem  = (TVTestItem*)item;
            GTestState* testState = tItem->testState;
            if (testState->isFailed()) {
                failed++;
            } else if (testState->isPassed()) {
                passed++;
            } 
        }

        else{
            assert(item->isSuite());
            TVTSItem* tItem  = (TVTSItem*)item;
            tItem->getTestsState(rPassed,rFailed,rNone);      
        }
    } 
    *rPassed=*rPassed+passed;
    *rFailed=*rFailed+failed;
    *rNone=((*rNone+total)-passed)-failed;

    return ;
}

QString TVTSItem::getUrl() const{
    if(ts!=NULL)return name;    
    else return (((TVTSItem*)parent())->getUrl()+name+"/");
        
} 
//---------------------------------------------------------------------------
TVTestItem::TVTestItem(GTestState* _t): TVItem(TVItem_Test), testState(_t) {
    updateVisual();
}

static QString getStateName(const GTestState* testState) {
    if (testState->isPassed()) {
        return TestViewController::tr("passed");
    }
    if (testState->isFailed()) {
        return TestViewController::tr("failed");
    }
    return TestViewController::tr("not_run");
}

void TVTestItem::updateVisual() {
    QString name=testState->getTestRef()->getShortName().section('/', -1 );
    setText(0, name);
    setToolTip(0, testState->getTestRef()->getURL());
    
    setText(1, getStateName(testState));
    //add icon
    this->setForeground(1,Qt::black);
    this->setIcon (0, ICON_NOTRUN_TEST );
    if (testState->isFailed()){
        this->setForeground(1,Qt::red);
        this->setIcon (0, ICON_FAILD_TEST );
    }
    if(testState->isPassed()){
        this->setForeground(1,Qt::darkGreen);
        this->setIcon (0, ICON_SUCCES_TEST );
    }
}

QString TVTestItem::getRichDesc() const {
    QString text = "<b>"+testState->getTestRef()->getShortName()+"</b><br>";
    text+="<b>"+TestViewController::tr("state: ")+"</b>" + getStateName(testState)+"<br>";
    if (testState->isFailed()) {
        text+="<b>"+TestViewController::tr("fail_desc: ") +"</b>"+testState->getErrorMessage()+"<br>";
    }

    text+="<b>"+TestViewController::tr("source_file:")+"</b>"+testState->getTestRef()->getURL()+"<br>";
    text+="<hr>";
    QFile myFile(testState->getTestRef()->getURL()); 
    QTextStream t( &myFile );
    if (myFile.open(QIODevice::ReadOnly) ) {       
        QTextStream t( &myFile );
        while (!t.atEnd() ) {           
            QString tempString= t.readLine();
            tempString.replace("<","&lt;");
            tempString.replace(">","&gt;");
            text+=tempString+"<br>";
        }
        // Close the file
        myFile.close();
    }
    return text;
}



}//namespace
