/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ItemViewStyle.h"
#include "WorkflowViewItems.h"
#include <workflow/WorkflowModel.h>

#include <core_api/Log.h>
#include <QBitmap>
#include <QtGui/QPainter>
#include <QtGui/QGraphicsTextItem>
#include <QtGui/QGraphicsSimpleTextItem>
#include <QtGui/QGraphicsSceneMouseEvent>
#include <QtGui/QStyleOptionGraphicsItem>
#include <QtGui/QGraphicsView>
#include <QtGui/QRadialGradient>
#include <QtGui/QTextDocument>
#include <qmath.h>


namespace GB2 {

const StyleId ItemStyles::SIMPLE = "simple";
const StyleId ItemStyles::EXTENDED = "ext";

SimpleProcStyle::SimpleProcStyle(WorkflowProcessItem* pit) {
    owner = (pit);
    owner->connect(owner->getProcess(), SIGNAL(si_labelChanged()), SLOT(sl_update()));
}

QRectF SimpleProcStyle::boundingRect(void) const {
    // extra space added for clean antialiased painting
    return QRectF(-R-2, -R-2, R*2+4, R*2+4);
}

QPainterPath SimpleProcStyle::shape () const {
    QPainterPath contour;
    contour.addEllipse(QPointF(0,0), R, R);
    return contour;
}

void SimpleProcStyle::paint(QPainter *painter,
                                const QStyleOptionGraphicsItem *option,
                                QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);
    //painter->fillRect(boundingRect(), QBrush(Qt::magenta, Qt::Dense6Pattern));
    painter->setRenderHint(QPainter::Antialiasing);
    QPainterPath contour;
    contour.addEllipse(QPointF(0,0), R, R);
    if (owner->isSelected()) {
        QPen pen;
        pen.setWidthF(2);
        pen.setStyle(Qt::DashLine);
        painter->setPen(pen);
    }

    QColor bgc(Qt::darkCyan);
    bgc.setAlpha(200);
    QRadialGradient rg(R/2, -R/2, R*2);
    rg.setColorAt(1, bgc);
    rg.setColorAt(0, QColor(Qt::white));
    QBrush procBrush(rg);
    painter->drawPath(contour);
    painter->fillPath(contour, procBrush);
    
    painter->save();
    QTextDocument d;
    d.setHtml("<center>" + Qt::escape(owner->getProcess()->getLabel()) + "</center>");
    d.setTextWidth(R*2);
    //d.setDefaultTextOption(QTextOption(Qt::AlignHCenter));
    painter->translate(-d.size().width()/2, -d.size().height()/2);
    //painter->translate(-R, -R);
    d.drawContents(painter/*, boundingRect()*/);
    painter->restore();
}

//QPainterPath shape () const;

#define MARGIN 5

ExtendedProcStyle::ExtendedProcStyle(WorkflowProcessItem* pit){
    owner = (pit);
    Actor* process = pit->getProcess();

    doc = process->getDescription();
    if (doc) {
        owner->connect(doc, SIGNAL(contentsChanged()), SLOT(sl_update()));
    } else {
        doc = new QTextDocument(pit);
        doc->setHtml(QString("<center><b>%1</b></center><hr>%2<br>aLSKDJALSK LASDJ LASKD LASJD ALSKDJ XCKLJSLC Jas dlkjsdf sdlkjsdlfj sdlkfjlsdkfjs dlkfjsdlkfjsld flsdkjflsd kfjlsdkfj lsdkfjlsd flskfjsldkfjsldf jsdlkfjsdlkfjsdlfkjsdlfj")
            .arg(process->getLabel()).arg(process->getProto()->getDocumentation()));
    }
    //doc->setDefaultTextOption(QTextOption(Qt::AlignJustify));
    refresh();
}

#define MINW 2*R
#define MAXW 6*R

void ExtendedProcStyle::refresh() {
    qreal w,h;
    int cycle = 0;
    do {
        QSizeF docFrame = doc->size();
        w = docFrame.width() + MARGIN*2;
        h = qMax(2*R, docFrame.height()) + MARGIN*2;
        //printf("ideal=%f, actual=%f\n",doc->idealWidth(),w);
        
        if (++cycle > 2) {
            break;
        }
        if ((h/w < 0.6 && w > (MINW+MAXW)/2)  //doc is disproportionately wide
            || (h/w > 1.6 && w < MAXW)        //doc is disproportionately long and can be widen
            || (w < MINW || w > MAXW)) {      //width is out of bounds
            doc->setTextWidth(qBound(MINW, h/1.6, MAXW-MARGIN*2));            
        }
    } while (true);

    bounds = QRectF(-R-MARGIN, -R-MARGIN,w,h);
}

QPainterPath ExtendedProcStyle::shape () const {
    QPainterPath contour;
    contour.addRoundedRect(bounds, MARGIN, MARGIN);
    return contour;
}


void ExtendedProcStyle::paint(QPainter *painter,
                            const QStyleOptionGraphicsItem *option,
                            QWidget *widget)
{
    Q_UNUSED(widget);
    if (owner->isSelected()) {
        ((QStyleOptionGraphicsItem*)option)->state |= QStyle::State_Selected;
    }

    QColor bgc(0x20,0xf0, 0x50, 64);
    QRectF tb = boundingRect();
    painter->fillRect(tb, QBrush(bgc));

    painter->save();
    painter->translate(-R, -R);
    doc->drawContents(painter);
    painter->restore();

    painter->setRenderHint(QPainter::Antialiasing);
    QPen pen;
    pen.setWidthF(1.5);
    if (owner->isSelected()) {
        pen.setStyle(Qt::DashLine);
    }
    painter->setPen(pen);
    painter->drawRoundedRect(tb, MARGIN, MARGIN);
}
}//namespace
