/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowIOTasks.h"
#include "SceneSerializer.h"
#include "WorkflowViewController.h"

#include <core_api/Log.h>
#include <util_gui/GUIUtils.h>

#include <QtCore/QFile>
#include <QtXml/qdom.h>

/* TRANSLATOR GB2::LocalWorkflow::WorkflowView */

namespace GB2 {
using namespace Workflow;

static LogCategory log(ULOG_CAT_WD);

SaveWorkflowSchemaTask::SaveWorkflowSchemaTask(WorkflowScene* scene, const Metadata& meta):
Task(WorkflowView::tr("Saving schema to file"), TaskFlag_DeleteWhenFinished),
    meta(meta), scene(scene){}


void SaveWorkflowSchemaTask::run() {
    const QString& url = meta.url;
    log.details(tr("Saving schema to file: %1").arg(url));

    QDomDocument xmlDoc(SchemaSerializer::WORKFLOW_DOC);
    SceneSerializer::scene2xml(scene, xmlDoc);
    QDomElement elem = xmlDoc.documentElement();
    SchemaSerializer::saveIterations(scene->getIterations(), elem);
    SchemaSerializer::saveMeta(&meta, elem);
    QByteArray rawData = xmlDoc.toByteArray();

    QFile f(url);
    if (f.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
        qint64 s = f.write(rawData);
        f.close();
        if (s != rawData.size()) {
            stateInfo.error = Translations::errorWritingFile(url);
        } else {
            scene->setModified(false);
        }
    } else {
        stateInfo.error = Translations::errorOpeningFileWrite(url);
        log.error(stateInfo.error);
    }
}

LoadWorkflowSchemaTask::LoadWorkflowSchemaTask(WorkflowScene* scene, Workflow::Metadata* meta, 
    const QString& url):
Task(WorkflowView::tr("Loading schema"), TaskFlag_DeleteWhenFinished),
    url(url), scene(scene), meta(meta) {
        xml = new QDomDocument();
}

void LoadWorkflowSchemaTask::run() {
    log.details(tr("Loading schema from file: %1").arg(url));

    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        stateInfo.error = Translations::errorOpeningFileRead(url);
        return;
    }
    QByteArray  xmlData = f.readAll();
    f.close();

    bool res = xml->setContent(xmlData);
    if (!res || xml->doctype().name() != SchemaSerializer::WORKFLOW_DOC) {
        stateInfo.error = WorkflowView::tr("Invalid content: %1").arg(url);
        xml->clear();
    }
}


Task::ReportResult LoadWorkflowSchemaTask::report() {
    if (!stateInfo.hasErrors()) {
        scene->sl_reset();
        QMap<ActorId, ActorId> remapping;
        stateInfo.error = SceneSerializer::xml2scene(xml->documentElement(), scene, remapping);
        if (stateInfo.hasErrors()) {
            scene->sl_reset();
            meta->reset();
        } else {
            SchemaSerializer::readMeta(meta, xml->documentElement());
            QList<Iteration> lst;
            SchemaSerializer::readIterations(lst, xml->documentElement(), remapping);
            scene->setIterations(lst);
            scene->setModified(false);
            meta->url = url;
        }
    }
    delete xml;

    return Task::ReportResult_Finished;
}


}//namespace
