/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowPalette.h"

#include <workflow/WorkflowRegistry.h>

#include <core_api/Log.h>
#include <core_api/Settings.h>

#include <QtGui/QMenu>
#include <QtGui/QToolBox>
#include <QtGui/QButtonGroup>
#include <QtGui/QToolButton>

#include <QtGui/QHeaderView>
#include <QtGui/QApplication>
#include <QtCore/QAbstractItemModel>
#include <QtGui/QTreeView>
#include <QtGui/QStyle>
#include <QtGui/QPainter>
#include <QtGui/QItemDelegate>
#include <QtGui/QTreeView>
#include <QtGui/QContextMenuEvent>


namespace GB2 {

const QString WorkflowPalette::MIME_TYPE("application/x-ugene-workflow-id");

class PaletteDelegate: public QItemDelegate
{
public:
    PaletteDelegate(WorkflowPalette *view) : QItemDelegate(view), m_view(view){}

    virtual void paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const;
    virtual QSize sizeHint(const QStyleOptionViewItem &opt, const QModelIndex &index) const;

private:
    WorkflowPalette *m_view;
};

QTreeWidgetItem* WorkflowPalette::createItemWidget(QAction *a) {
    QTreeWidgetItem* item = new QTreeWidgetItem();
    item->setToolTip(0, a->toolTip());
    item->setData(0, Qt::UserRole, qVariantFromValue<QAction*>(a));
    actionMap[a] = item;
    connect(a, SIGNAL(triggered()), SLOT(handleItemAction()));
    connect(a, SIGNAL(toggled(bool)), SLOT(handleItemAction()));

    return item;
}

void WorkflowPalette::handleItemAction() {
    QAction* a = qobject_cast<QAction*>(sender());
    assert(a);
    assert(actionMap[a]);
    if (a) {
        update(indexFromItem(actionMap[a]));
    }
}

QAction* WorkflowPalette::createItemAction(ActorPrototype* item) {
    QAction* a = new QAction(item->getDisplayName(), this);
    a->setToolTip(item->getDocumentation());
    a->setCheckable(true);
    if (item->icon.isNull()) {
        item->icon = QIcon(":workflow_designer/images/green_circle.png");
    }
    a->setIcon(item->icon);
    a->setData(qVariantFromValue(item));
    connect(a, SIGNAL(triggered(bool)), SLOT(sl_selectProcess(bool)));
    connect(a, SIGNAL(toggled(bool)), SLOT(sl_selectProcess(bool)));
    return a;
}

void WorkflowPalette::sl_selectProcess(bool checked) {
    if (currentAction && currentAction != sender()) {
        currentAction->setChecked(false);
    }
    if (!checked) {
        currentAction = NULL;
    } else {
        currentAction = qobject_cast<QAction*>(sender());
        assert(currentAction);
    }
    emit processSelected(currentAction ? 
        qVariantValue<Workflow::ActorPrototype*>(currentAction->data()) : NULL );
}

void WorkflowPalette::resetSelection() {
    if (currentAction) {
        currentAction->setChecked(false);
        currentAction = NULL;
    }
}

void WorkflowPalette::rebuild() {
    setMouseTracking(false);
    resetSelection();
    WProtoRegistry* reg = qobject_cast<WProtoRegistry*>(sender());
    if (reg) {
        clear();
        setContent(reg);
    }
    setMouseTracking(true);
}

WorkflowPalette::WorkflowPalette(WProtoRegistry* reg, QWidget *parent) 
: QTreeWidget(parent), overItem(NULL), currentAction(NULL)
{
    setFocusPolicy(Qt::NoFocus);
    setSelectionMode(QAbstractItemView::NoSelection);
    setItemDelegate(new PaletteDelegate(this));
    setRootIsDecorated(false);
    //setAnimated(true);
    setMouseTracking(true);
    setColumnCount(1);
    header()->hide();
    header()->setResizeMode(QHeaderView::Stretch);
    //setTextElideMode (Qt::ElideMiddle);
    setContent(reg);
    connect(reg, SIGNAL(si_registryModified()), SLOT(rebuild()));
}

void WorkflowPalette::setContent(WProtoRegistry* reg) {
    QMapIterator<Descriptor, QList<ActorPrototype*> > it(reg->getProtos());
    while (it.hasNext()) {
        it.next();
        QTreeWidgetItem* category = new QTreeWidgetItem(this);
        category->setText(0, it.key().getDisplayName());
        category->setData(0, Qt::UserRole, it.key().getId());
        addTopLevelItem(category);
        foreach(ActorPrototype* proto, it.value()) {
            QAction* action = createItemAction(proto);
            categoryMap[it.key().getDisplayName()] << action;
            category->addChild(createItemWidget(action));
        }
    }
}

QMenu* WorkflowPalette::createMenu(const QString& name) {
    QMenu* itemsMenu = new QMenu(name, this);
    QMapIterator<QString, QList<QAction*> > it(categoryMap);
    while (it.hasNext()) {
        it.next();
        QMenu* grpMenu = new QMenu(it.key(), itemsMenu);
        itemsMenu->addMenu(grpMenu);
        foreach(QAction* a, it.value()) {
            grpMenu->addAction(a);
        }
    }

    return itemsMenu;
}

void WorkflowPalette::mouseMoveEvent ( QMouseEvent * event ) {
    if ((event->buttons() & Qt::LeftButton) && !dragStartPosition.isNull()) {
        if ((event->pos() - dragStartPosition).manhattanLength() <= QApplication::startDragDistance()) return;
        QTreeWidgetItem* item = itemAt(event->pos());
        if (!item) return;
        QAction* action = qVariantValue<QAction*>(item->data(0, Qt::UserRole));
        if (!action) return;
        ActorPrototype* proto = action->data().value<ActorPrototype*>();
        assert(proto);
        QMimeData* mime = new QMimeData();
        mime->setData(MIME_TYPE, proto->getId().toAscii());
        mime->setText(proto->getId());
        QDrag *drag = new QDrag(this);
        drag->setMimeData(mime);
        drag->setPixmap(action->icon().pixmap(QSize(44, 44)));

        resetSelection();
        dragStartPosition = QPoint();
        Qt::DropAction dropAction = drag->exec(Qt::CopyAction, Qt::CopyAction);
        Q_UNUSED(dropAction);
        return;
    }
    QTreeWidgetItem* prev = overItem;
    overItem = itemAt(event->pos());
    if (prev) update(indexFromItem(prev));
    if (overItem) update(indexFromItem(overItem));
}

void WorkflowPalette::leaveEvent ( QEvent * event ) {
    Q_UNUSED(event);
    QTreeWidgetItem* prev = overItem;
    overItem = NULL;
    if (prev) update(indexFromItem(prev));
}

void WorkflowPalette::mousePressEvent ( QMouseEvent * event ) {
    dragStartPosition = QPoint();
    if ((event->buttons() & Qt::LeftButton)) {
        QTreeWidgetItem* item = itemAt(event->pos());
        if (!item) return;
        event->accept();
        if (item->parent() == 0) {
            setItemExpanded(item, !isItemExpanded(item));
            return;
        }

        QAction* action = qVariantValue<QAction*>(item->data(0, Qt::UserRole));
        if (action) {
            action->toggle();
            dragStartPosition = event->pos();
        }
    }
}

void WorkflowPalette::contextMenuEvent(QContextMenuEvent *e)
{
    QMenu menu;
    menu.addAction(tr("Expand all"), this, SLOT(expandAll()));
    menu.addAction(tr("Collapse all"), this, SLOT(collapseAll()));
    e->accept();
    menu.exec(mapToGlobal(e->pos()));
}


void PaletteDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    const QAbstractItemModel *model = index.model();
    Q_ASSERT(model);

    if (!model->parent(index).isValid()) {
        // this is a top-level item.
        QStyleOptionButton buttonOption;

        buttonOption.state = option.state;
#ifdef Q_WS_MAC
        buttonOption.state |= QStyle::State_Raised;
#endif
        buttonOption.state &= ~QStyle::State_HasFocus;

        buttonOption.rect = option.rect;
        buttonOption.palette = option.palette;
        buttonOption.features = QStyleOptionButton::None;
        m_view->style()->drawControl(QStyle::CE_PushButton, &buttonOption, painter, m_view);

        QStyleOption branchOption;
        static const int i = 9; // ### hardcoded in qcommonstyle.cpp
        QRect r = option.rect;
        branchOption.rect = QRect(r.left() + i/2, r.top() + (r.height() - i)/2, i, i);
        branchOption.palette = option.palette;
        branchOption.state = QStyle::State_Children;

        if (m_view->isExpanded(index))
            branchOption.state |= QStyle::State_Open;

        m_view->style()->drawPrimitive(QStyle::PE_IndicatorBranch, &branchOption, painter, m_view);

        // draw text
        QRect textrect = QRect(r.left() + i*2, r.top(), r.width() - ((5*i)/2), r.height());
        QString text = elidedText(option.fontMetrics, textrect.width(), Qt::ElideMiddle, 
            model->data(index, Qt::DisplayRole).toString());
        m_view->style()->drawItemText(painter, textrect, Qt::AlignCenter,
            option.palette, m_view->isEnabled(), text);

    } else {
        QStyleOptionToolButton buttonOption;
        buttonOption.state = option.state;
        buttonOption.state &= ~QStyle::State_HasFocus;
        buttonOption.direction = option.direction;
        buttonOption.rect = option.rect;
        buttonOption.font = option.font;
        buttonOption.fontMetrics = option.fontMetrics;
        buttonOption.palette = option.palette;
        buttonOption.subControls = QStyle::SC_ToolButton;
        buttonOption.features = QStyleOptionToolButton::None;

        QAction* action = qVariantValue<QAction*>(index.data(Qt::UserRole));
        buttonOption.text = action->text();
        buttonOption.icon = action->icon();
        if (!buttonOption.icon.isNull()) {
            buttonOption.iconSize = QSize(22, 22);
        }
        if (action->isChecked()) {
            buttonOption.state |= QStyle::State_On;
            buttonOption.state |= QStyle::State_Sunken;
            buttonOption.activeSubControls = QStyle::SC_ToolButton;
        } else {
            buttonOption.state |= QStyle::State_Raised;
            buttonOption.activeSubControls = QStyle::SC_None;
        }

        if (m_view->overItem == m_view->itemFromIndex(index)) {
            buttonOption.state |= QStyle::State_MouseOver;
        }

        buttonOption.state |= QStyle::State_AutoRaise;

        buttonOption.toolButtonStyle = Qt::ToolButtonTextBesideIcon;
        m_view->style()->drawComplexControl(QStyle::CC_ToolButton, &buttonOption, painter, m_view);

        //QItemDelegate::paint(painter, option, index);
    }
}

QSize PaletteDelegate::sizeHint(const QStyleOptionViewItem &opt, const QModelIndex &index) const
{
    const QAbstractItemModel *model = index.model();
    Q_ASSERT(model);

    QStyleOptionViewItem option = opt;
    bool top = !model->parent(index).isValid();
    QSize sz = QItemDelegate::sizeHint(opt, index) + QSize(top?2:20, top?2:20);
    return sz;
}

QVariant WorkflowPalette::saveState() const {
    QVariantMap m;
    for (int i = 0, count = topLevelItemCount(); i < count; ++i) {
        QTreeWidgetItem* it = topLevelItem(i);
        m.insert(it->data(0, Qt::UserRole).toString(), it->isExpanded());
    }
    return m;
}

void WorkflowPalette::restoreState(const QVariant& v) {
    QMapIterator<QString,QVariant> it(v.toMap());
    while (it.hasNext())
    {
        it.next();
        for (int i = 0; i < topLevelItemCount(); i++) {
            if (topLevelItem(i)->data(0, Qt::UserRole) == it.key()) {
                topLevelItem(i)->setExpanded(it.value().toBool());
                break;
            }
        }
    }
}


}//namespace
