/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GenericReadActor.h"
#include "CoreLib.h"
#include "DocActors.h"

#include <workflow/TypeSet.h>
#include <workflow/WorkflowEnv.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>

#include <core_api/DocumentModel.h>
#include <gobjects/GObjectTypes.h>
#include <util_gui/DialogUtils.h>

#include <QtCore/QFileInfo>


/* TRANSLATOR GB2::Workflow::SeqReadPrompter */
/* TRANSLATOR GB2::Workflow::CoreLib */

namespace GB2 {
namespace Workflow {

const QString GenericSeqActorProto::MODE_ATTR("mode");
const QString GenericSeqActorProto::GAP_ATTR("gap");

const QString GenericSeqActorProto::TYPE("generic.seq");
const QString GenericMAActorProto::TYPE("generic.ma");

GenericSeqActorProto::GenericSeqActorProto() : BusActorPrototype(CoreLib::GENERIC_SEQ_ACTOR)
{    
    setDisplayName(GB2::Workflow::CoreLib::tr("Generic sequence reader"));
    desc = GB2::Workflow::CoreLib::tr("Reads sequences from files of any format supported by UGENE. The files can be local or Internet URLs.");
    QMap<Descriptor, DataTypePtr> m;
    m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
    m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
    DataTypePtr seqTypeset(new DataTypeSet(Descriptor(TYPE), m));
    bool treg = WorkflowEnv::getDataTypeRegistry()->registerEntry(seqTypeset);
    Q_UNUSED(treg);assert(treg);

    ports << new PortDescriptor(
        Descriptor(CoreLib::DATA_PORT_ID, 
        GB2::Workflow::CoreLib::tr("Sequence"), 
        GB2::Workflow::CoreLib::tr("A sequence of any type (nucleotide, protein).")), 
        seqTypeset, false, true);

    {
        Descriptor md(GenericSeqActorProto::MODE_ATTR, SeqReadPrompter::tr("Mode"), 
            SeqReadPrompter::tr("If the file contains more than one sequence, \"split\" mode sends them as is to output," 
            "while \"merge\" appends all the sequences and outputs the sole merged sequence."));
        Descriptor gd(GenericSeqActorProto::GAP_ATTR, SeqReadPrompter::tr("Merging gap"), 
            SeqReadPrompter::tr("In the \"merge\" mode, insert the specified number of gaps between original sequences."
            "This is helpful e.g. to avoid finding false positives at the merge boundaries."));

        attrs << new Attribute(CoreLib::URL_ATTR(), CoreDataTypes::STRING_TYPE(), true);
        attrs << new Attribute(md, CoreDataTypes::NUM_TYPE(), true, SPLIT);
        attrs << new Attribute(gd, CoreDataTypes::NUM_TYPE(), false, 0);
    }

    QMap<QString, PropertyDelegate*> delegates;
    QVariantMap modeMap; 
    modeMap[SeqReadPrompter::tr("split")] = SPLIT;
    modeMap[SeqReadPrompter::tr("merge")] = MERGE;
    delegates[MODE_ATTR] = new ComboBoxDelegate(modeMap);
    delegates[CoreLib::URL_ATTR_ID] = new URLDelegate(DialogUtils::prepareDocumentsFileFilter(true), QString(), true);
    setEditor(new DelegateEditor(delegates));
    setPrompter(new ReadDocPrompter(GB2::Workflow::CoreLib::tr("Read all sequences from <u>%1</u> and output each in turn.")));
}

GenericMAActorProto::GenericMAActorProto() : BusActorPrototype(CoreLib::GENERIC_MA_ACTOR) 
{
    setDisplayName(GB2::Workflow::CoreLib::tr("Generic alignment reader"));
    desc = GB2::Workflow::CoreLib::tr("Reads alignments from files of any format supported by UGENE. The files can be local or Internet URLs."
        "<p>Besides the known alignment formats, supports composing an alignment from a set of sequences in a corresponing file (e.g. FASTA or Genbank).");

    QMap<Descriptor, DataTypePtr> m;
    m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
    m[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    DataTypePtr blockTypeset(new DataTypeSet(Descriptor(TYPE), m));
    bool treg = WorkflowEnv::getDataTypeRegistry()->registerEntry(blockTypeset);
    Q_UNUSED(treg);assert(treg);

    ports << new PortDescriptor(
        Descriptor(CoreLib::DATA_PORT_ID, 
        GB2::Workflow::CoreLib::tr("Multiple sequence alignment"), 
        ""), 
        blockTypeset, false, true);

    attrs << new Attribute(CoreLib::URL_ATTR(), CoreDataTypes::STRING_TYPE(), true);
    setEditor(new DelegateEditor(CoreLib::URL_ATTR_ID, new URLDelegate(DialogUtils::prepareDocumentsFileFilter(true), QString(),true)));
    setPrompter(new ReadDocPrompter(GB2::Workflow::CoreLib::tr("Read MSA (multiple sequence alignment) blocks from <u>%1</u>.")));
}

bool GenericMAActorProto::isAcceptableDrop(const QMimeData* md, QVariantMap* params) const {
    QList<DocumentFormat*> fs;
    QString url = DesignerUtils::getDropUrl(fs, md);
    DocumentFormatConstraints mc;
    mc.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT);
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(mc)) {
            if (params) {
                params->insert(CoreLib::URL_ATTR_ID, url);
            }
            return true;
        }
    }
    DocumentFormatConstraints sc;
    sc.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(sc)) {
            if (params) {
                params->insert(CoreLib::URL_ATTR_ID, url);
            }
            return true;
        }
    }
    if (QFileInfo(url).isDir()) {
        if (params) {
            params->insert(CoreLib::URL_ATTR_ID, url + "/*");
        }
        return true;
    }

    return false;
}

bool GenericSeqActorProto::isAcceptableDrop(const QMimeData* md, QVariantMap* params) const {
    QList<DocumentFormat*> fs;
    QString url = DesignerUtils::getDropUrl(fs, md);
    DocumentFormatConstraints constraints;
    constraints.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
    DocumentFormat* format = NULL;
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(constraints)) {
            if (params) {
                params->insert(CoreLib::URL_ATTR_ID, url);
            }
            return true;
        }
    }
    if (QFileInfo(url).isDir()) {
        if (params) {
            params->insert(CoreLib::URL_ATTR_ID, url + "/*");
        }
        return true;
    }

    return false;
}

QString SeqReadPrompter::composeRichDoc() {
    //TODO finish
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLib::DATA_PORT_ID));
    QString url = getURL(CoreLib::URL_ATTR_ID);
    return QString("%1").arg(url);
}

}//namespace Workflow
}//namespace GB2
