/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconSearchTask.h"

namespace GB2 {

SiteconSearchTask::SiteconSearchTask(const SiteconModel& m, const char* s, int l, DNATranslation* ctt, bool co, int ms) 
: Task(tr("sitecon_search"), TaskFlags_NR_DWF | TaskFlags_FAIL_OSCOE), model(m), minScore(ms), complOnly(co)
{
    model.checkState(true);
    model.matrix = SiteconAlgorithm::normalize(model.matrix, model.settings);
    SequenceWalkerConfig c;
    c.seq = s;
    c.seqSize = l;
    c.complTrans  = ctt;
    c.aminoTrans = NULL;

    c.chunkSize = l;
    c.overlapSize = 0;
    c.exOverlapSize = 0;
    c.parallel = true;
    c.maxThreads = 2;

    SequenceWalkerTask* t = new SequenceWalkerTask(c, this, tr("sitecon_search_parallel"));
    addSubTask(t);
}

void SiteconSearchTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti) {
//TODO: process border case as if there are 'N' chars before 0 and after seqlen
    if (complOnly && !t->isDNAComplemented()) {
        return;
    }
    int seqLen = t->getRegionSequenceLen();
    const char* seq = t->getGlobalConfig().seq + t->getGlobalRegion().startPos;;
    int modelSize = model.settings.windowSize;
    ti.progress =0;
    int lenPerPercent = seqLen / 100;
    int pLeft = lenPerPercent;
    DNATranslation* complTT = t->isDNAComplemented() ? t->getGlobalConfig().complTrans : NULL;
    for (int i = 0, n = seqLen - modelSize; i < n && !ti.cancelFlag; i++, --pLeft) {
        float psum = SiteconAlgorithm::calculatePSum(seq+i, modelSize, model.matrix, model.settings, model.deviationThresh, complTT);
        if (psum < 0 || psum >=1) {
            ti.error = tr("internal_error_invalid_psum:%1").arg(psum);
            return;
        }
        float perc = 100*psum;
        if (perc >= minScore) {//report result
            SiteconSearchResult r;
            r.modelInfo = model.modelFileName;
            r.complement = t->isDNAComplemented();
            r.region.startPos = t->getGlobalRegion().startPos +  i;
            r.region.len = modelSize;
            r.psum = perc;
            r.err1 = model.err1[perc];
            r.err2 = model.err2[perc];
            addResult(r);
        }
        if (pLeft == 0) {
            ti.progress++;
            pLeft = lenPerPercent;
        }
    }
}


void SiteconSearchTask::addResult(const SiteconSearchResult& r) {
    lock.lock();
    results.append(r);
    lock.unlock();
}

QList<SiteconSearchResult> SiteconSearchTask::takeResults() {
    lock.lock();
    QList<SiteconSearchResult> res = results;
    results.clear();
    lock.unlock();
    return res;
}

}//namespace
