/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ShutdownTask.h"
#include "MainWindowImpl.h"

#include <core_api/Log.h>
#include <core_api/ServiceModel.h>
#include <core_api/PluginModel.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>

namespace GB2 {

static LogCategory log(ULOG_CAT_USER_INTERFACE);

ShutdownTask::ShutdownTask(MainWindowImpl* _mw) : Task(tr("shutdown_task_name"), TaskFlags_NR_DWF_SSSOR), mw(_mw) 
{
}

static bool isReadyToBeDisabled(Service* s, ServiceRegistry* sr) {
	ServiceType st = s->getType();
	int nServicesOfTheSameType = sr->findServices(st).size();
	assert(nServicesOfTheSameType >= 1);
	foreach(Service* child, sr->getServices()) {
		if (!child->getParentServiceTypes().contains(st) || !child->isEnabled()) {
			continue;
		}
		if (nServicesOfTheSameType == 1) {
			return false;
		}
	}
	return true;
}

static Service* findServiceToDisable(ServiceRegistry* sr) {
	int nEnabled = 0;
	foreach(Service* s, sr->getServices()) {
		nEnabled+= s->isEnabled() ? 1 : 0;
		if (s->isEnabled() && isReadyToBeDisabled(s, sr)) {
			return s;
		}
	}
	assert(nEnabled == 0);
	return NULL;
}

void ShutdownTask::prepare() {
    log.info(tr("Starting shutdown process..."));

	ServiceRegistry* sr = AppContext::getPluginSupport()->getServiceRegistry();
	Service* s = findServiceToDisable(sr);
	if (s!=NULL) {
		addSubTask(sr->disableServiceTask(s));
	}
}

QList<Task*> ShutdownTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;

    if (subTask->hasErrors()) {
        return res; //stop shutdown process
    }

	ServiceRegistry* sr = AppContext::getPluginSupport()->getServiceRegistry();
	Service* s = findServiceToDisable(sr);
	if (s!=NULL) {
		res.append(sr->disableServiceTask(s));
	}
	return res;
}

Task::ReportResult ShutdownTask::report() {
    if (stateInfo.cancelFlag) {
        log.info(tr("Shutdown was canceled"));
        return Task::ReportResult_Finished;
    } 
    if (propagateSubtaskError()) {
        log.error(tr("Shutdown failed, error: %1").arg(stateInfo.error));
        return Task::ReportResult_Finished;
    }

#ifdef _DEBUG
	const QList<Service*>& services = AppContext::getPluginSupport()->getServiceRegistry()->getServices();
	foreach(Service* s, services) {
		assert(s->isDisabled());
	}
#endif
    mw->close();
	return Task::ReportResult_Finished;
}



}//namespace
