/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QApplication>
#include <QtGui/QHeaderView>
#include <QtCore/QSet>
#include <QtGui/QFileDialog>
#include <QtCore/QDebug>
#include <QtGui/QMessageBox>

#include <core_api/AppContext.h>
#include <util_tasks/MultiTask.h>
#include <util_tasks/ReloadScriptTask.h>
#include "ScriptManagerWidgetImpl.h"

namespace GB2
{
ScriptManagerWidgetImpl::ScriptManagerWidgetImpl( QWidget * p, ScriptRegistryService * _sr ) :
QWidget(p), sr(_sr) {
    setupUi( this );
    updateState();

    scriptTable->verticalHeader()->hide();
    scriptTable->horizontalHeader()->setResizeMode( COL_URL, QHeaderView::Stretch );
    scriptTable->horizontalHeader()->setHighlightSections( false );
    scriptTable->horizontalHeader()->setClickable( false );
    scriptTable->setColumnWidth( COL_URL, scriptTable->columnWidth(COL_URL) * 3 );
    scriptTable->setEditTriggers( QAbstractItemView::NoEditTriggers );
    

    connect( addButton, SIGNAL(clicked()), SLOT(sl_Add()) );
    connect( delButton, SIGNAL(clicked()), SLOT(sl_Delete()) );
    connect( reloadButton, SIGNAL(clicked()), SLOT(sl_Reload()) );
    connect( AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_scriptReloaded(Task*)) );
}

void ScriptManagerWidgetImpl::updateItem( int row, Script * scr ) {
    QTableWidgetItem * twi_name = new QTableWidgetItem( scr->getName() );
    QTableWidgetItem * twi_type = new QTableWidgetItem( scr->getType() );
    QTableWidgetItem * twi_url = new QTableWidgetItem( scr->getUrl() );
    QTableWidgetItem * twi_state = new QTableWidgetItem( scr->getStateString() );
    scriptTable->setItem( row, COL_NAME, twi_name );
    scriptTable->setItem( row, COL_TYPE, twi_type );
    scriptTable->setItem( row, COL_URL, twi_url );
    scriptTable->setItem( row, COL_STATE, twi_state );

}

void ScriptManagerWidgetImpl::updateState(){
    scripts = sr->getAllScripts();
    scriptTable->setRowCount( scripts.size() );
    int row = 0;
    foreach( Script * scr, scripts ) {
        updateItem( row, scr );
        scriptTable->setRowHeight( row, QWidget::fontMetrics().height() );
        ++row;
    }
}

void ScriptManagerWidgetImpl::sl_Add() {
    QString dir = QApplication::applicationDirPath();
    QString filter = tr( "Qt scripts" ) + " ( *.qs)\n" + tr( "All files" ) + " ( *.*)\n";
    QStringList new_urls = QFileDialog::getOpenFileNames( this, tr("add_scripts_caption"), dir, filter );
    foreach( QString new_url, new_urls ) {
        if( sr->registerScript(new_url) == NULL ) {
            QString scr_name = QFileInfo( new_url ).fileName();
            QMessageBox::warning( this, tr("Warning"), tr("script_%1_already_registered").arg(scr_name) );
        }
    }
    updateState();
}

namespace {
QSet<int> selectedIndexes( const QList<QTableWidgetItem*> & items ) {
    QSet<int> indexes;
    foreach( QTableWidgetItem * item, items ) {
        indexes << item->row();
    }
    return indexes;
}
}//anonymous ns

void ScriptManagerWidgetImpl::sl_Delete() {
    QSet<int> indexes = selectedIndexes( scriptTable->selectedItems() );
    if( indexes.empty() ) {
        return;
    }
    int res = 0;
    res = QMessageBox::question( this, tr("delete_question"), tr("really_want_delete?"), QMessageBox::Ok, QMessageBox::Cancel);
    if( QMessageBox::Ok != res ) {
        return;
    }
    foreach( int idx, indexes ) {
        sr->unregisterScript( scripts.at(idx) );
    }
    updateState();
}
void ScriptManagerWidgetImpl::sl_Reload() {
    QSet<int> indexes = selectedIndexes( scriptTable->selectedItems() );
    if( indexes.empty() ) {
        return;
    }
    QList<Task *> reload_tasks;
    foreach( int idx, indexes ) {
        Script * scr = scripts.at( idx );
        Task * t = new ReloadScriptTask( scr );
        scheduled_tasks.push_back( t );
        AppContext::getTaskScheduler()->registerTopLevelTask( t );
//        reload_tasks.push_back( new ReloadScriptTask(scr) );
    }
//    AppContext::getTaskScheduler()->registerTopLevelTask( new MultiTask(tr("reload_scripts_task"), reload_tasks) );
}

void ScriptManagerWidgetImpl::sl_scriptReloaded( Task * t ) {
    Script * scr = 0;
    if( scheduled_tasks.contains(t) && t->isFinished() ) {
        scheduled_tasks.removeAll(t);
        ReloadScriptTask * rst = qobject_cast<ReloadScriptTask*>(t);
        assert( rst );
        scr = rst->getScript();
        int row = scripts.indexOf( scr );
        assert( row >= 0 );
        updateItem( row, scr );
    }
}

} //namespace
