/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TaskViewController.h"
#include "../CoreServiceTypes.h"

#include <core_api/Task.h>
#include <core_api/Settings.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>

#include <QtGui/QVBoxLayout>
#include <QtGui/QHeaderView>
#include <QtGui/QMenu>

/* TRANSLATOR GB2::TaskViewDockWidget */

namespace GB2 {

static LogCategory log(ULOG_CAT_TASKS);

#define SETTINGS_ROOT QString("task_view/")

TaskViewDockWidget::TaskViewDockWidget() {
    waitingIcon = QIcon(":ugene/images/hourglass.png");
    activeIcon = QIcon(":ugene/images/hourglass_go.png");
    wasErrorIcon = QIcon(":ugene/images/hourglass_err.png");
    finishedIcon  = QIcon(":ugene/images/hourglass_ok.png");

    setObjectName(DOCK_TASK_VIEW);
    setWindowTitle(tr("Tasks"));
    setWindowIcon(QIcon(":ugene/images/clock.png"));

    QVBoxLayout* l = new QVBoxLayout();
    l->setSpacing(0);
    l->setMargin(0);
    setLayout(l);

    tree = new QTreeWidget(this);
    l->addWidget(tree);
    
    buildTree();
    tree->setColumnWidth(0, 400); //TODO: save geometry!
    tree->setColumnWidth(1, 250);
    tree->setContextMenuPolicy(Qt::CustomContextMenu);

    connect(tree, SIGNAL(customContextMenuRequested(const QPoint &)), SLOT(sl_onContextMenuRequested(const QPoint &)));
    connect(tree, SIGNAL(itemSelectionChanged()), SLOT(sl_onTreeSelectionChanged()));
    connect(tree, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)), SLOT(sl_itemDoubleClicked(QTreeWidgetItem*, int)));

    initActions();
    updateState();
}

TaskViewDockWidget::~TaskViewDockWidget() {
    QByteArray headerState = tree->header()->saveState();
    AppContext::getSettings()->setValue(SETTINGS_ROOT + "tree_header", headerState);
}

void TaskViewDockWidget::initActions() {
    cancelTaskAction  = new QAction(tr("cancel_task_action"), this);
    cancelTaskAction->setIcon(QIcon(":ugene/images/cancel.png"));
    connect(cancelTaskAction, SIGNAL(triggered()), SLOT(sl_onCancelTask()));

    viewReportAction = new QAction(tr("view_task_report_action"), this);
    viewReportAction->setIcon(QIcon(":ugene/images/task_report.png"));
    connect(viewReportAction, SIGNAL(triggered()), SLOT(sl_onViewTaskReport()));

    removeReportAction = new QAction(tr("remove_task_report_action"), this);
    removeReportAction->setIcon(QIcon(":ugene/images/bin_empty.png"));
    connect(removeReportAction, SIGNAL(triggered()), SLOT(sl_onRemoveTaskReport()));


    TaskScheduler* s = AppContext::getTaskScheduler();
    connect(s, SIGNAL(si_topLevelTaskRegistered(Task*)), SLOT(sl_onTopLevelTaskRegistered(Task*)));
    connect(s, SIGNAL(si_topLevelTaskUnregistered(Task*)), SLOT(sl_onTopLevelTaskUnregistered(Task*)));
    connect(s, SIGNAL(si_stateChanged(Task*)), SLOT(sl_onStateChanged(Task*)));
}

void TaskViewDockWidget::updateState() {
    QList<QTreeWidgetItem*> selItems = tree->selectedItems();
    bool hasActiveTask = false;
    bool hasTaskWithReport = false;
    foreach(QTreeWidgetItem* i, selItems) {
        TVTreeItem* ti = (TVTreeItem*)i;
        hasActiveTask = hasActiveTask || ti->task!=NULL;
        hasTaskWithReport = hasTaskWithReport || ti->reportButton!=NULL;
    }
    cancelTaskAction->setEnabled(hasActiveTask);
    viewReportAction->setEnabled(hasTaskWithReport && selItems.size() == 1);
    removeReportAction->setEnabled(hasTaskWithReport && selItems.size() == 1);
}

void TaskViewDockWidget::buildTree() {
    tree->setColumnCount(TVColumns_NumCols);
    QStringList labels;
    for(int i=0;i < TVColumns_NumCols; i++) {
        switch((TVColumns)i) {
            case TVColumns_Name:    labels<<tr("task_name_column");         break;
            case TVColumns_Desc:    labels<<tr("task_current_desc_column"); break;
            case TVColumns_Progress:labels<<tr("task_progress_column");     break;
            case TVColumns_Actions: labels<<tr("task_actions_column");      break;
            case TVColumns_NumCols:                                         break;
        }
    }
    tree->setColumnWidth(TVColumns_Actions, 50);
    tree->setHeaderLabels(labels);
    tree->setSortingEnabled(false);
    QByteArray headerState = AppContext::getSettings()->getValue(SETTINGS_ROOT + "tree_header", QByteArray()).toByteArray();
    if (!headerState.isEmpty()) {
        tree->header()->restoreState(headerState);	
    }

    const QList<Task*>& topLevelTasks = AppContext::getTaskScheduler()->getTopLevelTasks();
    foreach(Task* t, topLevelTasks) {
        addTaskTree(NULL, t);
    }
}


void TaskViewDockWidget::addTaskTree(TVTreeItem* parentItem, Task* task) {
    TVTreeItem* i = new TVTreeItem(this, task);
    foreach(Task* t, task->getSubtasks()) {
        addTaskTree(i, t);
    }
    if (parentItem!=NULL) {
        parentItem->addChild(i);
    } else {
        tree->addTopLevelItem(i);
        QWidget* w = new QWidget();
        QHBoxLayout* l = new QHBoxLayout();
        l->addStretch(10);
        l->setMargin(0);
        l->setSpacing(10);
        l->addSpacing(10);
        w->setLayout(l);
        
        
        i->cancelButton = new TVButton(i);
        i->cancelButton->setFlat(true);
        i->cancelButton->setIcon(cancelTaskAction->icon());
        i->cancelButton->setFixedSize(QSize(20, 20));
        connect(i->cancelButton, SIGNAL(clicked()), SLOT(sl_cancelTaskByButton()));
        
        l->insertWidget(1, i->cancelButton);

        tree->setItemWidget(i, TVColumns_Actions, w);

    }
    connect(task, SIGNAL(si_subtaskAdded(Task*)), SLOT(sl_onSubtaskAdded(Task*)));
    connect(task, SIGNAL(si_progressChanged()), SLOT(sl_onTaskProgress()));
    connect(task, SIGNAL(si_descriptionChanged()), SLOT(sl_onTaskDescription()));
}

void TaskViewDockWidget::sl_itemDoubleClicked(QTreeWidgetItem *item, int column) {
    Q_UNUSED(column);
    TVTreeItem* ti = (TVTreeItem*)item;
    if (ti!=NULL && ti->reportButton != NULL) {
        activateReport(ti);
    }
}

void TaskViewDockWidget::sl_cancelTaskByButton() {
    TVButton* b = qobject_cast<TVButton*>(sender());
    if (b->ti->task!=NULL) {
        b->ti->task->cancel();
    } else {
       removeReport(b->ti);
    }
}

TVTreeItem* TaskViewDockWidget::findItem(Task* t, bool topLevelOnly) const {
    for (int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        QTreeWidgetItem* item = tree->topLevelItem(i);
        TVTreeItem* ti = (TVTreeItem*)item;
        if (ti->task == t) {
            return ti;
        }
        if (ti->task != t->getTopLevelParentTask()) {
            continue;
        }
        if (!topLevelOnly) {
            TVTreeItem* res = findChildItem(ti, t);
            if (res != NULL) {
                return res;
            }
        }
    }
    return NULL;
}

TVTreeItem* TaskViewDockWidget::findChildItem(TVTreeItem* ti, Task* t) const {
    for (int i=0, n = ti->childCount(); i<n; i++) {
        QTreeWidgetItem* item = ti->child(i);
        TVTreeItem* cti = (TVTreeItem*)item;
        if (cti->task == t) {
            return cti;
        }
        TVTreeItem* res = findChildItem(cti, t);
        if (res != NULL) {
            return res;
        }
    }
    return NULL;
}

void TaskViewDockWidget::sl_onTopLevelTaskRegistered(Task* t) {
    addTaskTree(NULL, t);
}

void TaskViewDockWidget::sl_onTopLevelTaskUnregistered(Task* t) {
    TVTreeItem* ti = findItem(t, true);
    assert(ti!=NULL);

    if (t->isReportingEnabled()) {
        log.info(tr("New report available from task: %1").arg(t->getTaskName()));
        
        ti->detachFromTask();
        ti->reportButton = new TVButton(ti);
        ti->reportButton->setFlat(true);
        ti->reportButton->setIcon(viewReportAction->icon());
        ti->reportButton->setFixedSize(QSize(20, 20));
        connect(ti->reportButton, SIGNAL(clicked()), SLOT(sl_activateReportByButton()));
        
        QWidget*  w = tree->itemWidget(ti, TVColumns_Actions);
        QHBoxLayout* l = (QHBoxLayout*)w->layout();
        l->insertWidget(1, ti->reportButton);

        ti->cancelButton->setIcon(removeReportAction->icon());

        emit si_reportsCountChanged();
    } else {
        delete ti;
    }
}

void TaskViewDockWidget::sl_activateReportByButton() {
    TVButton* b = qobject_cast<TVButton*>(sender());
    activateReport(b->ti);
}

void TaskViewDockWidget::sl_onViewTaskReport() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    TVTreeItem* ti = (TVTreeItem*)items.first();
    activateReport(ti);
}

int TaskViewDockWidget::countAvailableReports() const {
    int res = 0;
    for (int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        QTreeWidgetItem* item = tree->topLevelItem(i);
        TVTreeItem* ti = (TVTreeItem*)item;
        res+=ti->reportButton == NULL ? 0 : 1;
    }
    return res;
}

void TaskViewDockWidget::removeReport(TVTreeItem* ti) {
    assert(ti->reportButton!=NULL);
    delete ti;
    emit si_reportsCountChanged();
}

void TaskViewDockWidget::sl_onRemoveTaskReport() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    TVTreeItem* ti = (TVTreeItem*)items.first();
    removeReport(ti);
}


void TaskViewDockWidget::activateReport(TVTreeItem* ti) {
    assert(ti->reportButton!=NULL);
    log.trace(tr("Activating task report: %1").arg(ti->taskName));

    MWMDIManager* mdi = AppContext::getMainWindow()->getMDIManager();
    MWMDIWindow* w = ti->reportWindowId!=-1 ? NULL : mdi->getWindowById(ti->reportWindowId);
    if (w != NULL) {
        mdi->activateWindow(w);
        return;
    } 
    w = new TVReportWindow(ti->taskName, ti->taskId, ti->taskReport);
    mdi->addMDIWindow(w);
    ti->reportWindowId =  w->getId();
}

void TaskViewDockWidget::sl_onSubtaskAdded(Task* sub) {
    Task* parent = qobject_cast<Task*>(sender());
    TVTreeItem* ti = findItem(parent, false);    
    assert(ti);
    addTaskTree(ti, sub);
    ti->updateVisual();
}

void TaskViewDockWidget::sl_onTaskProgress()  {
    Task* t = qobject_cast<Task*>(sender());
    TVTreeItem* ti = findItem(t, false);    
    assert(ti);
    ti->updateVisual();
}

void TaskViewDockWidget::sl_onTaskDescription()  {
    Task* t = qobject_cast<Task*>(sender());
    TVTreeItem* ti = findItem(t, false);    
    assert(ti);
    ti->updateVisual();
}

void TaskViewDockWidget::sl_onStateChanged(Task* t) {
    TVTreeItem* ti = findItem(t, false);
    assert(ti!=NULL);
    if (t->isFinished()) {
        ti->wasError = t->hasErrors();
        ti->wasCanceled = t->isCanceled();
    }
    ti->updateVisual();
}

void TaskViewDockWidget::sl_onTreeSelectionChanged() {
    updateState();
}

void TaskViewDockWidget::sl_onContextMenuRequested(const QPoint & pos) {
    Q_UNUSED(pos);

    QMenu popup;
    popup.addAction(viewReportAction);
    if (removeReportAction->isEnabled()) {
        popup.addAction(removeReportAction);
    } else {
        popup.addAction(cancelTaskAction);
    }
    popup.exec(QCursor::pos());
}


void TaskViewDockWidget::sl_onCancelTask() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(!items.isEmpty());
    foreach(QTreeWidgetItem* item, items) {
        TVTreeItem* tv = (TVTreeItem*)item;
        tv->task->cancel();
    }
}

void TaskViewDockWidget::selectTask(Task* t) {
    TVTreeItem* ti = findItem(t, true);
    if (ti == NULL) {
        return;
    }
    tree->scrollToItem(ti);
    ti->setSelected(true);

}

TVReportWindow::TVReportWindow(const QString& taskName, qint64 tid, const QString& report) 
: MWMDIWindow(genWindowName(taskName)), taskId(tid)
{
    QVBoxLayout* l = new QVBoxLayout();
    l->setMargin(0);
    setLayout(l);

    textEdit = new QTextEdit();
    textEdit->setAcceptRichText(true);
    textEdit->setReadOnly(true);
    textEdit->setText(report);
    
    l->addWidget(textEdit);
}

QString TVReportWindow::genWindowName(const QString& taskName) {
    return tr("report_for_%1").arg(taskName);
}


QString TVReportWindow::prepareReportHTML(Task* t) {
    assert(t->isFinished() && t->isReportingSupported() && t->isReportingEnabled());
    QString report = "<table>";
    report+="<tr><td><center><h1>"+genWindowName(t->getTaskName())+"</h1></center><br><br></td></tr>";
    report+="<tr><td>";

    report+="<table>";
    QString status = t->hasErrors() ? tr("Failed") : t->isCanceled() ? tr("Canceled") : tr("Finished");
    report+="<tr><td width=200><b>"+tr("status")+"</b></td><td>" +status+ "</td></tr>";
    if (t->hasErrors()) {
    report+="<tr><td><b>"+tr("error:")+"</b></td><td>" + t->getError()+ "</td></tr>";
    }

    int msecs = t->getTimeInfo().startTime.time().msecsTo(t->getTimeInfo().finishTime.time());
    QString time = QTime().addMSecs(msecs).toString("h.mm.ss.zzz");

    report+="<tr><td><b>"+tr("time")+"</b></td><td>" +time+ "</td></tr>";
    report+="</td></tr>";
    report+="</table>";

    report+="</td></tr><tr><td>";


    report+=t->generateReport()+"<br>";
    report+="</td></tr></table>";
    return report;
}



//////////////////////////////////////////////////////////////////////////
// Tree Items
TVTreeItem::TVTreeItem(TaskViewDockWidget* _w, Task* t) : task(t), w(_w) {
    reportWindowId = -1;
    taskId = task->getTaskId();
    taskName = task->getTaskName();
    assert(!taskName.isEmpty());
    reportButton = NULL;
    cancelButton = NULL;
    wasCanceled = false;
    wasError = false;
    updateVisual();
}

void TVTreeItem::updateVisual() {
    setText(TVColumns_Name, taskName);
    if (task->isFinished()) {
        setIcon(TVColumns_Name, wasError ? w->wasErrorIcon : w->finishedIcon);
    } else {
        setIcon(TVColumns_Name, task->isRunning() ? w->activeIcon : w->waitingIcon);
    }
    if (task != NULL) {
        QString state;
        if (task->isCanceled()) {
            if (task->isFinished()) {
                state = TaskViewDockWidget::tr("Canceled");
            } else {
                state = TaskViewDockWidget::tr("Canceling...");
            }
        } else {
            state = task->getStateInfo().stateDesc;
            if (state.isEmpty() || task->isFinished()) {
                state = AppContext::getTaskScheduler()->getStateName(task);
            }
        }

        setText(TVColumns_Desc, state);
        if (task->hasErrors()) {
            setToolTip(TVColumns_Name, task->getStateInfo().error);
            setToolTip(TVColumns_Desc, task->getStateInfo().error);
        }

        int progress = task->getProgress();
        setText(TVColumns_Progress, progress < 0 ? "?" : QString::number(progress) + "%");
    } else {
        setText(TVColumns_Desc, TaskViewDockWidget::tr("finished"));
        setText(TVColumns_Progress, "100%");
    }
}

void TVTreeItem::detachFromTask() {
    if (task->isTopLevelTask()) {
        assert(task->isReportingEnabled());
        taskReport = TVReportWindow::prepareReportHTML(task);
    }
    for (int i=0;i<childCount(); i++) {
        TVTreeItem* ci = (TVTreeItem*)child(i);
        ci->detachFromTask();
    }
    task = NULL;
}






}//namespace
