/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ShutdownTask.h"

#include <core_api/Log.h>
#include <core_api/ServiceModel.h>
#include <core_api/PluginModel.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>

namespace GB2 {

static LogCategory log(ULOG_CAT_USER_INTERFACE);

ShutdownTask::ShutdownTask(QCoreApplication* app) : Task(tr("shutdown_task_name"), TaskFlags_NR_FOSCOE | TaskFlag_NoAutoDelete), app(app) 
{
    connect(AppContext::getTaskScheduler(), SIGNAL(si_topLevelTaskUnregistered(Task*)), SLOT(startShutdown()));
    connect(app, SIGNAL(aboutToQuit()), SLOT(startShutdown()));
}

void ShutdownTask::startShutdown() {
    
    if (sender() == app) {
        log.info("Shutdown initiated by user");
    } else {
        if (AppContext::getTaskScheduler()->getTopLevelTasks().size() != 0) {
            return;
        }
        log.info("All tasks finished, shutting down");
    }
    app->disconnect(this, SLOT(startShutdown()));
    AppContext::getTaskScheduler()->disconnect(this, SLOT(startShutdown()));
    AppContext::getTaskScheduler()->registerTopLevelTask(this);
}

static bool isReadyToBeDisabled(Service* s, ServiceRegistry* sr) {
	ServiceType st = s->getType();
	int nServicesOfTheSameType = sr->findServices(st).size();
	assert(nServicesOfTheSameType >= 1);
	foreach(Service* child, sr->getServices()) {
		if (!child->getParentServiceTypes().contains(st) || !child->isEnabled()) {
			continue;
		}
		if (nServicesOfTheSameType == 1) {
			return false;
		}
	}
	return true;
}

static Service* findServiceToDisable(ServiceRegistry* sr) {
	int nEnabled = 0;
	foreach(Service* s, sr->getServices()) {
		nEnabled+= s->isEnabled() ? 1 : 0;
		if (s->isEnabled() && isReadyToBeDisabled(s, sr)) {
			return s;
		}
	}
	assert(nEnabled == 0);
	return NULL;
}

class CancelAllTask : public Task {
public:
    CancelAllTask() : Task(GB2::ShutdownTask::tr("Cancel active tasks"), TaskFlag_NoRun) {}
    void prepare() {
        // cancel all tasks but ShutdownTask
        QList<Task*> activeTopTasks = AppContext::getTaskScheduler()->getTopLevelTasks();
        activeTopTasks.removeOne(getTopLevelParentTask());
        foreach(Task* t, activeTopTasks) {
            log.trace(QString("Canceling: %1").arg(t->getTaskName()));
            t->cancel();
        }
    }

    ReportResult report() {
        foreach(Task* t, AppContext::getTaskScheduler()->getTopLevelTasks()) {
            if (t->isCanceled() && !t->isFinished()) {
                return ReportResult_CallMeAgain;
            }
        }
        return ReportResult_Finished;
    }
};


void ShutdownTask::prepare() {
    log.info(tr("Starting shutdown process..."));

    addSubTask(new CancelAllTask());
}

QList<Task*> ShutdownTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;

    if (isCanceled() || subTask->hasErrors()) {
        return res; //stop shutdown process
    }

	ServiceRegistry* sr = AppContext::getServiceRegistry();
	Service* s = findServiceToDisable(sr);
	if (s!=NULL) {
		res.append(sr->disableServiceTask(s));
	}
	return res;
}

Task::ReportResult ShutdownTask::report() {
    if (stateInfo.cancelFlag) {
        log.info(tr("Shutdown was canceled"));
        return Task::ReportResult_Finished;
    } 
    if (propagateSubtaskError()) {
        log.error(tr("Shutdown failed, error: %1").arg(stateInfo.getError()));
        return Task::ReportResult_Finished;
    }

#ifdef _DEBUG
	const QList<Service*>& services = AppContext::getServiceRegistry()->getServices();
	foreach(Service* s, services) {
		assert(s->isDisabled());
	}
#endif
    app->quit();
	return Task::ReportResult_Finished;
}

}//namespace
