/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNATranslationImpl.h"

#include <util_text/TextUtils.h>
#include <core_api/DNAAlphabet.h>

#include <math.h>

namespace GB2 {


DNATranslation1to1Impl::DNATranslation1to1Impl(const QString& _name, DNAAlphabet* src, DNAAlphabet* dst, QByteArray mapper121)
: DNATranslation(_name, _name, src, dst)
{
	map = mapper121;
	assert(map.size() == 256);
}

int DNATranslation1to1Impl::translate(const char* src, int src_len, char* dst, int dst_capacity) const {
	int len = qMin(src_len, dst_capacity);
	TextUtils::translate(map, src, len, dst);
	return len;
}

int DNATranslation1to1Impl::translate(char* src_and_dst, int len) const {
    TextUtils::translate(map, src_and_dst, len);
    return len;
}

//////////////////////////////////////////////////////////////////////////
/// 3->1

DNATranslation3to1Impl::DNATranslation3to1Impl(const QString& _id, const QString& _name, 
                                               DNAAlphabet* src, DNAAlphabet* dst, 
                                               const QList<Mapping3To1<char> >& rm, char dc,
                                               const QMap<DNATranslationRole,QList<Triplet> >& _roles)
:  DNATranslation(_id, _name, src, dst), index(rm, dc), roles(_roles)
{
    codons = new char*[DNATranslationRole_Num];
    cod_lens = new int[DNATranslationRole_Num];
    for (int j=0;j<DNATranslationRole_Num; j++ )
    {
        cod_lens[j] = 0;
    }
    
    QMapIterator<DNATranslationRole,QList<Triplet> > i(roles);
    while (i.hasNext()) {
        i.next();
        QList<Triplet> l = i.value();
        DNATranslationRole r = i.key();
        cod_lens[r] = l.size()*3;
        codons[r] = new char[l.size()*3];
        for (int j = 0; j<l.size(); j++)
        {
            char* codon = codons[r] + j*3; 
            *codon++ = l.at(j).c[0];
            *codon++ = l.at(j).c[1];
            *codon++ = l.at(j).c[2];
        }
    }
}

DNATranslation3to1Impl::~DNATranslation3to1Impl() {
    for (int j=0;j<DNATranslationRole_Num; j++ ) {
        if (cod_lens[j] != 0) {
            delete[] codons[j];
        }
    }
    delete[] codons;
    delete[] cod_lens;
}

int DNATranslation3to1Impl::translate(const char* src, int src_len, char* dst, int dst_capacity) const {
	bool caseSensitive = srcAlphabet->isCaseSensitive();
	int resLen = qMin(src_len/3, dst_capacity);
	if (caseSensitive) {
		for(int dstIdx=0, srcIdx=0; dstIdx < resLen; dstIdx++, srcIdx+=3) {
			char c = index.map(src + srcIdx);
			dst[dstIdx] = c;
		}
	} else {
		char uc[3];
		const QByteArray& ucMap = TextUtils::UPPER_CASE_MAP;
		for(int dstIdx=0, srcIdx=0; dstIdx < resLen; dstIdx++, srcIdx+=3) {
			TextUtils::translate(ucMap, src+srcIdx, 3, uc);
			char c = index.map(src + srcIdx);
			dst[dstIdx] = c;
		}
	}
	return resLen;
}

int DNATranslation3to1Impl::translate(char* src_and_dst, int len) const {
    return translate(src_and_dst, len, src_and_dst, len);
}

//////////////////////////////////////////////////////////////////////////
// 3-1 index

Index3To1::Index3To1() {
    indexSize = 0;
    bitsPerChar = 0;
    bitsPerCharX2 = 0;
    std::fill(maskByChar, maskByChar+256, 0);
}

void Index3To1::init(const QList<Triplet>& ts) {
    assert(indexSize == 0);
    // count number of different input chars in map
    QBitArray usedChars(256, false);
    foreach(const Triplet& t, ts) {
        usedChars[(quint8)t.c[0]] = true;
        usedChars[(quint8)t.c[1]] = true;
        usedChars[(quint8)t.c[2]] = true;
    }
    int nChars = usedChars.count(true);
    assert(nChars > 0);

    // count number of bit enough to encode all input chars
    bitsPerChar = 0;
    for (int chars = 1; chars <= nChars; chars*=2) {
        bitsPerChar++;
    }
#ifdef _DEBUG
    int prevPow = (int)pow((float)2, bitsPerChar-1);
    int nextPow = (int)pow((float)2, bitsPerChar);
    assert(nChars < nextPow && nChars >= prevPow);
    assert(bitsPerChar<=8); //validity check
    assert(bitsPerChar<=6);//TODO: optimize large tables!!
#endif
    bitsPerCharX2 = bitsPerChar * 2; //cache this value -> used for optimization
    
    // assign bit mask for every char
    char mask = 1;
    for (int i = 0; i < 256; i++) {
        if (usedChars[i]) {
            maskByChar[i] = mask;
            mask++;
        } else {
            maskByChar[i] = 0;
        }
    }
    mask--; //last value was not used

    indexSize = 1 + (mask << bitsPerCharX2) + (mask<<bitsPerChar) + mask;

    assert(indexSize > 0);
}

void Index3To1::init(const QByteArray& alphabetChars) {
    int nChars = alphabetChars.size();
    QList<Triplet> ts;
    for (int i1=0; i1 < nChars; i1++) {
        char c1 = alphabetChars[i1];
        for (int i2=0; i2 < nChars; i2++) {
            char c2 = alphabetChars[i1];
            for (int i3=0; i3 < nChars; i3++) {
                char c3 = alphabetChars[i1];                
                Triplet m(c1, c2, c3);
                ts.append(m);
            }
        }
    }
    init(ts);
}

}//namespace
 
