/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_DYN_TABLE_H_
#define _GB2_DYN_TABLE_H_

#include "RollingMatrix.h"

namespace GB2 {

struct GB2_COREAPI_EXPORT MatchScores
{
    int match;
    int mismatch;
    int ins;
    int del;
};

class GB2_COREAPI_EXPORT DynTable : public RollingMatrix {
public:
    enum FillStrategy
    {
        Strategy_Max,
        Strategy_Min
    };
    DynTable(int n, int m, bool _allowInsDel) 
        : RollingMatrix(n, m), allowInsDel(_allowInsDel) 
    {
        init();
        scores.match = DEFAULT_SCORE_MATCH;
        scores.ins = DEFAULT_SCORE_INS;
        scores.del = DEFAULT_SCORE_DEL;
        scores.mismatch = DEFAULT_SCORE_MISMATCH;
        strategy = Strategy_Min;
    }
    DynTable(): RollingMatrix(0,0), allowInsDel(false)
    {
    }
    DynTable( int n, int m, bool _allowInsDel, MatchScores _scores, FillStrategy _strategy = Strategy_Min )
        : RollingMatrix(n, m), allowInsDel(_allowInsDel), scores(_scores), strategy(_strategy)
    {
        init();
    }

    int getLast() const {
        return get_value(n-1, m-1);
    }
    
    int getLastLen() const {
        return getLen(n-1, m-1);
    }
    
    void match(int y, bool ok) {
        match( n-1, y, ok );
    }

    virtual int get(int x, int y) const {
        if (y<0) {return 0;}
        if (x<0) {return y+1;}
        return RollingMatrix::get(x, y);
    }
protected:
    void match( int x, int y, bool ok )
    {
        int d = get_value( x-1, y-1 );
        int res = d + (ok ? scores.match : scores.mismatch);
        if( allowInsDel ) {
            int u = get_value(x, y-1);
            int l = get_value(x-1, y);
            int insdelRes = 0;
            switch( strategy )
            {
            case Strategy_Min:
                insdelRes = qMin( l+scores.ins, u+scores.del );
                res = qMin( insdelRes, res );
                break;
            default:
                assert( false );
            }
        }
        set_pair( x, y, res, ok );
    }

    int getLen(int x, int y) const { 
        if (y == -1) {
            return 0;
        }
        assert(x!=-1);
        
        if (!allowInsDel) {
            return 1 + getLen(x-1, y-1);
        }
        int v = get_value(x, y);
        bool match = get_flag( x, y );
        int d = get_value(x-1, y-1);
        int l = get_value(x-1, y);
        int u = get_value(x, y-1);
                
        if( match && v == d + scores.match ) {
            return 1 + getLen( x-1, y-1 );
        }
        if ( v == u + scores.del ) { //prefer deletion in X sequence to minimize result len
            return getLen(x, y-1);
        } 
        if ( !match && v == d + scores.mismatch ) { // prefer mismatch instead of insertion into X sequence
            return 1 + getLen(x-1, y-1);
        } 
        assert( v == l + scores.ins ); Q_UNUSED(l);
        return 1 + getLen(x-1, y); // this is insertion into X sequence
    }
private:
    void init()
    {
        for (int i=0; i<n; i++) {
            for (int j=0; j<m; j++) {
                set_pair(i, j, j+1, false);
            }
        }
    }
    void set_value( int x, int y, int val )
    {
        assert( !(val & MASK_FLAG_MISMATCH) );
        int oldval = get( x, y );
        set( x, y, (oldval & MASK_FLAG_MISMATCH) | val );
    }
    void set_mm_flag( int x, int y, bool flag )
    {
        int oldval = get( x, y );
        set( x, y, (oldval & MASK_VALUE)|(flag ? Flag_Match : Flag_Mismatch) );
    }
    void set_pair( int x, int y, int val, bool flag )
    {
        assert( !(val & MASK_FLAG_MISMATCH) );
        set( x, y, val | (flag ? Flag_Match : Flag_Mismatch) );
    }
    int get_value( int x, int y ) const
    {
        return get( x, y ) & MASK_VALUE;
    }
    bool get_flag( int x, int y ) const
    {
        return get( x, y ) & MASK_FLAG_MISMATCH;
    }
protected:
    bool allowInsDel;
private:
    const static int DEFAULT_SCORE_MATCH = 0;
    const static int DEFAULT_SCORE_MISMATCH = 1;
    const static int DEFAULT_SCORE_DEL = 1;
    const static int DEFAULT_SCORE_INS = 1;

    const static int MASK_VALUE = 0x7FFFFFFF;
    const static int MASK_FLAG_MISMATCH = 0x80000000;
    enum MismatchFlag
    {
        Flag_Mismatch = 0x00000000,
        Flag_Match = 0x80000000
    };

    MatchScores scores;
    FillStrategy strategy;
};

} //namespace

#endif
