/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DetView.h"
#include "ADVSequenceObjectContext.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/DNATranslation.h>
#include <core_api/AppContext.h>

#include <document_format/DNATranslationImpl.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/AnnotationSettings.h>

#include <selection/DNASequenceSelection.h>
#include <util_gui/GraphUtils.h>
#include <util_gui/GScrollBar.h>
#include <util_text/TextUtils.h>

#include <QtGui/QTextEdit>
#include <QtGui/QPainter>
#include <QtGui/QFontMetrics>
#include <QtGui/QMenu>
#include <QtGui/QApplication>


namespace GB2 {

DetView::DetView(QWidget* p, ADVSequenceObjectContext* ctx) 
: GSequenceLineViewAnnotated(p, ctx)
{
    showComplementAction = new QAction(tr("show_complement"), this);
    showComplementAction->setIcon(QIcon(":core/images/show_compl.png"));

    showTranslationAction = new QAction(tr("show_translation"), this);
    showTranslationAction->setIcon(QIcon(":core/images/show_trans.png"));

    showComplementAction->setCheckable(true);
    showTranslationAction->setCheckable(true);

    bool hasComplement = ctx->getComplementTT()!=NULL;
    showComplementAction->setChecked(hasComplement);
    
    bool hasAmino = ctx->getAminoTT()!=NULL; 
    showTranslationAction->setChecked(hasAmino);

    connect(showComplementAction, SIGNAL(triggered(bool)), SLOT(sl_showComplementToggle(bool)));
    connect(showTranslationAction, SIGNAL(triggered(bool)), SLOT(sl_showTranslationToggle(bool)));

    assert(ctx->getSequenceObject()!=NULL);
    featureFlags&=!GSLV_FF_SupportsCustomRange;
    renderArea = new DetViewRenderArea(this);
    renderArea->setObjectName("render_area");

    connect(ctx, SIGNAL(si_aminoTranslationChanged()), SLOT(sl_onAminoTTChanged()));

	pack();
}

void DetView::updateSize() {
    ((DetViewRenderArea*)renderArea)->updateSize();
    setFixedHeight(renderArea->height() + scrollBar->height());
}

void DetView::resizeEvent(QResizeEvent *e) { 
    int w = width();
    int charWidth = renderArea->getCharWidth();
    int visibleSymbolsCount = w / charWidth;
    
    if (visibleSymbolsCount > seqLen) {
        visibleRange.startPos = 0;
        visibleRange.len = seqLen;
    } else {
        visibleRange.len = visibleSymbolsCount;
        if (visibleRange.endPos() > seqLen) {
            visibleRange.startPos = seqLen - visibleSymbolsCount;
        }
    }

    Q_ASSERT(visibleRange.startPos >= 0 && visibleRange.endPos()<=seqLen);

    GSequenceLineView::resizeEvent(e);

    onVisibleRangeChanged();
}

void DetView::updateActions() {
    bool visible = isVisible();

    bool hasComplement = ctx->getComplementTT()!=NULL;
    showComplementAction->setEnabled(hasComplement && visible);

    bool hasAmino = ctx->getAminoTT()!=NULL; 
    showTranslationAction->setEnabled(hasAmino && visible);
}

void DetView::showEvent(QShowEvent * e) {
    updateActions();
    GSequenceLineViewAnnotated::showEvent(e);
}

void DetView::hideEvent(QHideEvent * e) {
    updateActions();
    GSequenceLineViewAnnotated::hideEvent(e);
}

void DetView::sl_onAminoTTChanged() {
    lastUpdateFlags|=GSLV_UF_NeedCompleteRedraw;
    update();
}

DNATranslation* DetView::getComplementTT() const {
    return showComplementAction->isChecked() ? ctx->getComplementTT() : NULL;
}

DNATranslation* DetView::getAminoTT() const {
    return showTranslationAction->isChecked() ? ctx->getAminoTT() : NULL;
}


void DetView::setShowComplement(bool t) {
    showComplementAction->disconnect(this);
    showComplementAction->setChecked(t);
    connect(showComplementAction, SIGNAL(triggered(bool)), SLOT(sl_showComplementToggle(bool)));
    
    updateSize();
}

void DetView::setShowTranslation(bool t) {
    showTranslationAction->disconnect(this);
    showTranslationAction->setChecked(t);
    connect(showTranslationAction, SIGNAL(triggered(bool)), SLOT(sl_showTranslationToggle(bool)));

    updateSize();
}

void DetView::mouseReleaseEvent(QMouseEvent* me) {
    //click with 'alt' shift selects single base in GSingleSeqWidget;
    //here we adjust this behavior -> if click was done in translation line -> select 3 bases
    Qt::KeyboardModifiers km = QApplication::keyboardModifiers();
    bool singleBaseSelectionMode = km.testFlag(Qt::AltModifier);
    if (me->button() == Qt::LeftButton && singleBaseSelectionMode) {
        QPoint areaPoint = toRenderAreaPoint(me->pos());
        if (((DetViewRenderArea*)renderArea)->isOnTranslationsLine(areaPoint.y())) {
            int pos = renderArea->coordToPos(areaPoint.x());
            if (pos == lastPressPos) {
                LRegion rgn(pos-1, 3);
                if (rgn.startPos >=0 && rgn.endPos() <= seqLen) {
                    setSelection(rgn);
                    lastPressPos=-1;
                }
            }
        }
    }
    GSequenceLineViewAnnotated::mouseReleaseEvent(me);
}



//////////////////////////////////////////////////////////////////////////
/// render
DetViewRenderArea::DetViewRenderArea(DetView* v) : GSequenceLineViewAnnotatedRenderArea(v, true) {
    updateSize();
}

void DetViewRenderArea::updateLines() {
	numLines = -1;
	rulerLine = -1;
	baseLine = -1;
	complementLine = -1;
	firstDirectTransLine = -1;
	firstComplTransLine = -1;

	DetView* detView = getDetView();
	if (detView->isOneLineMode()) {
		baseLine = 0;
		rulerLine = 1;
		numLines = 2;
	} else if (detView->hasComplementStrand() && detView->hasTranslations()) {
		firstDirectTransLine = 0;
		baseLine = 3;
		rulerLine = 4;
		complementLine = 5;
		firstComplTransLine = 6;
		numLines = 9;
	} else if (detView->hasComplementStrand()) {
		assert(!detView->hasTranslations());	
		baseLine = 0;
		rulerLine = 1;
		complementLine = 2;
		numLines = 3;
	} else {
		assert(!detView->hasComplementStrand() && detView->hasTranslations());
		firstDirectTransLine = 0;
		baseLine = 3;
		rulerLine = 4;
		numLines = 5;
	}
	assert(numLines > 0);
}

LRegion DetViewRenderArea::getAnnotationYRange(Annotation* a, const LRegion& r, const AnnotationSettings* as) const {
	bool complement = a->isOnComplementStrand() && getDetView()->hasComplementStrand();
    TriState aminoState = a->getAminoStrand();
    if (aminoState == TriState_Unknown) {
        aminoState = as->amino ? TriState_Yes : TriState_No;
    }
    bool transl = getDetView()->hasTranslations() && aminoState == TriState_Yes;
	int line = -1;
	if (complement) {
		if (transl) {
			line = posToComplTransLine(r.endPos());
		} else {
			line = complementLine;
		}
	} else {
		if (transl) {
			line = posToDirectTransLine(r.startPos);
		} else {
			line = baseLine;
		}
	}
	assert(line!=-1);
	int y = getLineY(line);
	return LRegion(y, lineHeight);
}

bool DetViewRenderArea::isOnTranslationsLine(int y) const {
    if (firstDirectTransLine != -1) {
        LRegion dtr(getLineY(firstDirectTransLine), 3*lineHeight);
        if (dtr.contains(y)) {
            return true;
        }
    }
    if (firstComplTransLine !=-1) {
        LRegion ctr(getLineY(firstComplTransLine), 3*lineHeight);
        if (ctr.contains(y)) {
            return true;
        }
    }
    return false;
}

void DetViewRenderArea::drawAll(QPaintDevice* pd) {
    GSLV_UpdateFlags uf = view->getUpdateFlags();
    bool completeRedraw = uf.testFlag(GSLV_UF_NeedCompleteRedraw)  || uf.testFlag(GSLV_UF_ViewResized)  ||
                          uf.testFlag(GSLV_UF_VisibleRangeChanged) || uf.testFlag(GSLV_UF_AnnotationsChanged);
    
    bool hasSelectedAnnotationInRange = isAnnotationSelectionInVisibleRange();

    if (completeRedraw) {
        QPainter pCached(cachedView); 
        pCached.fillRect(0, 0, pd->width(), pd->height(), Qt::white);
        pCached.setPen(Qt::black);
        drawAnnotations(pCached);

        drawDirect(pCached);
        drawComplement(pCached);
        drawTranslations(pCached);

        drawRuler(pCached);

        pCached.end();
    } 

    QPainter p(pd);
    p.drawPixmap(0, 0, *cachedView);

    drawAnnotationsSelection(p);

    if (hasSelectedAnnotationInRange) {
        drawDirect(p);
        drawComplement(p);
        drawTranslations(p);
    }
 
    drawSequenceSelection(p);

    if (view->hasFocus()) {
        drawFocus(p);
    }
}


void DetViewRenderArea::drawDirect(QPainter& p) {
	p.setFont(sequenceFont);
    p.setPen(Qt::black);

	const LRegion visibleRange = view->getVisibleRange();
	assert(visibleRange.len * charWidth <= width());

	const QByteArray& sequence = view->getSequenceContext()->getSequenceData();
	const char* seq = sequence.data() + visibleRange.startPos;

    /// draw base line;
    int y = getTextY(baseLine);
    for(int i=0;i < visibleRange.len; i++) {
		char nucl = seq[i];
    	p.drawText(i*charWidth + xCharOffset, y, QString(nucl));
	}
}

void DetViewRenderArea::drawComplement(QPainter& p) {
	p.setFont(sequenceFont);
    p.setPen(Qt::black);

	DetView* detView = getDetView();
	if (complementLine > 0) {
		const LRegion visibleRange = detView->getVisibleRange();

		const QByteArray& sequence = detView->getSequenceContext()->getSequenceData();
		const char* seq = sequence.data() + detView->getVisibleRange().startPos;

		DNATranslation* complTrans = detView->getComplementTT();
		QByteArray map = complTrans->getOne2OneMapper();
		int y = getTextY(complementLine);
		for(int i=0;i< visibleRange.len; i++) {
			char nucl = seq[i];
			char complNucl = map.at(nucl);
			p.drawText(i*charWidth + xCharOffset, y, QString(complNucl));
		}
	}
}

static QByteArray translate(DNATranslation* t, const char* seq, int seqLen) {
	QByteArray res(seqLen / 3, 0);
	int n = t->translate(seq, seqLen, res.data(), res.length());
        assert(n == res.length()); Q_UNUSED(n);
	return res;
}

void DetViewRenderArea::drawTranslations(QPainter& p) {
	p.setFont(sequenceFont);

	if (firstDirectTransLine < 0 && firstComplTransLine < 0) {
		return;
	}
	DetView* detView = getDetView();

	DNATranslation3to1Impl* aminoTable = (DNATranslation3to1Impl*)detView->getAminoTT();
	assert(aminoTable!=NULL && aminoTable->isThree2One());

	const QByteArray& sequence = detView->getSequenceContext()->getSequenceData();
	const LRegion& visibleRange  = detView->getVisibleRange();
	
	int maxUsablePos = qMin(visibleRange.endPos() + 1, sequence.length());
	int minUsablePos = qMax(visibleRange.startPos - 1, 0);

    QColor startC(0,0x99,0);
    QColor stopC(0x99,0,0);
    QFont fontB = sequenceFont;
    fontB.setBold(true);
	QFont fontI = sequenceFont;
	fontI.setItalic(true);

	{//direct translations
		for(int i = 0; i < 3; i++) {
			int indent = (visibleRange.startPos + i) % 3;
			int dnaStartPos = visibleRange.startPos + indent - 3;
			if (dnaStartPos < minUsablePos) {
				dnaStartPos+=3;
			}
			int dnaLen = maxUsablePos - dnaStartPos;
            const char* dnaSeq = sequence.data() + dnaStartPos;
			QByteArray amino = translate(aminoTable, dnaSeq, dnaLen);

			int line = dnaStartPos % 3;
			int y = getTextY(firstDirectTransLine + line);
			int dx = dnaStartPos - visibleRange.startPos;
			for(int j = 0, n = amino.length(); j < n ; j++, dnaSeq += 3) {
				char amin = amino[j];
				int xpos = 3 * j + 1 + dx;
				assert(xpos >= 0 && xpos < visibleRange.len);
				int x =  xpos * charWidth + xCharOffset;
                if (aminoTable->isStartCodon(dnaSeq)) {
                    p.setPen(startC);
                    p.setFont(fontB);
				} else if (aminoTable->isCodon(DNATranslationRole_Start_Alternative, dnaSeq)) {
					p.setPen(startC);
					p.setFont(fontI);
				} else if (aminoTable->isStopCodon(dnaSeq)) {
                    p.setPen(stopC);
                    p.setFont(fontB);
                } else {
                    p.setPen(Qt::black);
                    p.setFont(sequenceFont);
                }
                p.drawText(x, y, QString(amin));
			}
		}
	}
	if (detView->hasComplementStrand()) {//reverse translations	
		DNATranslation* complTable = detView->getComplementTT();
		assert(complTable!=NULL);
		int seqLen = sequence.length();
		int usableSize = maxUsablePos - minUsablePos;
		QByteArray revComplDna(usableSize, 0);
		complTable->translate(sequence.data() + minUsablePos, usableSize, revComplDna.data(), usableSize);
		TextUtils::reverse(revComplDna.data(), revComplDna.size());
		for(int i = 0; i < 3; i++) {
			int indent = (seqLen - visibleRange.endPos() + i) % 3;
			int revComplStartPos = visibleRange.endPos() - indent + 3; //start of the reverse complement sequence in direct coords
			if (revComplStartPos > maxUsablePos) {
				revComplStartPos-=3;
			}
			int dnaLen = revComplStartPos - minUsablePos;
			
			int revComplDnaOffset = maxUsablePos - revComplStartPos;
			assert(revComplDnaOffset >= 0);
            const char* dnaSeq = revComplDna.data() + revComplDnaOffset;
			QByteArray amino = translate(aminoTable, dnaSeq, dnaLen);
			int line = (seqLen - revComplStartPos) % 3;
			int y = getTextY(firstComplTransLine + line);
			int dx = visibleRange.endPos() - revComplStartPos;
			for(int j = 0, n = amino.length(); j < n ; j++, dnaSeq +=3) {
				char amin = amino[j];
				int xpos = visibleRange.len - (3 * j + 2 + dx);
				assert(xpos >= 0 && xpos < visibleRange.len);
				int x =  xpos * charWidth + xCharOffset;
                if (aminoTable->isStartCodon(dnaSeq)) {
                    p.setPen(startC);
                    p.setFont(fontB);
				} else if (aminoTable->isCodon(DNATranslationRole_Start_Alternative, dnaSeq)) {
					p.setPen(startC);
					p.setFont(fontI);
                } else if (aminoTable->isStopCodon(dnaSeq)) {
                    p.setPen(stopC);
                    p.setFont(fontB);
                } else {
                    p.setPen(Qt::black);
                    p.setFont(sequenceFont);
                }
                p.drawText(x, y, QString(amin));
			}
		}
	}
	p.setPen(Qt::black);
	p.setFont(sequenceFont);
}

void DetViewRenderArea::drawSequenceSelection(QPainter& p) {
	DetView* detView = getDetView();
	DNASequenceSelection* sel = detView->getSequenceContext()->getSequenceSelection();
	if (sel->isEmpty()) {
		return;
	}
	
    QPen pen1(Qt::black, 1, Qt::DashLine);
    p.setPen(pen1);

	foreach(const LRegion& r, sel->getSelectedRegions()) {
		highlight(p, r, baseLine);
		if (detView->hasComplementStrand()) {
			highlight(p, r, complementLine);
		}
		if (detView->hasTranslations()) {
			int translLine = posToDirectTransLine(r.startPos);
			highlight(p, r, translLine);			
			if (detView->hasComplementStrand()) {
				int complTransLine = posToComplTransLine(r.endPos());
				highlight(p, r, complTransLine);			
			}
		}
	}
}

void DetViewRenderArea::drawRuler(QPainter& p) {
	int y = getLineY(rulerLine) + 2;
	const LRegion& visibleRange = view->getVisibleRange();
    int firstCharStart = posToCoord(visibleRange.startPos);
    int lastCharStart = posToCoord(visibleRange.endPos()-1);
    int firstCharCenter = firstCharStart + charWidth / 2;
    int firstLastLen = lastCharStart - firstCharStart;
    GraphUtils::RulerConfig c;
    GraphUtils::drawRuler(p, QPoint(firstCharCenter, y), firstLastLen, visibleRange.startPos + 1, visibleRange.endPos(), rulerFont, c);	
}


int DetViewRenderArea::posToDirectTransLine(int p) const {
	assert(firstDirectTransLine >= 0);
	return firstDirectTransLine + p % 3;
}

int DetViewRenderArea::posToComplTransLine(int p) const {
	assert(firstComplTransLine >= 0);
	return firstComplTransLine + (view->getSequenceLen() - p) % 3;
}


void DetViewRenderArea::highlight(QPainter& p, const LRegion& r, int line) {
	const LRegion& visibleRange = view->getVisibleRange();
	if (!visibleRange.intersects(r)) {
		return;
	}
	LRegion visibleRegion = visibleRange.intersect(r);
	int x = posToCoord(visibleRegion.startPos);
	int width = posToCoord(visibleRegion.endPos()) - x;

	int ymargin = yCharOffset / 2;
	int y = getLineY(line) + ymargin;
	int height = lineHeight - 2 * ymargin;
	p.drawRect(x, y, width, height);
}

int DetViewRenderArea::coordToPos(int x) const {
	LRegion visibleRange = view->getVisibleRange();
        int pos = visibleRange.startPos + int(x / (float)charWidth + 0.5f);
	if (pos > visibleRange.endPos()) {
		pos = visibleRange.endPos();
	}
	return pos;
}

float DetViewRenderArea::posToCoordF(int x, bool useVirtualSpace) const {
	const LRegion& visible = view->getVisibleRange();
	if (!useVirtualSpace && !visible.contains(x) && visible.endPos()!=x) {
		return -1;
	}
	float res = (float)(x - visible.startPos) * charWidth;
	assert(useVirtualSpace || (res >=0 && res <= width()));
	return res;
}

float DetViewRenderArea::getCurrentScale() const {
	assert(0); //TODO: must never be called. Not tested if called
	return (float)charWidth;
} 

void DetViewRenderArea::updateSize()  {
    updateLines();
    int h = numLines * lineHeight + 5;
    setFixedHeight(h); //todo: remove +5 and fix ruler drawing to fit its line
}

}//namespace
