/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowIOTasks.h"

#include "SchemaSerializer.h"

#include <core_api/Log.h>
#include <util_gui/GUIUtils.h>

#include <QtCore/QFile>
#include <QtXml/qdom.h>

namespace GB2 {
using namespace Workflow;

static LogCategory log(ULOG_CAT_WD);

SaveWorkflowTask::SaveWorkflowTask(Schema* schema, const Metadata& meta)
: Task(tr("Saving schema to file"), TaskFlag_None), url(meta.url) 
{
    if (schema) {
        QDomDocument xmlDoc(SchemaSerializer::WORKFLOW_DOC);
        SchemaSerializer::schema2xml(*schema, xmlDoc);
        QDomElement elem = xmlDoc.documentElement();
        SchemaSerializer::saveIterations(schema->iterations, elem);
        SchemaSerializer::saveMeta(&meta, elem);
        rawData = xmlDoc.toByteArray();
    }
}

void SaveWorkflowTask::run() {
    log.details(tr("Saving schema to file: %1").arg(url));

    QFile f(url);
    if (f.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
        qint64 s = f.write(rawData);
        f.close();
        if (s != rawData.size()) {
            stateInfo.setError(Translations::errorWritingFile(url));
        }
    } else {
        stateInfo.setError(Translations::errorOpeningFileWrite(url));
    }
}

LoadWorkflowTask::LoadWorkflowTask(Schema* s, Workflow::Metadata* meta, const QString& url):
Task(tr("Loading schema"), TaskFlag_None),
    url(url), schema(s), meta(meta) {
        xml = new QDomDocument();
}

void LoadWorkflowTask::run() {
    log.details(tr("Loading schema from file: %1").arg(url));

    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        stateInfo.setError(  Translations::errorOpeningFileRead(url) );
        return;
    }
    QByteArray  xmlData = f.readAll();
    f.close();

    bool res = xml->setContent(xmlData);
    if (!res || xml->doctype().name() != SchemaSerializer::WORKFLOW_DOC) {
        stateInfo.setError( tr("Invalid content: %1").arg(url) );
        xml->clear();
    }
}

Task::ReportResult LoadWorkflowTask::report() {
    if (!stateInfo.hasErrors()) {
        stateInfo.setError(SchemaSerializer::xml2schema(xml->documentElement(), schema, remapping));
        if (stateInfo.hasErrors()) {
            schema->reset();
            if (meta) meta->reset();
        } else {
            if (meta) {
                SchemaSerializer::readMeta(meta, xml->documentElement());
                meta->url = url;
            }
            SchemaSerializer::readIterations(schema->iterations, xml->documentElement(), remapping);
        }
    }
    delete xml;

    return Task::ReportResult_Finished;
}

}//namespace
