/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CollocationsDialogController.h"

#include <gobjects/AnnotationSettings.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectUtils.h>

#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVAnnotationCreation.h>

#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>

#include <util_gui/GUIUtils.h>
#include <util_gui/CreateAnnotationWidgetController.h>
#include <util_gui/CreateAnnotationDialog.h>


#include <QtGui/QToolButton>
#include <QtGui/QMenu>
#include <QtCore/QFileInfo>

namespace GB2 {
class DNASequenceObject;

//TODO: support results separation on complement and direct strands

CollocationsDialogController::CollocationsDialogController(QStringList _names, ADVSequenceObjectContext* _ctx) 
: allNames(_names), ctx(_ctx)
{
    task = NULL;
    qSort(allNames);
    setupUi(this);
    
    QStringList list;
    list.append(tr("click_to_add_new_annotation"));
    QTreeWidgetItem* item = new QTreeWidgetItem(annotationsTree, list);
    plusButton = new QToolButton(annotationsTree);
    plusButton->setText("+");
    annotationsTree->addTopLevelItem(item);
    annotationsTree->setItemWidget(item, 1, plusButton);
    
    int w = annotationsTree->minimumWidth();
    annotationsTree->setColumnWidth(1, 20);
    annotationsTree->setColumnWidth(0, w - 30);
    annotationsTree->setUniformRowHeights(true);
    
    connect(plusButton,   SIGNAL(clicked()), SLOT(sl_plusClicked()));
    connect(searchButton, SIGNAL(clicked()), SLOT(sl_searchClicked()));
    connect(cancelButton, SIGNAL(clicked()), SLOT(sl_cancelClicked()));
    connect(clearResultsButton, SIGNAL(clicked()), SLOT(sl_clearClicked()));
    connect(saveResultsButton, SIGNAL(clicked()), SLOT(sl_saveClicked()));
    connect(resultsList, SIGNAL(itemActivated(QListWidgetItem*)), SLOT(sl_onResultActivated(QListWidgetItem*)));

    timer = new QTimer(this);
    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_onTaskFinished(Task*)));
    connect(timer, SIGNAL(timeout()), SLOT(sl_onTimer()));

    updateState();
}

void CollocationsDialogController::updateState() {
    bool hasActiveTask = task!=NULL;
    searchButton->setEnabled(!hasActiveTask);
    bool readyToSearch = usedNames.size() >= 2;
    searchButton->setEnabled(!hasActiveTask && readyToSearch);
    saveResultsButton->setEnabled(!hasActiveTask && resultsList->count() > 0);
    clearResultsButton->setEnabled(!hasActiveTask && resultsList->count() > 0);
    cancelButton->setText(hasActiveTask ? tr("stop") : tr("cancel"));
    updateStatus();
}

void CollocationsDialogController::updateStatus() {
    if (task!=NULL) {
        statusBar->setText(tr("searching__found_%1_items_progress_%2").arg(resultsList->count()).arg(task->getProgress()));
    } else if (resultsList->count() > 0) {
        statusBar->setText(tr("found_%1_items").arg(resultsList->count()));
    } else {
        statusBar->setText(searchButton->isEnabled() ? tr("ready") : tr("select_annotations"));
    }
}

void CollocationsDialogController::sl_plusClicked() {
    if (task != NULL) {
        return; 
    }
    QMenu m;
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(const QString& name, allNames) {
        if (usedNames.contains(name)) {
            continue;
        }            
        QColor c = asr->getAnnotationSettings(name)->color;
        QAction* a = m.addAction(GUIUtils::createSquareIcon(c, 10), name, this, SLOT(sl_addName()));
        assert(a->parent() == &m);
    }
    if (m.isEmpty()) {
        m.addAction(tr("no_more_annotations_left"));
    }
    m.exec(QCursor::pos());
}


void CollocationsDialogController::sl_addName() {
    QString name = ((QAction*)sender())->text();
    assert(allNames.contains(name));
    assert(!usedNames.contains(name));
    
    usedNames.insert(name);
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    QColor c = asr->getAnnotationSettings(name)->color;

    QTreeWidgetItem* item = new QTreeWidgetItem();
    item->setText(0, name);
    item->setIcon(0, GUIUtils::createSquareIcon(c, 10));
    QToolButton* minusButton = new QToolButton(annotationsTree);
    minusButton->setMinimumSize(plusButton->size());
    minusButton->setText("-");
    minusButton->setObjectName(name);
    annotationsTree->insertTopLevelItem(annotationsTree->topLevelItemCount()-1, item);
    annotationsTree->setItemWidget(item, 1, minusButton);    

    connect(minusButton, SIGNAL(clicked()), SLOT(sl_minusClicked()));
    updateState();
}


void CollocationsDialogController::sl_minusClicked() {
    if (task != NULL) {
        return; 
    }

    QObject* o = sender();
    QString name = o->objectName();
    
    assert(usedNames.contains(name));
    usedNames.remove(name);
    for (int i=0, n = annotationsTree->topLevelItemCount(); i<n; i++) {
        QTreeWidgetItem* item = annotationsTree->topLevelItem(i);
        if (item->text(0) == name) {
            annotationsTree->takeTopLevelItem(i);
            delete item;
            break;
        }
    }
    updateState();
}

/*
static LRegion getRange(const QList<AnnotationTableObject*>& aObjects) {
    LRegion res;
    foreach(AnnotationTableObject* ao, aObjects) {
        foreach(Annotation*a, ao->getAnnotations()) {
            foreach(const LRegion& r, a->getLocation()) {
                res = LRegion::join(res, r);
            }
        }
    }
    return res;
}
*/

void CollocationsDialogController::sl_searchClicked() {
    resultsList->clear();
    assert(usedNames.size() >= 2);
    CollocationsAlgorithmSettings cfg;
    cfg.distance = regionSpin->value();
    assert(task == NULL);
    const QList<AnnotationTableObject*>& aObjects = ctx->getAnnotationObjects().toList();
    cfg.searchRegion = LRegion(0, ctx->getSequenceLen());
    if (!wholeAnnotationsBox->isChecked()) {
        cfg.st = CollocationsAlgorithm::PartialSearch;
    }
    task = new CollocationSearchTask(aObjects, usedNames, cfg);
		

    AppContext::getTaskScheduler()->registerTopLevelTask(task);
    timer->start(400);
    updateState();
}

void CollocationsDialogController::sl_cancelClicked() {
    reject();
}

void CollocationsDialogController::sl_clearClicked() {
    resultsList->clear();
    updateState();
}

void CollocationsDialogController::sl_saveClicked() {
    assert(resultsList->count() > 0);

    CreateAnnotationModel m;
    m.sequenceObjectRef = ctx->getSequenceGObject();
    m.hideLocation = true;
    CreateAnnotationDialog d(this, m);
    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    QList<SharedAnnotationData> list;
    for (int i=0, n = resultsList->count(); i<n; ++i) {
        CDCResultItem* item = (CDCResultItem*)resultsList->item(i);
        SharedAnnotationData data = m.data;
        data->location.append(item->r);
        data->complement = false;
        data->aminoStrand = TriState_No;
        list.append(data);
    }

    ADVCreateAnnotationsTask* t = new ADVCreateAnnotationsTask(ctx->getAnnotatedDNAView(), m.getAnnotationObject(), m.groupName, list);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);

}


void CollocationsDialogController::reject() {
    if (task!=NULL) {
        task->cancel();
        return;
    }
    QDialog::reject();
}


void CollocationsDialogController::sl_onTimer() {
    importResults();
    updateState();
}

void CollocationsDialogController::sl_onTaskFinished(Task* t) {
    if (t != task || t->getState()!= Task::State_Finished) {
        return;
    }
    importResults();
    task = NULL;
    updateState();
    timer->stop();
}

void CollocationsDialogController::importResults() {
    if (task == NULL) {
        return;
    }

    QList<LRegion> newResults = task->popResults();
    
    foreach(const LRegion& r, newResults) {
        CDCResultItem* item = new CDCResultItem(r);
        bool inserted = false;
        for(int i=0, n = resultsList->count(); i<n; i++) {
            CDCResultItem* tmp = (CDCResultItem*)resultsList->item(i);
            assert(!tmp->r.contains(r) && !r.contains(tmp->r));
            if (tmp->r.startPos > r.startPos) {
                resultsList->insertItem(i, item);
                inserted = true;
            }
        }
        if (!inserted) {
            resultsList->addItem(item);
        }
    }
}


void CollocationsDialogController::sl_onResultActivated(QListWidgetItem * item) {
    assert(item!=NULL);
    CDCResultItem* ri = (CDCResultItem*)item;
    Q_UNUSED(ri);
    //todo: add to selection?
    //ctx->getPanView()->setVisibleRange(ri->r);
    //ctx->getDetView()->setCenterPos(ri->r.startPos);
}


CDCResultItem::CDCResultItem(const LRegion& _r) : r(_r) {
    setText(QString("[%1, %2]").arg(QString::number(r.startPos+1)).arg(r.endPos()));
}


//////////////////////////////////////////////////////////////////////////
// task
CollocationSearchTask::CollocationSearchTask(const QList<AnnotationTableObject*> &table, const QSet<QString>& names,
                                             const CollocationsAlgorithmSettings& cfg) 
: Task(tr("collocation_search_task"), TaskFlag_None), cfg(cfg)
{
    assert(cfg.distance >= 0);
    assert(!names.isEmpty());
    foreach(const QString& name, names) {
        getItem(name);
    }
    foreach(AnnotationTableObject* ao, table) {
        foreach(Annotation* a, ao->getAnnotations()) {
            const QString& name = a->getAnnotationName();
            if (names.contains(name)) {
                CollocationsAlgorithmItem& item = getItem(name);
                foreach(const LRegion& r, a->getLocation()) {
                    if (cfg.searchRegion.intersects(r)) {
                        item.regions.append(r);
                    }
                }
            }
        }
    }
}

CollocationSearchTask::CollocationSearchTask(const QList<SharedAnnotationData> &table, const QSet<QString>& names, 
                      const CollocationsAlgorithmSettings& cfg) 
: Task(tr("collocation_search_task"), TaskFlag_None), cfg(cfg)
{
    assert(cfg.distance >= 0);
    assert(!names.isEmpty());
    foreach(const QString& name, names) {
        getItem(name);
    }
    foreach(SharedAnnotationData a, table) {
        const QString& name = a->name;
        if (names.contains(name)) {
            CollocationsAlgorithmItem& item = getItem(name);
            foreach(const LRegion& r, a->location) {
                if (cfg.searchRegion.intersects(r)) {
                    item.regions.append(r);
                }
            }
        }
    }
}

CollocationsAlgorithmItem& CollocationSearchTask::getItem(const QString& name) {
    if (!items.contains(name)) {
        items[name] = CollocationsAlgorithmItem(name);
    }
    return items[name];
}

void CollocationSearchTask::run() {
	CollocationsAlgorithm::find(items.values(), stateInfo, this, cfg);
}


void CollocationSearchTask::onResult(const LRegion& r) {
    lock.lock();
    results.append(r);
    lock.unlock();
}

QList<LRegion> CollocationSearchTask::popResults() {
    lock.lock();
    QList<LRegion> tmp = results;
    results.clear();
    lock.unlock();
    return tmp;
}

}//namespace
