/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QApplication>
#include <QtScript/QScriptEngine>
#include <QtScript/QScriptContext>
#include <QtScript/QScriptValue>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include "ScriptEngineContext.h"

/* TRANSLATOR GB2::ScriptHttpAnnotatorContext */

namespace GB2
{

const static char * HTTP_SCRIPT_ALPHABET_AMINO             = "amino";
const static char * HTTP_SCRIPT_ALPHABET_NUCLEO            = "nucleic";
const static char * HTTP_SCRIPT_STRAND_BOTH                = "both";
const static char * HTTP_SCRIPT_STRAND_SINGLE              = "single";
const static char * HTTP_SCRIPT_PROPERTY_MAX_QUERY_LEN     = "max_query_len";
const static char * HTTP_SCRIPT_PROPERTY_ALPHABET          = "alphabet";
const static char * HTTP_SCRIPT_PROPERTY_STRAND            = "strand";
const static char * HTTP_SCRIPT_PROPERTY_QUERY             = "query";
const static char * HTTP_SCRIPT_PROPERTY_MAXRESLEN         = "max_res_len";
const static char * HTTP_SCRIPT_PROPERTY_MINRESLEN         = "min_res_len";
const static char * HTTP_SCRIPT_PROPERTY_LOG               = "log";
const static char * HTTP_SCRIPT_PROPERTY_TASK_STATE_INFO   = "stateInfo";
const static char * HTTP_SCRIPT_PROPERTY_GET               = "get";
const static char * HTTP_SCRIPT_PROPERTY_CUSTOM_SETTINGS   = "configure"; //should be a function
const static char * HTTP_SCRIPT_PROPERTY_ANNOTATIONS       = "annotations";
const static char * HTTP_SCRIPT_ANNOTATION_DATA            = "AnnotationData";

const QString ScriptTypes::SCRIPT_TYPE_HTTP_ANNOTATOR("http_annotator");

QScriptValue url_get( QScriptContext * ctx, QScriptEngine * engine )
{
    assert( 1 == ctx->argumentCount() );
    QScriptValue url = ctx->argument(0);
    if( !url.isString() ) {
        assert( false );
        return ctx->throwError( QScriptContext::ReferenceError, ScriptHttpAnnotatorContext::tr("url_is_bad_object") );
    }

    IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::HTTP_FILE );
    IOAdapter * io = iof->createIOAdapter();

    if( !io->open( url.toString(), IOAdapterMode_Read ) ) {
        return ctx->throwError( QScriptContext::UnknownError, ScriptHttpAnnotatorContext::tr("cannot_open_ioadapter") );
    }
    
    int offs = 0;
    int read = 0;
    const static int CHUNK_SIZE = 1024;
    QByteArray response( CHUNK_SIZE, 0 );
    do {
        read = io->readBlock( response.data() + offs, CHUNK_SIZE );
        offs += read;
        response.resize( offs + CHUNK_SIZE );
    } while( read == CHUNK_SIZE );

    if( read < 0 ) {
        return ctx->throwError( QScriptContext::UnknownError, ScriptHttpAnnotatorContext::tr("unknown_io_error") );
    }
    response.resize( offs );
    return engine->newVariant( QVariant(QString(response)) );
}

// ================== Annotation Data =======================

static QScriptValue annotation_data_ctor( QScriptContext * ctx, QScriptEngine * engine ) {
    Q_UNUSED(ctx);
    QVariant var = QVariant::fromValue( AnnotationData() );
    return engine->newVariant( var );
}


AnnotationDataPrototype::AnnotationDataPrototype( QObject * o ) : QObject(o){

}

QString AnnotationDataPrototype::name() const {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        return data->name;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return QString();
}

void AnnotationDataPrototype::setName( const QString & name ) {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        data->name = name;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

bool AnnotationDataPrototype::complement() const {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        return data->complement;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return false;
}

void AnnotationDataPrototype::setComplement( bool c ) {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        data->complement = c;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

void AnnotationDataPrototype::setAlpha( bool amino ) {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        data->aminoStrand = amino ? TriState_Yes : TriState_No;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

void AnnotationDataPrototype::addLocation( int start, int len ) {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        data->location.push_back( LRegion(start, len) );
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

void AnnotationDataPrototype::addQualifier( const QString & name, const QString & val ) {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        data->qualifiers.push_back( Qualifier(name, val) );
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

QString AnnotationDataPrototype::toString() const {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        
        QString res( "Annotation Data:" );
        res += data->name;

        res += "\namino: ";
        switch( data->aminoStrand ) {
            case TriState_Yes:
                res += "yes\n";
                break;
            case TriState_No:
                res += "no\n";
                break;
            default:
                res += "unknown\n";
        }

        res += "complement: ";
        res += data->complement ? "yes\n" : "no\n";

        res += "location:\n";
        foreach( LRegion l, data->location ) {
            res += QString::number( l.startPos ) + " " + QString::number( l.len ) + "\n";
        }

        res += "qualifiers:\n";
        foreach( Qualifier q, data->qualifiers ) {
            res += q.getQualifierName() + " : " + q.getQualifierValue() + "\n";
        }

        return res;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return QString();
}

int AnnotationDataPrototype::getSummaryLen() const {
    AnnotationData * data = qscriptvalue_cast<AnnotationData *>( thisObject() );
    if( data ) {
        int res = 0;
        foreach( LRegion r, data->location ) {
            res += r.len;
        }
        return res;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return false;     
}

// ====================== LogCategoryPrototype =======================

LogCategoryPrototype::LogCategoryPrototype( QObject * o /* = 0 */ ) : QObject(o) {

}

void LogCategoryPrototype::info( QString msg ) {
    LogCategory * lc = qscriptvalue_cast<LogCategory*>( thisObject() );
    if( lc ) {
        lc->info( msg );
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

void LogCategoryPrototype::error( QString msg ) {
    LogCategory * lc = qscriptvalue_cast<LogCategory*>( thisObject() );
    if( lc ) {
        lc->error( msg );
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

void LogCategoryPrototype::debug( QString msg ) {
    LogCategory * lc = qscriptvalue_cast<LogCategory*>( thisObject() );
    if( lc ) {
        lc->details( msg );
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

// ===================== TaskStateInfoPrototype =====================

TaskStateInfoPrototype::TaskStateInfoPrototype( QObject * o /* = 0 */ ) :
QObject(o) {
}

int TaskStateInfoPrototype::progress() const {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo*>( thisObject() );
    if( tsi ) {
        return tsi->progress;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return -1;
}

void TaskStateInfoPrototype::setProgress( int p ) {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo*>( thisObject() );
    if( tsi ) {
        tsi->progress = p;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

bool TaskStateInfoPrototype::cancelFlag() const {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        return tsi->cancelFlag;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return false;
}

void TaskStateInfoPrototype::setCancelFlag( bool cf ) {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        tsi->cancelFlag = cf;
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

QString TaskStateInfoPrototype::stateDesc() const {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        return tsi->getStateDesc();
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return QString();
}

void TaskStateInfoPrototype::setStateDesc( QString sd ) {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        tsi->setStateDesc(sd);
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

QString TaskStateInfoPrototype::error() const {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        return tsi->getError();
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
    return QString();
}

void TaskStateInfoPrototype::setError( QString sd ) {
    TaskStateInfo * tsi = qscriptvalue_cast<TaskStateInfo *>( thisObject() );
    if( tsi ) {
        tsi->setError(sd);
    } else {
        assert( context() );
        context()->throwError( QScriptContext::TypeError, tr("bad_cast") );
    }
}

// ===================== HttpAnnotatorContext =====================

void ScriptHttpAnnotatorContext::setDefaultProperties( QScriptEngine * engine ) {
    assert( engine );

    Script::importExtensions( engine );

    QScriptValue val_url_get = engine->newFunction( url_get, 1 );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_GET, val_url_get );

    AnnotationDataPrototype * adProto = new AnnotationDataPrototype( engine );
    QScriptValue adProto_val = engine->newQObject( adProto );
    engine->setDefaultPrototype( qMetaTypeId<AnnotationData>(), adProto_val );
    QScriptValue ad_ctor = engine->newFunction( annotation_data_ctor, adProto_val );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_ANNOTATION_DATA, ad_ctor );

    LogCategoryPrototype * logProto = new LogCategoryPrototype( engine );
    QScriptValue logProto_val = engine->newQObject( logProto );
    engine->setDefaultPrototype( qMetaTypeId<LogCategory*>(), logProto_val );

    TaskStateInfoPrototype * tsiProto = new TaskStateInfoPrototype( engine );
    QScriptValue tsiProto_val = engine->newQObject( tsiProto );
    engine->setDefaultPrototype( qMetaTypeId<TaskStateInfo*>(), tsiProto_val );
}

int ScriptHttpAnnotatorContext::getMaxQueryLen( QScriptEngine * engine ) {
    assert( engine );
    QScriptValue val = Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_MAX_QUERY_LEN );
    return val.toInt32();
}
HttpAnnotatorAlphabet ScriptHttpAnnotatorContext::getAlphabet( QScriptEngine * engine, QString * alpha_str  ) {
    assert( engine );
    QScriptValue val = Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_ALPHABET );
    QString str = val.toString();
    *alpha_str = str;
    if( HTTP_SCRIPT_ALPHABET_NUCLEO == str ) {
        return HttpAnnotatorAlphabet_Nucleo;
    } else if( HTTP_SCRIPT_ALPHABET_AMINO  == str ) {
        return HttpAnnotatorAlphabet_Amino;
    }
    return HttpAnnotatorAlphabet_Any;
}

HttpAnnotatorStrand ScriptHttpAnnotatorContext::getStrand( QScriptEngine * engine ) {
    assert( engine );
    QScriptValue val = Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_STRAND );
    if( HTTP_SCRIPT_STRAND_BOTH == val.toString() ) {
        return HttpAnnotatorStrand_Both;
    } else if( HTTP_SCRIPT_STRAND_SINGLE == val.toString() ) {
        return HttpAnnotatorStrand_Single;
    }
    return HttpAnnotatorStrand_Unknown;
}

bool ScriptHttpAnnotatorContext::hasCustomSettings( QScriptEngine * engine ) {
    assert( engine );
    QScriptValue val = Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_CUSTOM_SETTINGS );
    return val.isFunction();
}

void ScriptHttpAnnotatorContext::setQuery( QScriptEngine * engine, const QString & q ) {
    assert( engine );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_QUERY, QScriptValue(engine, q) );
}

void ScriptHttpAnnotatorContext::setMaxResLen( QScriptEngine * engine, int maxrl ) {
    assert( engine );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_MAXRESLEN, QScriptValue(engine, maxrl) );
}

void ScriptHttpAnnotatorContext::setMinResLen( QScriptEngine * engine, int minrl ) {
    assert( engine );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_MINRESLEN, QScriptValue(engine, minrl) );
}

void ScriptHttpAnnotatorContext::setLog( QScriptEngine * engine, LogCategory * lc ) {
    assert( engine );
    QScriptValue val = engine->newVariant( QVariant::fromValue(lc) );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_LOG, val );
}

void ScriptHttpAnnotatorContext::setTaskStateInfo( QScriptEngine * engine, TaskStateInfo * tsi ) {
    assert( engine );
    QScriptValue val = engine->newVariant( QVariant::fromValue(tsi) );
    Script::getGlobal(engine).setProperty( HTTP_SCRIPT_PROPERTY_TASK_STATE_INFO, val );
}

#define MAX_Q_VLEN 4096
#define MAX_Q_NLEN 15
static void fixAnnotationData(AnnotationData* ad) {
    //fix qualifier names and values len
    QVector<Qualifier> newQs;
    foreach(const Qualifier& q, ad->qualifiers) {
        QString name = q.getQualifierName().trimmed();
        if (name.length() > MAX_Q_NLEN) {
            name = name.left(MAX_Q_NLEN).trimmed();
        }
        QString val = q.getQualifierValue().trimmed();
        if (val.length() > MAX_Q_VLEN) {
            val = val.left(MAX_Q_VLEN).trimmed();
        }
        Qualifier newQ(name, val);
        newQs.append(newQ);
    }
    ad->qualifiers = newQs;
}


QList<SharedAnnotationData> ScriptHttpAnnotatorContext::getAnnotations( QScriptEngine * engine ) {
    QScriptValue val = Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_ANNOTATIONS );
    assert( engine );
    QList<SharedAnnotationData> res;
    if( !val.isArray() ) {
        return res;
    }

    QVariantList m = engine->fromScriptValue<QVariantList>( val );
    foreach( QVariant v, m) {
        AnnotationData * d = new AnnotationData();
        *d = qVariantValue<AnnotationData>(v);
        fixAnnotationData(d);
        res.append( SharedAnnotationData(d) );
    }
    return res;
}

void ScriptHttpAnnotatorContext::callCustomSettings( QScriptEngine * engine ) {
    assert( engine );
    Script::getGlobal(engine).property( HTTP_SCRIPT_PROPERTY_CUSTOM_SETTINGS ).call();
}

} //namespace
