/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMSearchTask.h"
#include "TaskLocalStorage.h"
#include "HMMIO.h"

#include <core_api/AppContext.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/DNATranslation.h>

#include <hmmer2/funcs.h>
#include <util_tasks/SequenceWalkerTask.h>

namespace GB2 {

HMMSearchTask::HMMSearchTask(plan7_s* _hmm, const DNASequence& _seq, const UHMMSearchSettings& s)
: Task("", TaskFlag_NoRun), 
  hmm(_hmm), seq(_seq), settings(s), complTrans(NULL), aminoTrans(NULL) 
{
    setTaskName(tr("HMM search with '%1'").arg(hmm->name));
}

void HMMSearchTask::prepare() {
    if (!checkAlphabets(hmm->atype, seq.alphabet, complTrans, aminoTrans)) {
        return;
    }
    SequenceWalkerConfig config;
    config.seq = seq.seq.data();
    config.seqSize = seq.seq.size();
    config.complTrans = complTrans;
    config.aminoTrans = aminoTrans;
    config.overlapSize = 2 * hmm->M;
    config.chunkSize = qMax(config.overlapSize, settings.searchChunkSize);
    if (settings.extraLen == -1) {
        config.lastChunkExtraLen = config.chunkSize / 2;
    } else {
        config.lastChunkExtraLen = settings.extraLen;
    }
    
    config.nThreads = MAX_PARALLEL_SUBTASKS_AUTO;
    
    addSubTask(new SequenceWalkerTask(config, this, tr("parallel_hmm_search_task")));
}


void HMMSearchTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& si) 
{
    const char* localSeq = t->getRegionSequence();
    int localSeqSize = t->getRegionSequenceLen();
    bool wasCompl = t->isDNAComplemented();
    bool wasAmino = t->isAminoTranslated();
    LRegion globalReg = t->getGlobalRegion();

    //set TLS data
    TaskLocalData::createHMMContext(t->getTaskId(), true);

    QList<UHMMSearchResult> sresults;
    try {
        sresults = UHMMSearch::search(hmm, localSeq, localSeqSize, settings, si);
    } catch (HMMException e) {
        stateInfo.setError(  e.error );
    }
    if (si.hasErrors()) {
        stateInfo.setError(  si.getError() );
    }
    if (sresults.isEmpty()  || stateInfo.cancelFlag || stateInfo.hasErrors()) {
        TaskLocalData::freeHMMContext(t->getTaskId());
        return;
    }
    
    //convert all UHMMSearchResults into HMMSearchTaskResult
    QMutexLocker locker( &lock );
    int halfOverlap = hmm->M;
    foreach(const UHMMSearchResult& sr, sresults) {
        HMMSearchTaskResult r;
        r.evalue = sr.evalue;
        r.score = sr.score;
        r.onCompl = wasCompl;
        r.onAmino = wasAmino;
        int resLen   = wasAmino ? sr.r.len * 3 : sr.r.len;
		int resStart = wasAmino ? sr.r.startPos * 3 : sr.r.startPos;
        if (wasCompl) {
            resStart = globalReg.len - resStart - resLen;
        }
        r.r.startPos = globalReg.startPos + resStart;
        r.r.len = resLen;
        if (t->intersectsWithOverlaps(r.r)) {
            //don't add to overlaps if it must be found in 2 regions
            bool add = true;
            if (!r.onCompl && t->hasRightOverlap()) { //check if will be found in a next chunk
                LRegion nextChunkRegion(globalReg.endPos() - halfOverlap, halfOverlap);
                add = !nextChunkRegion.contains(r.r);
            } else if (r.onCompl && t->hasLeftOverlap()) { //check if will found in a prev chunk
                LRegion prevChunkRegion(globalReg.startPos, halfOverlap);
                add = !prevChunkRegion.contains(r.r);
            }
            if (add) {
                r.borderResult = (t->hasLeftOverlap() && r.r.startPos == globalReg.startPos)
                    || (t->hasRightOverlap() && r.r.endPos() == globalReg.endPos());
                overlaps.append(r);
            }
        } else {
            results.append(r);
        }
    }

    TaskLocalData::freeHMMContext(t->getTaskId());
}

static bool HMMSearchResult_LessThan(const HMMSearchTaskResult& r1, const HMMSearchTaskResult& r2) {
    if (r1.evalue == r2.evalue) {
        if (r1.r == r2.r) {
            if (r1.onCompl == r2.onCompl) {
                return &r1 < &r2;
            }
            return r2.onCompl;
        } 
        return r1.r < r2.r;
    }
    return r1.evalue < r2.evalue;
}

Task::ReportResult HMMSearchTask::report() {
    if (hasErrors()) {
        return ReportResult_Finished;
    }

    //postprocess overlaps
    int maxCommonLen = hmm->M / 2; //if 2 results have common part of 'maxCommonLen' or greater -> select best one
    for(int i=0; i < overlaps.count(); i++){
        HMMSearchTaskResult& r1 = overlaps[i];
        if (r1.filtered) {
            continue;
        }
        for(int j=i+1; j < overlaps.count(); j++){
            HMMSearchTaskResult& r2 = overlaps[j];
            if (r2.filtered) {
                continue;
            }
            if (r1.onCompl != r2.onCompl) { //check both regions are on the same strand
                continue;
            }
            if (r1.onAmino) { //check both regions have the same amino frame
                int s1 = r1.onCompl ? r1.r.endPos() % 3 : r1.r.startPos % 3;
                int s2 = r2.onCompl ? r2.r.endPos() % 3 : r2.r.startPos % 3;
                if (s1 != s2) {
                    continue;
                }
            }
            if (r1.r.contains(r2.r)) {
                r2.filtered = true;
            } else if (r2.r.contains(r1.r)) {
                r1.filtered = true;
                break;
            } else if (r1.r.intersect(r2.r).len >= maxCommonLen) {
                bool useR1 = r2.score <= r1.score;
                if (r1.score == r2.score && r1.evalue == r2.evalue && r1.borderResult && !r2.borderResult) {
                    useR1 = false;
                }
                if (useR1) {
                    r2.filtered = true;
                } else {
                    r1.filtered = true;
                    break;
                }
            }
        }
    }

    foreach(const HMMSearchTaskResult& r, overlaps) {
        if (!r.filtered) {
            results.append(r);
        }
    }
   
    // sort results by E-value
    qSort(results.begin(), results.end(), HMMSearchResult_LessThan);
    return ReportResult_Finished;
}

QList<SharedAnnotationData> HMMSearchTask::getResultsAsAnnotations(const QString& name) const {
    QList<SharedAnnotationData>  annotations;
    foreach(const HMMSearchTaskResult& hmmRes, results) {
        AnnotationData* a = new AnnotationData();
        a->name = name;
        a->complement = hmmRes.onCompl;
        a->aminoStrand = hmmRes.onAmino ? TriState_Yes :TriState_No;
        a->location.append(hmmRes.r);

        QString str; /*add zeros at begin of evalue exponent part, so exponent part must contains 3 numbers*/
        str.sprintf("%.2g", ((double) hmmRes.evalue));
        QRegExp rx("\\+|\\-.+");
        int pos = rx.indexIn(str,0);
        if(pos!=-1){
            str.insert(pos+1,"0");
        }
        QString info = hmm->name;
        if (hmm->flags & PLAN7_ACC) {
            info += QString().sprintf("\nAccession number in PFAM : %s", hmm->acc);
        }
        if (hmm->flags & PLAN7_DESC) {
            info += QString().sprintf("\n%s", hmm->desc);
        }
        if (!info.isEmpty()) {
            a->qualifiers.append(Qualifier("HMM-model", info));
        }
        //a->qualifiers.append(Qualifier("E-value", QString().sprintf("%.2lg", ((double) hmmRes.evalue))));
        a->qualifiers.append(Qualifier("E-value", str));
        a->qualifiers.append(Qualifier("Score", QString().sprintf("%.1f", hmmRes.score)));
        annotations.append(SharedAnnotationData(a));
    }
    return annotations;
}

bool HMMSearchTask::checkAlphabets(int hmmAlType, DNAAlphabet* seqAl, DNATranslation*& complTrans, DNATranslation*& aminoTrans) 
{
    assert(stateInfo.getError().isEmpty());
    DNAAlphabetType hmmAl = HMMIO::convertHMMAlphabet(hmmAlType);
    if (hmmAl == DNAAlphabet_RAW) {
        stateInfo.setError(  tr("invalid_hmm_alphabet_type") );
        return false;
    }
    if (seqAl->isRaw()) {
        stateInfo.setError(  tr("invalid_sequence_alphabet_type") );
        return false;
    }

    complTrans = NULL;
    aminoTrans = NULL;
    if (seqAl->isNucleic()) {
        DNATranslationRegistry* tr = AppContext::getDNATranslationRegistry();
        QList<DNATranslation*> complTs = tr->lookupTranslation(seqAl, DNATranslationType_NUCL_2_COMPLNUCL);
        if (!complTs.empty()) {
            complTrans = complTs.first();
        }
        if (hmmAl == DNAAlphabet_AMINO) {
            QList<DNATranslation*> aminoTs = tr->lookupTranslation(seqAl, DNATranslationType_NUCL_2_AMINO);
            if (!aminoTs.empty()) {
                aminoTrans = aminoTs.first();
            }
        }
    } else {
        assert(seqAl->isAmino());
    }

    // check the result;
    if (hmmAl == DNAAlphabet_AMINO) {
        if (seqAl->isAmino()) {
            assert(complTrans == NULL && aminoTrans == NULL);
        } else {
            if (aminoTrans == NULL) {
                stateInfo.setError(  tr("can_t_find_amino") );
                return false;
            }
        }
    }

    return true;
}

}//endif
