/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMSearchWorker.h"
#include "HMMIOWorker.h"
#include "HMMSearchTask.h"

#include <hmmer2/funcs.h>
 
#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>

#include <datatype/AnnotationData.h>
#include <datatype/DNASequence.h>
#include <core_api/DNATranslation.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <util_tasks/FailTask.h>
#include <util_tasks/MultiTask.h>
#include <util_tasks/TaskSignalMapper.h>

#include <QtGui/QApplication>
/* TRANSLATOR GB2::LocalWorkflow::HMMSearchWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString HMM_PORT("hmm");
static const QString SEQ_PORT("seq");
static const QString OUT_PORT("out");

static const QString NAME_ATTR("1name");
static const QString NSEQ_ATTR("3nseq");
static const QString DOM_E_ATTR("4domE");
static const QString DOM_T_ATTR("5domT");

const QString HMMSearchWorkerFactory::ACTOR("uhmmer.search");

static LogCategory log(ULOG_CAT_WD);

void HMMSearchWorkerFactory::init() {
    
    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor hd(HMM_PORT, HMMSearchWorker::tr("HMM profile"), HMMSearchWorker::tr("HMM profile(s) to search with."));
        Descriptor sd(SEQ_PORT, HMMSearchWorker::tr("Input sequence"), HMMSearchWorker::tr("An input sequence (nucleotide or protein) to search in."));
        Descriptor od(OUT_PORT, HMMSearchWorker::tr("HMM annotations"), HMMSearchWorker::tr("Annotations marking found similar sequence regions."));

        p << new PortDescriptor(hd, HMMLib::HMM_PROFILE_TYPE(), true /*input*/, false, BusPort::BLIND_INPUT);
        p << new PortDescriptor(sd, BioDataTypes::DNA_SEQUENCE_TYPE(), true /*input*/);
        p << new PortDescriptor(od, BioDataTypes::ANNOTATION_TABLE_TYPE(), false /*input*/, true);
    }

    {
        Descriptor nd(NAME_ATTR, HMMSearchWorker::tr("Result annotation"), HMMSearchWorker::tr("A name of the result annotations."));
        Descriptor nsd(NSEQ_ATTR, HMMSearchWorker::tr("Number of seqs"), QApplication::translate("HMMSearchDialog", "e_value_as_nsec_tip", 0, QApplication::UnicodeUTF8));
        Descriptor ded(DOM_E_ATTR, HMMSearchWorker::tr("Filter by high E-value"), QApplication::translate("HMMSearchDialog", "results_evalue_cutoff_tip", 0, QApplication::UnicodeUTF8));
        Descriptor dtd(DOM_T_ATTR, HMMSearchWorker::tr("Filter by low score"), QApplication::translate("HMMSearchDialog", "results_score_cutoff_tip", 0, QApplication::UnicodeUTF8));

        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, QVariant("hmm_signal"));
        a << new Attribute(nsd, CoreDataTypes::NUM_TYPE(), false, QVariant(1));
        a << new Attribute(ded, CoreDataTypes::NUM_TYPE(), false, QVariant(-1));
        a << new Attribute(dtd, CoreDataTypes::NUM_TYPE(), false, QVariant((double)-1e+09));
    }
 
    Descriptor desc(HMMSearchWorkerFactory::ACTOR, HMMSearchWorker::tr("HMM search"), 
        HMMSearchWorker::tr("Searches each input sequence for significantly similar sequence matches to all specified HMM profiles."
        " In case several profiles were supplied, searches with all profiles one by one and outputs united set of annotations for each sequence"));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    
    {
        QVariantMap eMap; eMap["prefix"]= ("1e"); eMap["minimum"] = (-99); eMap["maximum"] = (0);
        delegates[DOM_E_ATTR] = new SpinBoxDelegate(eMap);
    }
    {
        QVariantMap nMap; nMap["maximum"] = (INT_MAX);
        delegates[NSEQ_ATTR] = new SpinBoxDelegate(nMap);
    }
    {
        QVariantMap tMap; tMap["decimals"]= (1); tMap["minimum"] = (-1e+09); tMap["maximum"] = (1e+09);
        tMap["singleStep"] = (0.1);
        delegates[DOM_T_ATTR] = new DoubleSpinBoxDelegate(tMap);
    }
 
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":/hmm2/images/hmmer_16.png");
    proto->setPrompter(new HMMSearchPrompter());
    WorkflowEnv::getProtoRegistry()->registerProto(HMMLib::HMM_CATEGORY(), proto);
 
    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new HMMSearchWorkerFactory());
}

static bool isDefaultCfg(PrompterBaseImpl* actor) {
    return 1 == actor->getParameter(NSEQ_ATTR).toInt()
        && -1 == actor->getParameter(DOM_E_ATTR).toInt()
        && double(-1e+09) == actor->getParameter(DOM_T_ATTR).toDouble();
}

QString HMMSearchPrompter::composeRichDoc() {
    Actor* hmmProducer = qobject_cast<BusPort*>(target->getPort(HMM_PORT))->getProducer(HMM_PORT);
    Actor* seqProducer = qobject_cast<BusPort*>(target->getPort(SEQ_PORT))->getProducer(SEQ_PORT);

    QString seqName = seqProducer ? tr("For each sequence from <u>%1</u>,").arg(seqProducer->getLabel()) : "";
    QString hmmName = hmmProducer ? tr("using all profiles provided by <u>%1</u>,").arg(hmmProducer->getLabel()) : "";

    QString resultName = getRequiredParam(NAME_ATTR);
    QString cfg = isDefaultCfg(this) ? tr("Use <u>default</u> settings.") : tr("Use <u>custom</u> settings.");

    QString doc = tr("%1 search HMM signals %2. %3"
        "<br>Output the list of found regions annotated as <u>%4</u>.")
        .arg(seqName)
        .arg(hmmName)
        .arg(cfg)
        .arg(resultName);

    return doc;
}


HMMSearchWorker::HMMSearchWorker(Actor* a) : BaseWorker(a, false), hmmPort(NULL), seqPort(NULL), output(NULL) {}
 
void HMMSearchWorker::init() {
    hmmPort = ports.value(HMM_PORT);
    seqPort = ports.value(SEQ_PORT);
    output = ports.value(OUT_PORT);
    seqPort->addComplement(output);
    output->addComplement(seqPort);

    cfg.domE = pow(10,(float) actor->getParameter(DOM_E_ATTR)->value.toInt());
    cfg.domT = (float)actor->getParameter(DOM_T_ATTR)->value.toDouble();
    cfg.eValueNSeqs = actor->getParameter(NSEQ_ATTR)->value.toInt();
    resultName = actor->getParameter(NAME_ATTR)->value.toString();
}
 
bool HMMSearchWorker::isReady() {
    return hmmPort->hasMessage() || ((!hmms.isEmpty() && hmmPort->isEnded() ) && seqPort->hasMessage());
}
 
Task* HMMSearchWorker::tick() {
    while (hmmPort->hasMessage()) {
        hmms << hmmPort->get().getData().value<plan7_s*>();
    }
    if (!hmmPort->isEnded() || hmms.isEmpty() || !seqPort->hasMessage()) {
        return NULL;
    }
    DNASequence dnaSequence = seqPort->get().getData().value<DNASequence>();
    if (dnaSequence.alphabet->getType() != DNAAlphabet_RAW) {
        QList<Task*> subtasks;
        foreach(plan7_s* hmm, hmms) {
            subtasks << new HMMSearchTask(hmm, dnaSequence, cfg);
        }
        Task* searchTask = new MultiTask(tr("Search HMM signals in %1").arg(dnaSequence.getName()), subtasks);
        connect(new TaskSignalMapper(searchTask), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
        return searchTask;
    }
    QString err = tr("Bad sequence supplied to input: %1").arg(dnaSequence.getName());
    if (failFast) {
        return new FailTask(err);
    } else {
        log.error(err);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()));
        if (seqPort->isEnded()) {
            output->setEnded();
        }
        return NULL;
    }
}
 
void HMMSearchWorker::sl_taskFinished(Task* t) {
    if (output) {
        QList<SharedAnnotationData> list;
        foreach(Task* sub, t->getSubtasks()) {
            HMMSearchTask* hst = qobject_cast<HMMSearchTask*>(sub);
            list += hst->getResultsAsAnnotations(resultName);
        }
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(list);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (seqPort->isEnded()) {
            output->setEnded();
        }
        log.info(tr("Found %1 HMM signals").arg(list.size()));
    }
}
 
bool HMMSearchWorker::isDone() {
    return seqPort->isEnded();
}
 
} //namespace LocalWorkflow
} //namespace GB2
