/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <memory>

#include <QtCore/QList>
#include <QtCore/QtAlgorithms>

#include <util_gui/GUIUtils.h>
#include <core_api/Task.h>
#include <util_text/TextUtils.h>

#include <gobject/uHMMObject.h>
#include "uHMMFormat.h"

using namespace GB2;

const QString UHMMFormat::READ_FAILED   = UHMMFormat::tr( "reading_file_failed!" );
const QString UHMMFormat::WRITE_FAILED  = UHMMFormat::tr( "write_to_file_failed" );

const int BUF_SZ        = 1024;
const int BAD_READ      = -1;
const int EMPTY_READ    = 0;
const char TERM_SYM     = '\0';

enum HMMMERHeaderTags {
    BAD_TAG = -1,
    NAME,
    ACC,
    DESC,
    LENG,
    ALPH,
    RF,
    CS,
    MAP,
    DATE,
    COM,
    NSEQ,
    EFFN,
    CKSUM,
    STATS,
    GA,
    TC,
    NC,
    HMM
}; // HMMMERHeaderTags

static QMap< QByteArray, HMMMERHeaderTags > getHeaderTagsMap() {
    QMap< QByteArray, HMMMERHeaderTags > ret;
    ret["NAME"]     = NAME;
    ret["ACC"]      = ACC;
    ret["DESC"]     = DESC;
    ret["LENG"]     = LENG;
    ret["ALPH"]     = ALPH;
    ret["RF"]       = RF;
    ret["CS"]       = CS;
    ret["MAP"]      = MAP;
    ret["DATE"]     = DATE;
    ret["COM"]      = COM;
    ret["NSEQ"]     = NSEQ;
    ret["EFFN"]     = EFFN;
    ret["CKSUM"]    = CKSUM;
    ret["STATS"]    = STATS;
    ret["GA"]       = GA;
    ret["TC"]       = TC;
    ret["NC"]       = NC;
    ret["HMM"]      = HMM;
    return ret;
}

static void getTagValue( const QByteArray& ln, QByteArray& tag, QByteArray& val ) {
    QString line( ln.trimmed() );
    QStringList words = line.split( QRegExp( "\\s+" ), QString::SkipEmptyParts );

    if( 1 >= words.size() ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "bad_line_in_header_section:%1" ).arg( QString( ln ) ) );
    } else {
        tag = words.first().toAscii();
        val = ln.mid( tag.size() ).trimmed();
        assert( !tag.isEmpty() );
        if( val.isEmpty() ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "no_value_found_in_line:%1" ).arg( QString(ln) ) );
        }
    }
}

static void checkReadThrowException( int readBytes ) {
    if( BAD_READ == readBytes ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::READ_FAILED );
    }
}

static void readLine( IOAdapter* io, QByteArray& to, QStringList* tokens = NULL ) {
    assert( NULL != io );
    to.clear();
    QByteArray buf( BUF_SZ, TERM_SYM );
    bool there = false;
    while( !there ) {
        int bytes = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Include, &there );
        checkReadThrowException( bytes );
        if( EMPTY_READ == bytes ) {
            break;
        }
        to.append( QByteArray( buf.data(), bytes ) );
    }
    to = to.trimmed();
    
    if( NULL != tokens ) {
        *tokens = QString( to ).split( QRegExp( "\\s+" ), QString::SkipEmptyParts );
    }
}

static bool checkHeader( const char* data, int sz ) {
    assert( NULL != data && 0 <= sz );
    if( UHMMFormat::HMMER_HEADER.size() >= sz ) {
        return false;
    }
    return QByteArray::fromRawData( data, sz ).trimmed().startsWith( UHMMFormat::HMMER_HEADER );
}

static void setInteger( int& num, const QByteArray& numStr ) {
    bool ok = false;
    int ret = numStr.toInt( &ok );
    if( !ok ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "cannot_parse_integer_number_from_string:%1" ).arg( QString(numStr) ) );
    } else {
        num = ret;
    }
}

static void setFloat( float& num, const QByteArray& numStr ) {
    bool ok = false;
    float ret = numStr.toFloat( &ok );
    if( !ok ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "cannot_parse_float_number_from_string:%1" ).arg( QString(numStr) ) );
    } else {
        num = ret;
    }
}

static void set2Floats( float& f1, float& f2, const QByteArray& str ) {
    QString line( str );
    QStringList words = line.split( QRegExp( "\\s+" ), QString::SkipEmptyParts );
    
    if( 2 != words.size() ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "cannot_parse_2_float_numbers_in_str:%1" ).arg( QString(str) ) );
    }
    setFloat( f1, words.at( 0 ).toAscii() );
    setFloat( f2, words.at( 1 ).toAscii() );
}

static void setUInteger( uint32_t& num, const QByteArray& numStr ) {
    bool ok = false;
    uint32_t ret = numStr.toUInt( &ok );
    if( !ok ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "cannot_parse_uinteger_number_from_string:%1" ).arg( QString(numStr) ) );
    } else {
        num = ret;
    }
}

static void setYesNoValue( int& flags, int val, const QByteArray& s ) {
    QByteArray str = s.toLower();
    if( "yes" == str ) {
        flags |= val;
    } else if( "no" != str ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "cannot_parse_yes/no_value_from_string:%1" ).arg( QString(str) ) );
    }
}

static void allocAndCopyStr( const QByteArray& from, char** to ) {
    assert( !from.isEmpty() && NULL != to );
    int sz = from.size();
    *to = (char*)calloc( sizeof( char ), sz + 1 );
    if( NULL == *to ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "no_memory:cannot_allocate_hmm_date" ) );
    }
    qCopy( from.data(), from.data() + sz, *to );
    (*to)[sz] = TERM_SYM;
}

const int ALPHA_VERSION_STATS_FIELDS_NUM = 3;
const int BETA_VERSION_STATS_FIELDS_NUM  = 4;

static void setHmmStats( float* params, const QByteArray& s, uint32_t& statstracker ) {
    assert( NULL != params );
    QString str( s );
    QStringList words = str.split( QRegExp( "\\s+" ), QString::SkipEmptyParts );
    int wordsSz = words.size();
    if( ALPHA_VERSION_STATS_FIELDS_NUM != wordsSz && BETA_VERSION_STATS_FIELDS_NUM != wordsSz ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "bad_stats_line:%1" ).arg( QString(str) ) );
    }
    if( "LOCAL" != words.at( 0 ).toUpper() ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "bad_stats_line:%1.\"LOCAL\" word is not found" ).arg( QString(str) ) );
    }
    
    if( ALPHA_VERSION_STATS_FIELDS_NUM == wordsSz ) {
        
    } else if( BETA_VERSION_STATS_FIELDS_NUM == wordsSz ) {
        
    } else {
        assert( false );
    }
    switch( wordsSz ) {
    case ALPHA_VERSION_STATS_FIELDS_NUM: // this one is for backward compatibility with hmmer3 alpha version
        {
            QByteArray  numStr = words.at( 2 ).toAscii();
            QString     tagStr = words.at( 1 ).toUpper();
            if( "VLAMBDA" == tagStr ) {
                setFloat( params[p7_MLAMBDA], numStr );
                setFloat( params[p7_VLAMBDA], numStr );
                setFloat( params[p7_FLAMBDA], numStr );
                statstracker |= 0x1;
            } else if( "VMU" == tagStr ) {
                setFloat( params[p7_MMU], numStr );
                setFloat( params[p7_VMU], numStr );
                statstracker |= 0x2;
            } else if( "FTAU" == tagStr ) {
                setFloat( params[p7_FTAU], numStr );
                statstracker |= 0x4;
            } else {
                throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "bad_stats_line:%1.%2 is not recognized" ).
                    arg( str ).arg( tagStr ) );
            }
        }
        break;
    case BETA_VERSION_STATS_FIELDS_NUM:
        {
            QString tagStr = words.at( 1 ).toUpper();
            QByteArray num1Str = words.at( 2 ).toAscii();
            QByteArray num2Str = words.at( 3 ).toAscii();
            if( "MSV" == tagStr ) {
                setFloat( params[p7_MMU], num1Str );
                setFloat( params[p7_MLAMBDA], num2Str );
                statstracker |= 0x1;
            } else if( "VITERBI" == tagStr ) {
                setFloat( params[p7_VMU], num1Str );
                setFloat( params[p7_VLAMBDA], num2Str );
                statstracker |= 0x2;
            } else if( "FORWARD" == tagStr ) {
                setFloat( params[p7_FTAU], num1Str );
                setFloat( params[p7_FLAMBDA], num2Str );
                statstracker |= 0x4;
            } else {
                throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "bad_stats_line:%1. %2 not recognized" ).
                    arg( str ).arg( tagStr ) );
            }
        }
        break;
    default:
        assert( false );
    }
    
}

static QByteArray getNextToken( QStringList& tokens ) {
    if( tokens.isEmpty() ) {
        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "unexpected_end_of_line" ) );
    }
    return tokens.takeFirst().toAscii();
}

static void setMainModelFloatVal( float& num, const QByteArray& str ) {
    if( "*" == str ) {
        num = 0.0f;
    } else {
        float tmp = 0;
        setFloat( tmp, str );
        num = expf( -1.0 * tmp );
    }
}

static void skipBlankLines( IOAdapter* io ) {
    assert( NULL != io && io->isOpen() );
    char c = 0;
    bool skip = true;
    while( skip ) {
        int ret = io->readBlock( &c, 1 );
        checkReadThrowException( ret );
        if( 0 == ret ) { return; }
        skip = TextUtils::LINE_BREAKS[(uchar)c] || TextUtils::WHITES[(uchar)c];
    }
    io->skip( -1 );
}

static void loadOne( IOAdapter* io, QList< GObject* >& objects, TaskStateInfo& si ) {
    assert( NULL != io && io->isOpen() );
    
    ESL_ALPHABET *abc  = NULL;
    P7_HMM       *hmm  = NULL;
    uint32_t      statstracker = 0;
    int           x = 0;
    bool          ok = false;
    
    try {
        QByteArray header;
        readLine( io, header );
        ok = checkHeader( header.data(), header.size() );
        if( !ok ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "invalid_header_line" ) );
        }
        
        if ((hmm = p7_hmm_CreateShell()) == NULL) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "no_memory:cannot_create_hmm_shell" ) );
        }
        
        /* Header section */
        QMap< QByteArray, HMMMERHeaderTags > headerTagsMap = getHeaderTagsMap();
        bool isHeaderSection = true;
        while( isHeaderSection ) {
            QByteArray line;
            QByteArray tagStr;
            QByteArray valueStr;
            HMMMERHeaderTags tag;
            
            readLine( io, line );
            getTagValue( line, tagStr, valueStr );
            tag = headerTagsMap.value( tagStr, BAD_TAG );
            switch( tag ) {
            case NAME:
                p7_hmm_SetName( hmm, valueStr.data() );
                break;
            case ACC:
                p7_hmm_SetAccession( hmm, valueStr.data() );
                break;
            case DESC:
                p7_hmm_SetDescription( hmm, valueStr.data() );
                break;
            case LENG:
                setInteger( hmm->M, valueStr );
                if( 0 >= hmm->M ) {
                    throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "length_of_a_model_should_be_positive.we_have:%1" ).arg( hmm->M ) );
                }
                break;
            case ALPH:
                {
                    int abcType = esl_abc_EncodeType( valueStr.data() );
                    if( eslUNKNOWN == abcType ) {
                        throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "unrecognized_alphabet_type:%1" ).arg( QString(valueStr) ) );
                    }
                    abc = esl_alphabet_Create( abcType );
                }
                break;
            case RF:
                setYesNoValue( hmm->flags, p7H_RF, valueStr );
                break;
            case CS:
                setYesNoValue( hmm->flags, p7H_CS, valueStr );
                break;
            case MAP:
                setYesNoValue( hmm->flags, p7H_MAP, valueStr );
                break;
            case DATE:
                allocAndCopyStr( valueStr, &(hmm->ctime) );
                break;
            case COM: // COM is command line that was used to create this hmm. we don't need it
                break;
            case NSEQ:
                setInteger( hmm->nseq, valueStr );
                if( 0 >= hmm->nseq ) {
                    throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "nseq_should_be_positive.we_have:%1" ).arg( hmm->nseq ) );
                }
                break;
            case EFFN:
                setFloat( hmm->eff_nseq, valueStr );
                if( 0 >= hmm->eff_nseq ) {
                    throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "effn_should_be_positive.we_have:%1" ).arg( hmm->eff_nseq ) );
                }
                break;
            case CKSUM:
                setUInteger( hmm->checksum, valueStr );
                hmm->flags |= p7H_CHKSUM;
                break;
            case STATS:
                setHmmStats( hmm->evparam, valueStr, statstracker );
                break;
            case GA:
                set2Floats( hmm->cutoff[p7_GA1], hmm->cutoff[p7_GA2], valueStr );
                hmm->flags |= p7H_GA;
                break;
            case TC:
                set2Floats( hmm->cutoff[p7_TC1], hmm->cutoff[p7_TC2], valueStr );
                hmm->flags |= p7H_TC;
                break;
            case NC:
                set2Floats( hmm->cutoff[p7_NC1], hmm->cutoff[p7_NC2], valueStr );
                hmm->flags |= p7H_NC;
                break;
            case HMM:
                isHeaderSection = false;
                continue;
            case BAD_TAG:
                throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "unrecognized_tag_in_header_section:%1" ).arg( QString(tagStr) ) );
            default:
                assert( 0 );
            }
        } /* end, loop over possible header tags */
        
        /* If we saw one STATS line, we needed all 3 */
        if( statstracker == 0x7 ) {
            hmm->flags |= p7H_STATS;
        } else if( statstracker != 0x0 ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "missing_one_or_more_STATS_parameter_lines" ) );
        }
        
        if( NULL == abc ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "failed_to_create_alphabet" ) );
        }
        
        if( p7_hmm_CreateBody( hmm, hmm->M, abc ) != eslOK ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "failed_to_allocate_body_of_hmm" ) );
        }
        
        QByteArray line;
        QStringList tokens;
        readLine( io, line );
        readLine( io, line, &tokens );
        QByteArray curToken = getNextToken( tokens );
        
        /* Optional model composition (filter null model) may immediately follow headers */
        if( "COMPO" == curToken.toUpper() ) {
            for( x = 0; x < abc->K; x++ ) {
                curToken = getNextToken( tokens );
                setMainModelFloatVal( hmm->compo[x], curToken );
            }
            hmm->flags |= p7H_COMPO;
            readLine( io, line, &tokens );
            curToken = getNextToken( tokens );
        }
        
        /* First two lines are node 0: insert emissions, then transitions from node 0 (begin) */
        setMainModelFloatVal( hmm->ins[0][0], curToken );
        for( x = 1; x < abc->K; x++ ) {
            curToken = getNextToken( tokens );
            setMainModelFloatVal( hmm->ins[0][x], curToken );
        }
        readLine( io, line, &tokens );
        for( x = 0; x < p7H_NTRANSITIONS; x++ ) {
            curToken = getNextToken( tokens );
            setMainModelFloatVal( hmm->t[0][x], curToken );
        }
        
        /* The main model section. */
        int k = 0;
        for( k = 1; k <= hmm->M; k++ ) {
            int n = 0;
            readLine( io, line, &tokens );
            curToken = getNextToken( tokens );
            setInteger( n, curToken );
            if( k != n ) {
                throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "expected_line_to_start_with:%1.it_starts_with:%2" ).arg( k ).arg( QString(curToken) ) );
            }
            for( x = 0; x < abc->K; x++ ) {
                curToken = getNextToken( tokens );
                setMainModelFloatVal( hmm->mat[k][x], curToken );
            }
            curToken = getNextToken( tokens );
            if( hmm->flags & p7H_MAP ) {
                int num = 0;
                setInteger( num, curToken );
                hmm->map[k] = num;
            }
            curToken = getNextToken( tokens );
            if (hmm->flags & p7H_RF) {
                assert( !curToken.isEmpty() );
                hmm->rf[k] = curToken.at( 0 );
            }
            
            curToken = getNextToken( tokens );
            if( hmm->flags & p7H_CS ) {
                assert( !curToken.isEmpty() );
                hmm->cs[k] = curToken.at( 0 );
            }
            
            readLine( io, line, &tokens );
            for( x = 0; x < abc->K; x++ ) {
                curToken = getNextToken( tokens );
                setMainModelFloatVal( hmm->ins[k][x], curToken );
            }
            
            readLine( io, line, &tokens );
            for( x = 0; x < p7H_NTRANSITIONS; x++ ) {
                curToken = getNextToken( tokens );
                setMainModelFloatVal( hmm->t[k][x], curToken );
            }
        }
        
        /* The closing // */
        readLine( io, line, &tokens );
        curToken = getNextToken( tokens );
        if( "//" != curToken ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "expected_closing_//.found_%1_instead" ).arg( QString(curToken) ) );
        }
        skipBlankLines( io );
        
        /* Finish up. */
        if( hmm->flags & p7H_RF ) {
            hmm->rf[0]  = ' ';
            hmm->rf[hmm->M+1] = '\0';
        }
        if( hmm->flags & p7H_CS ) {
            hmm->cs[0]  = ' '; 
            hmm->cs[hmm->M+1] = '\0';
        }
        if( hmm->flags & p7H_MAP ) {
            hmm->map[0] = 0;
        }
        if( hmm->name == NULL ) {
            throw UHMMFormat::UHMMReadException( UHMMFormat::tr( "no_name_found_in_hmm" ) );
        }
        assert( 0 <= hmm->M );
        assert( NULL != hmm->abc );
    } catch( const UHMMFormat::UHMMReadException& ex ) {
        si.setError( ex.what );
    } catch(...) {
        si.setError( HMMER3_UNKNOWN_ERROR );
    }
    esl_alphabet_Destroy( abc );
    
    if( si.hasErrors() ) {
        p7_hmm_Destroy( hmm );
        return;
    }
    
    assert( NULL != hmm );
    QString objName( hmm->name );
    UHMMObject* obj = new UHMMObject( hmm, objName );
    objects.append( obj );
}

static void loadAll( IOAdapter* io, QList< GObject* >& objects, TaskStateInfo& ti ) {
    assert( NULL != io && io->isOpen() );
    
    while( !io->isEof() && !ti.hasErrors() && !ti.cancelFlag ) {
        loadOne( io, objects, ti );
    }
}

static void checkBytesWrittenThrowException( qint64 wantedToWrite, qint64 written, const QString& msg ) {
    if( wantedToWrite != written ) {
        throw UHMMFormat::UHMMWriteException( msg );
    }
}

static void writeHMMASCIIStr( IOAdapter* io, const QByteArray& str ) {
    assert( NULL != io && io->isOpen() );
    qint64 bytesWritten = io->writeBlock( str );
    checkBytesWrittenThrowException( str.size(), bytesWritten, UHMMFormat::WRITE_FAILED );
}

static void writeHMMASCIIStr( IOAdapter* io, const char* str, int num ) {
    assert( NULL != io && io->isOpen() );
    assert( NULL != str && 0 < num );
    qint64 bytesWritten = io->writeBlock( str, num );
    checkBytesWrittenThrowException( num, bytesWritten, UHMMFormat::WRITE_FAILED );
}

static void writeHMMASCIIStr( IOAdapter* io, const char* s1, const char* s2 ) {
    QString str;
    QTextStream txtStream( &str );
    txtStream << s1 << s2 << "\n";
    writeHMMASCIIStr( io, str.toAscii() );
}

static void writeHMMASCIIStr( IOAdapter* io, const char* s1, const QString& s2 ) {
    QString str;
    QTextStream txtStream( &str );
    txtStream << s1 << s2 << "\n";
    writeHMMASCIIStr( io, str.toAscii() );
}

static void writeHMMHeaderASCII( IOAdapter* io ) {
    QTextStream txtStream;
    QString headerStr;
    txtStream.setString( &headerStr );
    txtStream << "HMMER3/b [" << HMMER_VERSION << " | " << HMMER_DATE << "]\n";
    writeHMMASCIIStr( io, headerStr.toAscii() );
}

static void writeHMMMultiLine( IOAdapter *io, const char *pfx, char *s ) {
    QString res;
    char *sptr  = s;
    char *end   = NULL;
    int   n     = 0;
    int   nline = 1;
    
    do {
        end = strchr(sptr, '\n');
        if (end != NULL) { 		             /* if there's no \n left, end == NULL */
            n = end - sptr;	                     /* n chars exclusive of \n */
            
            res = QString().sprintf( "%s [%d] ", pfx, nline++ );
            writeHMMASCIIStr( io, res.toAscii() );
            writeHMMASCIIStr( io, sptr, n );
            writeHMMASCIIStr( io, QByteArray( "\n" ) );
            sptr += n + 1;	                     /* +1 to get past \n */
        } else {
            res = QString().sprintf( "%s [%d] %s\n", pfx, nline++, sptr );
            writeHMMASCIIStr( io, res.toAscii() );
        }
    } while (end != NULL  && *sptr != '\0');   /* *sptr == 0 if <s> terminates with a \n */
}

static void writeHMMProb( IOAdapter* io, int fieldwidth, float p ) {
    assert( NULL != io && io->isOpen() );
    QString res;
    if( p == 0.0 ) {
        res = QString().sprintf( " %*s", fieldwidth, "*" );
        writeHMMASCIIStr( io, res.toAscii() );
    } else if( p == 1.0 ) {
        res = QString().sprintf( " %*.5f", fieldwidth, 0.0 );
        writeHMMASCIIStr( io, res.toAscii() );
    } else {
        res = QString().sprintf( " %*.5f", fieldwidth, -logf(p) );
        writeHMMASCIIStr( io, res.toAscii() );
    }
}

static void saveOne( IOAdapter* io, const P7_HMM* hmm, TaskStateInfo& ti ) {
    assert( NULL != hmm );
    assert( NULL != io && io->isOpen() );
    assert( !ti.hasErrors() );
    
    try {
        int k = 0;
        int x = 0;
        QString res;
        
        writeHMMHeaderASCII( io );
        writeHMMASCIIStr( io, "NAME  ", hmm->name );
        
        if (hmm->flags & p7H_ACC) {
            writeHMMASCIIStr( io, "ACC   ", hmm->acc );
        }
        if (hmm->flags & p7H_DESC) {
            writeHMMASCIIStr( io, "DESC  ", hmm->desc );
        }
        writeHMMASCIIStr( io, "LENG  ", QString::number( hmm->M ) );
        writeHMMASCIIStr( io, "ALPH  ", esl_abc_DecodeType( hmm->abc->type ) );
        writeHMMASCIIStr( io, "RF    ", ( hmm->flags & p7H_RF )? "yes" : "no" );
        writeHMMASCIIStr( io, "CS    ", ( hmm->flags & p7H_CS )? "yes" : "no" );
        writeHMMASCIIStr( io, "MAP   ", ( hmm->flags & p7H_MAP )? "yes" : "no" );
        
        if (hmm->ctime    != NULL) {
            writeHMMASCIIStr( io, "DATE  ", hmm->ctime );
        }
        if (hmm->comlog   != NULL) {
            writeHMMMultiLine( io, "COM  ", hmm->comlog );
        }
        if (hmm->nseq     >= 0) {
            writeHMMASCIIStr( io, "NSEQ  ", QString::number( hmm->nseq ) );
        }
        if (hmm->eff_nseq >= 0) {
            res = QString().sprintf( "EFFN  %f\n", hmm->eff_nseq );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        if (hmm->flags & p7H_CHKSUM) {
            writeHMMASCIIStr( io, "CKSUM ", QString::number( hmm->checksum ) );
        }
        
        if (hmm->flags & p7H_GA) {
            res = QString().sprintf( "GA    %.2f %.2f\n", hmm->cutoff[p7_GA1], hmm->cutoff[p7_GA2] );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        if (hmm->flags & p7H_TC) {
            res = QString().sprintf( "TC    %.2f %.2f\n", hmm->cutoff[p7_TC1], hmm->cutoff[p7_TC2] );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        if (hmm->flags & p7H_NC) {
            res = QString().sprintf( "NC    %.2f %.2f\n", hmm->cutoff[p7_NC1], hmm->cutoff[p7_NC2] );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        
        if (hmm->flags & p7H_STATS) {
            res = QString().sprintf( "STATS LOCAL MSV      %8.4f %8.5f\n", hmm->evparam[p7_MMU],  hmm->evparam[p7_MLAMBDA] );
            writeHMMASCIIStr( io, res.toAscii() );
            res = QString().sprintf( "STATS LOCAL VITERBI  %8.4f %8.5f\n", hmm->evparam[p7_VMU],  hmm->evparam[p7_VLAMBDA] );
            writeHMMASCIIStr( io, res.toAscii() );
            res = QString().sprintf( "STATS LOCAL FORWARD  %8.4f %8.5f\n", hmm->evparam[p7_FTAU], hmm->evparam[p7_FLAMBDA] );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        writeHMMASCIIStr( io, QByteArray( "HMM     " ) );
        
        for (x = 0; x < hmm->abc->K; x++) {
            res = QString().sprintf( "     %c   ", hmm->abc->sym[x] );
            writeHMMASCIIStr( io, res.toAscii() );
        }
        writeHMMASCIIStr( io, QByteArray( "\n" ) );
        res = QString().sprintf( "        %8s %8s %8s %8s %8s %8s %8s\n", "m->m", "m->i", "m->d", "i->m", "i->i", "d->m", "d->d" );
        writeHMMASCIIStr( io, res.toAscii() );
        
        if (hmm->flags & p7H_COMPO) {
            writeHMMASCIIStr( io, QByteArray( "  COMPO " ) );

            for (x = 0; x < hmm->abc->K; x++) {
                writeHMMProb( io, 8, hmm->compo[x] );
            }
            writeHMMASCIIStr( io, QByteArray( "\n" ) );
        }
        
        /* node 0 is special: insert emissions, and B-> transitions */
        writeHMMASCIIStr( io, QByteArray( "        " ) );
        
        for (x = 0; x < hmm->abc->K; x++) {
            writeHMMProb( io, 8, hmm->ins[0][x] );
        }
        writeHMMASCIIStr( io, QByteArray( "\n" ) );
        writeHMMASCIIStr( io, QByteArray( "        " ) );
        
        for (x = 0; x < p7H_NTRANSITIONS; x++) {
            writeHMMProb( io, 8, hmm->t[0][x] );
        }
        writeHMMASCIIStr( io, QByteArray( "\n" ) );
        
        for (k = 1; k <= hmm->M; k++) {
            /* Line 1: k; match emissions; optional map, RF, CS */ 
            res = QString().sprintf( " %6d ",  k );
            writeHMMASCIIStr( io, res.toAscii() );
            
            for (x = 0; x < hmm->abc->K; x++) {
                writeHMMProb( io, 8, hmm->mat[k][x] );
            }
            if (hmm->flags & p7H_MAP) {
                res = QString().sprintf( " %6d", hmm->map[k] );
                writeHMMASCIIStr( io, res.toAscii() );
            } else {
                res = QString().sprintf( " %6s", "-" );
                writeHMMASCIIStr( io, res.toAscii() );
            }
            res = QString().sprintf( " %c",   (hmm->flags & p7H_RF) ? hmm->rf[k] : '-' );
            writeHMMASCIIStr( io, res.toAscii() );
            res = QString().sprintf( " %c\n", (hmm->flags & p7H_CS) ? hmm->cs[k] : '-' );
            writeHMMASCIIStr( io, res.toAscii() );
            
            /* Line 2:   insert emissions */
            writeHMMASCIIStr( io, QByteArray( "        " ) );
            for (x = 0; x < hmm->abc->K; x++) {
                writeHMMProb( io, 8, hmm->ins[k][x] );
            }
            
            /* Line 3:   transitions */
            writeHMMASCIIStr( io, QByteArray( "\n        " ) );
            for (x = 0; x < p7H_NTRANSITIONS; x++) {
                writeHMMProb( io, 8, hmm->t[k][x] );
            }
            writeHMMASCIIStr( io, QByteArray( "\n" ) );
        }
        writeHMMASCIIStr( io, QByteArray( "//\n" ) );
    } catch( const UHMMFormat::UHMMWriteException& ex ) {
        ti.setError( ex.what );
    } catch(...) {
        ti.setError( UHMMFormat::tr( "unknown_error_occurred" ) );
    }
}

static void saveAll( IOAdapter* io, const QList< GObject* >& objects, TaskStateInfo& ti ) {
    assert( NULL != io && io->isOpen() );
    QList< const P7_HMM* > hmms;
    
    foreach( const GObject* obj, objects ) {
        const UHMMObject* hmmObj = qobject_cast< const UHMMObject* >( obj );
        if( NULL == hmmObj ) {
            ti.setError( Translations::badArgument( "Objects in document" ) );
            return;
        }
        hmms.append( hmmObj->getHMM() );
    }
    
    foreach( const P7_HMM* hmm, hmms ) {
        saveOne( io, hmm, ti );
        if( ti.hasErrors() || ti.cancelFlag ) {
            return;
        }
    }
}

namespace GB2 {

const DocumentFormatId  UHMMFormat::UHHMER_FORMAT_ID       = "hmmer_document_format";
const QByteArray        UHMMFormat::HMMER_HEADER        = "HMMER";
const QString           UHMMFormat::WRITE_LOCK_REASON   = UHMMFormat::tr( "hmm_files_are_read_only" );

UHMMFormat::UHMMFormat( QObject* obj ) : DocumentFormat( obj ) {
    formatName = tr( "hmmer_format" );
}

DocumentFormatId UHMMFormat::getFormatId() const {
    return UHHMER_FORMAT_ID;
}

const QString& UHMMFormat::getFormatName() const {
    return formatName;
}

QStringList UHMMFormat::getSupportedDocumentFileExtensions() {
    QStringList res;
    res << "hmm";
    return res;
}

Document* UHMMFormat::loadExistingDocument( IOAdapterFactory* iof, const QString& url, TaskStateInfo& ti, const QVariantMap& hints ) {
    if( NULL == iof ) {
        ti.setError( Translations::badArgument( "io_adapter_factory" ) );
        return NULL;
    }
    
    std::auto_ptr< IOAdapter > io( iof->createIOAdapter() );
    if( NULL == io.get() ) {
        ti.setError( tr( "cannot_create_io_adapter" ) );
        return NULL;
    }
    if( !io->open( url, IOAdapterMode_Read ) ) {
        ti.setError( Translations::errorOpeningFileRead( url ) );
        return NULL;
    }
    
    QList< GObject* > objects;
    loadAll( io.get(), objects, ti );
    io->close();
    if( ti.hasErrors() || ti.cancelFlag ) {
        qDeleteAll( objects );
        return NULL;
    }
    return new Document( this, iof, url, objects, hints, WRITE_LOCK_REASON );
}

Document* UHMMFormat::loadExistingDocument( IOAdapter* io, TaskStateInfo& ti, const QVariantMap& fs ) {
    if( NULL == io || !io->isOpen() ) {
        ti.setError( Translations::badArgument( "io_adapter" ) );
        return NULL;
    }
    
    QList< GObject* > objects;
    loadOne( io, objects, ti );
    if( ti.hasErrors() || ti.cancelFlag ) {
        assert( objects.isEmpty() );
        return NULL;
    }
    assert( 1 == objects.size() );
    return new Document( this, io->getFactory(), io->getUrl(), objects, fs, WRITE_LOCK_REASON );
}

void UHMMFormat::storeDocument( Document* doc, TaskStateInfo& ti, IOAdapterFactory* iof, const QString& newURL ) {
    if( NULL == doc ) {
        ti.setError( Translations::badArgument( "Document" ) );
        return;
    }
    
    iof = iof? iof : doc->getIOAdapterFactory();
    std::auto_ptr< IOAdapter > io( iof->createIOAdapter() );
    QString url = newURL.isEmpty()? doc->getURL() : newURL;
    assert( !url.isEmpty() );
    
    if( !io->open( url, IOAdapterMode_Write ) ) {
        ti.setError( Translations::errorOpeningFileWrite( url ) );
        return;
    }
    saveAll( io.get(), doc->getObjects(), ti );
    io->close();
}

void UHMMFormat::storeDocument( Document* doc, TaskStateInfo& ti, IOAdapter* io ) {
    if( NULL == doc ) {
        ti.setError( Translations::badArgument( "Document" ) );
        return;
    }
    if( NULL == io || !io->isOpen() ) {
        ti.setError( Translations::badArgument( "IOAdapter" ) );
        return;
    }
    saveAll( io, doc->getObjects(), ti );
}

bool UHMMFormat::isDataFormatSupported( const char* data, int size ) const {
    if( NULL == data || 0 > size ) {
        return false;
    }
    return checkHeader( data, size );
}

bool UHMMFormat::isObjectOpSupported( const Document* d , DocumentFormat::DocObjectOp op, GObjectType t ) const {
    assert( NULL != d );
    if( UHMMObject::UHMM_OT != t ) {
        return false;
    }
    if ( op == DocumentFormat::DocObjectOp_Add ) {
        return d->getObjects().isEmpty();
    }
    
    return false;
}

bool UHMMFormat::checkConstraints( const DocumentFormatConstraints& constr ) const {
    bool ret = true;
    foreach( GObjectType t, constr.supportedObjectTypes ) {
        ret = ret && ( UHMMObject::UHMM_OT == t );
    }
    if ( constr.checkRawData ) {
        ret = ret && isDataFormatSupported( constr.rawData.constData(), constr.rawData.size() );
    }
    if( constr.supportsStreamingRead ) {
        ret = ret && true;
    }
    return ret;
}

} // GB2
