/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtGui/QMenu>
#include <QtGui/QAction>
#include <QtGui/QMessageBox>

#include <core_api/MainWindow.h>
#include <core_api/AppContext.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/ProjectView.h>
#include <util_algorithm/GAutoDeleteList.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <selection/GObjectSelection.h>

#include <test_framework/GTestFrameworkComponents.h>
#include <test_framework/xmltest/XMLTestFormat.h>

#include <tests/uhmmer3Tests.h>
#include <format/uHMMFormat.h>

#include <build/uHMM3BuildDialogImpl.h>
#include <search/uHMM3SearchDialogImpl.h>
#include <phmmer/uHMM3PhmmerDialogImpl.h>

#include "uHMM3Plugin.h"

namespace GB2 {

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    UHMM3Plugin * plug = new UHMM3Plugin();
    return plug;
}

UHMM3Plugin::UHMM3Plugin() : Plugin( tr( "hmm3_plugin_name" ), tr( "hmm3_plugin_desc" ) ) {
    // UHMMFormat
    DocumentFormatRegistry* dfRegistry = AppContext::getDocumentFormatRegistry();
    assert( NULL != dfRegistry );
    bool ok = dfRegistry->registerFormat( new UHMMFormat( dfRegistry ) );
    assert( ok ); Q_UNUSED( ok );
    
    // Tests
    GTestFormatRegistry* tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    assert( NULL != tfr );
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert( NULL != xmlTestFormat );
    
    GAutoDeleteList<XMLTestFactory>* l = new GAutoDeleteList<XMLTestFactory>(this);
    l->qlist = UHMMER3Tests::createTestFactories();
    
    foreach( XMLTestFactory* f, l->qlist ) { 
        bool res = xmlTestFormat->registerTestFactory( f );
        assert(res);
        Q_UNUSED(res);
    }
    
    // HMMER3 menu
    MainWindow * mainWnd = AppContext::getMainWindow();
    if( mainWnd ) {
        MWMenuManager * menuMngr = mainWnd->getMenuManager();
        assert( NULL != menuMngr );
        QMenu * toolsMenu = menuMngr->getTopLevelMenu( MWMENU_TOOLS );
        assert( NULL != toolsMenu );
        QMenu * toolsHMMSub = toolsMenu->addMenu(QIcon( ":/hmm3/images/hmmer_16.png" ), tr("HMM3 tools"));
        assert( NULL != toolsHMMSub );
        
        QAction * buildAction = new QAction( tr( "build profile" ), this );
        connect( buildAction, SIGNAL( triggered() ), SLOT( sl_buildProfile() ) );
        toolsHMMSub->addAction( buildAction );
        
        QAction * searchAction = new QAction( tr( "search HMM signals" ), this );
        connect( searchAction, SIGNAL( triggered() ), SLOT( sl_searchHMMSignals() ) );
        toolsHMMSub->addAction( searchAction );
        
        QAction * phmmerAction = new QAction( tr( "phmmer search" ), this );
        connect( phmmerAction, SIGNAL( triggered() ), SLOT( sl_phmmerSearch() ) );
        toolsHMMSub->addAction( phmmerAction );
    }
}

void UHMM3Plugin::sl_buildProfile() {
    UHMM3BuildDialogImpl buildDlg;
    buildDlg.exec();
}

/* TODO: same as in function sl_search in uHMMPlugin.cpp */
DNASequenceObject * UHMM3Plugin::getDnaSequenceObject() const {
    DNASequenceObject * seqObj = NULL;
    GObjectViewWindow * activeWnd = qobject_cast< GObjectViewWindow* >( AppContext::getMainWindow()->getMDIManager()->getActiveWindow() );
    if( NULL != activeWnd ) {
        AnnotatedDNAView * dnaView = qobject_cast< AnnotatedDNAView* >( activeWnd->getObjectView() );
        seqObj = NULL != dnaView ? seqObj = dnaView->getSequenceInFocus()->getSequenceObject() : NULL;
    }
    if( NULL == seqObj ) {
        ProjectView * projView = AppContext::getProjectView();
        if( NULL != projView ) {
            const GObjectSelection * objSelection = projView->getGObjectSelection();
            GObject* obj = objSelection->getSelectedObjects().size() == 1 ? objSelection->getSelectedObjects().first() : NULL;
            seqObj = qobject_cast< DNASequenceObject* >( obj );
        }
    }
    return seqObj;
}

void UHMM3Plugin::sl_searchHMMSignals() {
    DNASequenceObject * seqObj = getDnaSequenceObject();
    if( NULL == seqObj ) {
        QMessageBox::critical( NULL, tr( "Error!" ), tr( "Target sequence not selected: no opened annotated dna view" ) );
        return;
    }
    
    UHMM3SearchDialogImpl searchDlg( seqObj );
    searchDlg.exec();
}

void UHMM3Plugin::sl_phmmerSearch() {
    DNASequenceObject * seqObj = getDnaSequenceObject();
    if( NULL == seqObj ) {
        QMessageBox::critical( NULL, tr( "Error!" ), tr( "Target sequence not selected: no opened annotated dna view" ) );
        return;
    }
    
    UHMM3PhmmerDialogImpl phmmerDlg( seqObj );
    phmmerDlg.exec();
}

UHMM3Plugin::~UHMM3Plugin() {
}

} // GB2
