/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MainWindowImpl.h"
#include "DockManagerImpl.h"
#include "MDIManagerImpl.h"
#include "ShutdownTask.h"
#include "MenuManager.h"
#include "ToolBarManager.h"
#include <core_api/AppGlobals.h>

#include "AboutDialogController.h"
#include "CheckUpdatesTask.h"

#include <core_api/Settings.h>
#include <core_api/AppContext.h>
#include <core_api/Task.h>
#include <core_api/Log.h>
#include <core_api/ProjectModel.h>

#include <util_gui/GUIUtils.h>

#include <QtGui/QToolBar>
#include <QtGui/QAction>
#include <QtGui/QPainter>
#include <QtGui/QFont>
#include <QtGui/QPixmap>

#include <algorithm>

namespace GB2 {

/* TRANSLATOR GB2::MainWindowImpl */

static LogCategory log(ULOG_CAT_USER_INTERFACE);


#define SETTINGS_DIR QString("main_window/")

class MWStub : public QMainWindow {
public:
    MWStub(MainWindowImpl* _owner)  : owner(_owner){
        setAttribute(Qt::WA_NativeWindow);
        setAcceptDrops(true);
    }
    virtual QMenu * createPopupMenu () {return NULL;} //todo: decide if we do really need this menu and fix it if yes?
protected:
	virtual void closeEvent(QCloseEvent* e); 
    virtual void dragEnterEvent(QDragEnterEvent *event);
    virtual void dropEvent ( QDropEvent * event );
protected:
	MainWindowImpl* owner;
};

void MWStub::closeEvent(QCloseEvent* e) {
    if (owner->getMenuManager() == NULL) {
        QMainWindow::closeEvent(e);
    } else {
        owner->runClosingTask();
        e->ignore();
    }
}

void MWStub::dragEnterEvent(QDragEnterEvent *event)
{
    if (event->mimeData()->hasUrls()) {
        event->acceptProposedAction();
    }
}

void MWStub::dropEvent(QDropEvent *event)
{
    if (event->mimeData()->hasUrls()) {
        Task* t = AppContext::getProjectLoader()->openProjectTask(event->mimeData()->urls(), true);
        AppContext::getTaskScheduler()->registerTopLevelTask(t);
        event->acceptProposedAction();
    }
}

//////////////////////////////////////////////////////////////////////////
// MainWindowController
//////////////////////////////////////////////////////////////////////////
MainWindowImpl::MainWindowImpl() {
	mw = NULL;
	mdi = NULL;
	menuManager = NULL;
	toolbarManager = NULL;
	mdiManager = NULL;
	dockManager = NULL;
	exitAction = NULL;
}

MainWindowImpl::~MainWindowImpl() {
	assert(mw == NULL);
}

void MainWindowImpl::show() {
    createActions();
    prepareGUI();

	bool maximized =AppContext::getSettings()->getValue(SETTINGS_DIR + "maximized", false).toBool();
	QRect geom =AppContext::getSettings()->getValue(SETTINGS_DIR + "geometry", QRect()).toRect();

	if (maximized) {
		mw->showMaximized();
	} else {
		if (!geom.isNull()) {
			mw->setGeometry(geom);
		}
	    mw->show();
	}
}

void MainWindowImpl::close() {
	AppContext::getSettings()->setValue(SETTINGS_DIR + "maximized", mw->isMaximized());
	AppContext::getSettings()->setValue(SETTINGS_DIR + "geometry", mw->geometry());

    delete dockManager;	dockManager = NULL;
    delete menuManager;	menuManager = NULL;
	delete toolbarManager; toolbarManager = NULL;
    delete mdiManager;	mdiManager = NULL;
    delete mdi;	mdi = NULL;
    mw->close();
	delete mw;	mw = NULL;
}


void MainWindowImpl::createActions() {
    exitAction = new QAction(tr("mw_menu_file_exit"), this);
    exitAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_X));
    exitAction->setShortcutContext(Qt::WindowShortcut);
    connect(exitAction, SIGNAL(triggered()), SLOT(sl_exitAction()));

    aboutAction = new QAction(tr("mw_menu_help_about"), this);
    aboutAction->setShortcut(QKeySequence(Qt::Key_F1));
    aboutAction->setShortcutContext(Qt::ApplicationShortcut);
    connect(aboutAction, SIGNAL(triggered()), SLOT(sl_aboutAction()));

    visitWebAction = new QAction(tr("Visit UGENE's Web Site"), this);
    connect(visitWebAction, SIGNAL(triggered()), SLOT(sl_visitWeb()));

    checkUpdateAction = new QAction(tr("Check for Updates"), this);
    connect(checkUpdateAction, SIGNAL(triggered()), SLOT(sl_checkUpdatesAction()));
}

void MainWindowImpl::sl_exitAction() {
	runClosingTask();
}

void MainWindowImpl::sl_aboutAction() {
    AboutDialogController d(visitWebAction);
    d.exec();
}


void MainWindowImpl::sl_checkUpdatesAction() {
    AppContext::getTaskScheduler()->registerTopLevelTask(new CheckUpdatesTask());
}


void MainWindowImpl::setWindowTitle(const QString& title) {
    if (title.isEmpty()) {
        mw->setWindowTitle(tr("main_window_title"));
    } else {
	    mw->setWindowTitle(title + " " + tr("main_window_title") );
    }
}

void MainWindowImpl::prepareGUI() {
	mw = new MWStub(this); //todo: parents?
    mw->setObjectName("main_window");
    setWindowTitle("");

    mdi = new QMdiArea(mw);
    mdi->setObjectName("MDI_Area");

	mw->setCentralWidget(mdi);
	mw->setCorner(Qt::BottomLeftCorner, Qt::LeftDockWidgetArea);
	mw->setCorner(Qt::BottomRightCorner, Qt::RightDockWidgetArea);
    
	toolbarManager = new MWToolBarManagerImpl(mw);

	menuManager = new MWMenuManagerImpl(this, mw->menuBar());

	exitAction->setObjectName(ACTION__EXIT);
    menuManager->getTopLevelMenu(MWMENU_FILE)->addAction(exitAction);

    aboutAction->setObjectName(ACTION__ABOUT);
    menuManager->getTopLevelMenu(MWMENU_HELP)->addAction(checkUpdateAction);
    menuManager->getTopLevelMenu(MWMENU_HELP)->addAction(visitWebAction);
    menuManager->getTopLevelMenu(MWMENU_HELP)->addAction(aboutAction);

	mdiManager = new MWMDIManagerImpl(this, mdi);

	dockManager = new MWDockManagerImpl(this);
}


void MainWindowImpl::runClosingTask() {
	AppContext::getTaskScheduler()->registerTopLevelTask(new ShutdownTask(this));
}

void MainWindowImpl::sl_visitWeb() {
    GUIUtils::runWebBrowser("http://ugene.unipro.ru");
}

}//namespace
