/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ProjectTasks.h"

#include "ProjectServiceImpl.h"
#include "ProjectImpl.h"
#include "ProjectLoaderImpl.h"
#include <core_services/AppContextImpl.h>

#include <util_tasks/SaveDocumentTask.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_gui/GUIUtils.h>
#include <document_format/DocumentFormatUtils.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/GHints.h>

#include <gobjects/UnloadedObject.h>

#include <QtXml/qdom.h>
#include <QtGui/QMessageBox>

namespace GB2 {

static LogCategory log(ULOG_CAT_CORE_SERVICES);

//////////////////////////////////////////////////////////////////////////
///Close project
CloseProjectTask::CloseProjectTask() : Task(tr("close_project_task_name"), TaskFlags_NR_FOSCOE)
{
}

void CloseProjectTask::prepare() {
	if (AppContext::getProject()==NULL) {
		stateInfo.setError(  tr("error_no_active_project") );
		return;
	}
	/* TODO: this is done by project view. Need to cleanup this part! 
        addSubTask(new SaveProjectTask(SaveProjectTaskKind_SaveProjectAndDocumentsAskEach));
    */
	ServiceRegistry* sr = AppContext::getServiceRegistry();
	QList<Service*> services = sr->findServices(Service_Project);
	assert(services.size() == 1);
	Service* projectService = services.first();
	addSubTask(sr->unregisterServiceTask(projectService));
}


//////////////////////////////////////////////////////////////////////////
/// OpenProjectTask
OpenProjectTask::OpenProjectTask(const QString& _url, bool _closeActiveProject, const QString& _name) 
: Task(tr("open_project_task_name"), TaskFlags_NR_FOSCOE), url(_url), loadProjectTask(NULL), closeActiveProject(_closeActiveProject)
{
}

OpenProjectTask::OpenProjectTask(const QList<QUrl>& list, bool _closeActiveProject) 
: Task(tr("open_project_task_name"), TaskFlags_NR_FOSCOE), urlList(list), loadProjectTask(NULL), closeActiveProject(_closeActiveProject) 
{
}



void OpenProjectTask::prepare() {
    if (urlList.size() == 1 && url.isEmpty()) {
        //FIXME issue 534
        QUrl tmp = urlList.takeFirst();
        if (tmp.isRelative() || tmp.scheme() == "file") {
            url = tmp.toLocalFile();
        } else {
            url = tmp.toString();
        }
    }
    
    if (url.endsWith(PROJECTFILE_EXT)) { // open a project
        QFileInfo f(url);
        if (f.exists() && !(f.isFile() && f.isReadable())) {
            stateInfo.setError(  tr("invalid_url%1").arg(url) );
            return;
        }
        
        // close current
        if (AppContext::getProject()!=NULL) {
            if (!closeActiveProject) {
                stateInfo.cancelFlag = true;
                log.info(tr("Stopped loading project: %1. Reason: active project found").arg(url));
                return;
            }
            addSubTask(new CloseProjectTask());
        }

        if (f.exists()) {
            loadProjectTask = new LoadProjectTask(url);
            addSubTask(loadProjectTask);
        } else {
            ProjectImpl* p =  new ProjectImpl(name, url);
            addSubTask(new SaveProjectTask(SaveProjectTaskKind_SaveProjectOnly, p));
            addSubTask(new RegisterProjectServiceTask(p));
        }
    } else { // load a (bunch of) documents
        if (!url.isEmpty()) {
            urlList << BaseIOAdapters::str2url(url);
        }
        Project* p = AppContext::getProject();
        if (!p) {
            // create anonymous project
            log.info(tr("Creating new project"));
            p = new ProjectImpl("", "");
            Task* rpt = new RegisterProjectServiceTask(p);
            rpt->setSubtaskProgressWeight(0);
            addSubTask(rpt);
        }
        foreach(QUrl _url, urlList) {
            //FIXME issue 534
            QString url = _url.toString();
            if (_url.isRelative() || _url.scheme() == "file") {
                url = _url.toLocalFile();
            }
            if (url.endsWith(PROJECTFILE_EXT)) {
                // skip extra project files
                stateInfo.setError(  tr("ignore %1").arg(url) );
                continue;
            }
            Document * doc = p->findDocumentByURL(url);
            if (!doc) {
                QList<DocumentFormat*> fs = DocumentFormatUtils::detectFormat(url);
                if (fs.isEmpty()) {
                    stateInfo.setError(  tr("unsupported_document%1").arg(url) );
                    continue;
                }

                DocumentFormat* format = fs.first();
                assert(format);
                IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
                doc = new Document(format, iof, url);
                Task * adt = new AddDocumentTask(doc);
                adt->setSubtaskProgressWeight(0);
                addSubTask(adt);
            } else if (doc->isLoaded()) {
                // skip duplicate files
                // TODO: activate view for the document
                log.info(tr("Document is already opened: %1").arg(url));
                continue;
            }

            if (getSubtasks().size() < 50) {
                addSubTask(new LoadUnloadedDocumentAndOpenViewTask(doc));
            }
        }
    }
}

QList<Task*> OpenProjectTask::onSubTaskFinished(Task* subTask) {
	QList<Task*> res;
	if (!isCanceled() && subTask == loadProjectTask && !loadProjectTask->hasErrors()) {
		Project* p =  loadProjectTask->detachProject();
		res.append(new RegisterProjectServiceTask(p));
	}
	return res;
}


//////////////////////////////////////////////////////////////////////////
/// Save project
SaveProjectTask::SaveProjectTask(SaveProjectTaskKind _k, Project* p, const QString& _url) 
: Task(tr("save_project_task_name"), TaskFlag_None), k(_k), proj(p), url(_url)
{
	lock = NULL;
}

SaveProjectTask::~SaveProjectTask () {
    assert(lock == NULL);
}

void SaveProjectTask::prepare() {
    if (proj == NULL) {
		proj = AppContext::getProject();
	}
	assert(proj!=NULL);
    if (url.isEmpty()) {
        url = proj->getProjectURL();
    }
    if (url.isEmpty() && (!proj->getGObjectViewStates().isEmpty() || proj->getDocuments().size() > 1)) {
        //ask if to save project
        int code = QMessageBox::question(NULL, tr("question"), tr("save_project_file_question"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
        if (code == QMessageBox::Yes) {
            ProjectDialogController d(ProjectDialogController::Save_Project);
            d.projectFolderEdit->setText(QDir::home().absolutePath());
            d.projectNameEdit->setText(tr("My project"));
            int rc = d.exec();
            if (rc == QDialog::Accepted) {
                AppContext::getProject()->setProjectName(d.projectNameEdit->text());
                url = d.projectFolderEdit->text() + "/" + d.projectFileEdit->text();
                if (!url.endsWith(PROJECTFILE_EXT)) {
                    url.append(PROJECTFILE_EXT);
                }
                AppContext::getProject()->setProjectURL(url);
            }
        }
    }
    
    lock = new StateLock(getTaskName(), StateLockFlag_LiveLock);
	proj->lockState(lock);

	if (k!=SaveProjectTaskKind_SaveProjectOnly) {
		QList<Document*> modifiedDocs = SaveMiltipleDocuments::findModifiedDocuments(AppContext::getProject()->getDocuments());
		if (!modifiedDocs.isEmpty()) {
			addSubTask(new SaveMiltipleDocuments(modifiedDocs, k == SaveProjectTaskKind_SaveProjectAndDocumentsAskEach));		
		}
	}
}

void SaveProjectTask::run() {
    if (proj!=NULL && !url.isEmpty()) {
        log.info(tr("Saving project %1").arg(url));
		saveProjectFile(stateInfo, proj, url);
	}
}


Task::ReportResult SaveProjectTask::report() {
	if (!stateInfo.hasErrors() && url == proj->getProjectURL()) {
		proj->setModified(false);
	}
	proj->unlockState(lock);
    delete lock;
    lock = NULL;
	return Task::ReportResult_Finished;
}


static QString map2String(const QVariantMap& map) {
	QByteArray a;
	QVariant v(map);
	QDataStream s(&a, QIODevice::WriteOnly);
	s << v;
	QString res(a.toBase64());
	return res;
}

void SaveProjectTask::saveProjectFile(TaskStateInfo& ts, Project* p, QString url) {
	//TODO: use Project!
	ProjectImpl* pi = qobject_cast<ProjectImpl*>(p);
	assert(pi);
	QDomDocument xmlDoc("GB2PROJECT");

	QDomElement projectElement = xmlDoc.createElement("gb2project");
	projectElement.setAttribute("name", pi->getProjectName());

	//save documents
	foreach(Document* gbDoc, pi->getDocuments()) {
        gbDoc->getDocumentFormat()->updateFormatSettings(gbDoc);

		//save document info
		QDomElement docElement = xmlDoc.createElement("document");
		docElement.setAttribute("url", gbDoc->getURL());
		docElement.setAttribute("io-adapter", gbDoc->getIOAdapterFactory()->getAdapterId());
		DocumentFormat* f = gbDoc->getDocumentFormat();
		QString formatId = f->getFormatId();
		docElement.setAttribute("format", formatId);
        docElement.setAttribute("readonly", gbDoc->hasUserModLock() ? 1 : 0);
        StateLock* l = gbDoc->getDocumentModLock(DocumentModLock_FORMAT_AS_INSTANCE);
        if (l!=NULL) {
            docElement.setAttribute("format-lock", 1);
        }
        if (!gbDoc->getGHintsMap().isEmpty()) {
            QString hintsStr = map2String(gbDoc->getGHintsMap());
            QDomText hintsNode = xmlDoc.createCDATASection(hintsStr);
            docElement.appendChild(hintsNode);
        }
        //now save unloaded objects info
        foreach(GObject* obj, gbDoc->getObjects()) {
            QDomElement objElement = xmlDoc.createElement("object");
            UnloadedObjectInfo info(obj);
            objElement.setAttribute("name", info.name);
            objElement.setAttribute("type", info.type);
            if (!info.hints.isEmpty()) {
                QString hintsStr = map2String(info.hints);
                QDomText hintsNode = xmlDoc.createCDATASection(hintsStr);
                objElement.appendChild(hintsNode);
            }
            docElement.appendChild(objElement);
        }
        projectElement.appendChild(docElement);
	}

	//save views
	foreach(GObjectViewState* view, pi->getGObjectViewStates()) {
		//save document info
		QDomElement viewElement = xmlDoc.createElement("view");
		viewElement.setAttribute("factory", view->getViewFactoryId());
		viewElement.setAttribute("viewName", view->getViewName());
		viewElement.setAttribute("stateName", view->getStateName());
		QString dataStr = map2String(view->getStateData());
		QDomText textNode = xmlDoc.createCDATASection(dataStr);
		viewElement.appendChild(textNode);

		projectElement.appendChild(viewElement);
	}

	xmlDoc.appendChild(projectElement);

	QByteArray rawData = xmlDoc.toByteArray();
	//	printf(">>%s", xmlDoc.toString().toStdString().c_str());

	QFile f(url);
	if (f.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
		qint64 s = f.write(rawData);
		f.close();
		if (s!=rawData.size()) {
            ts.setError(Translations::errorWritingFile(url));
		}
	} else {
        ts.setError(Translations::errorOpeningFileWrite(url));
	}
}


//////////////////////////////////////////////////////////////////////////
/// LoadProjectTask
LoadProjectTask::LoadProjectTask(const QString& _url) 
: Task(tr("load_project_task_name"), TaskFlag_None), proj(NULL), url(_url)
{
	xmlDoc = new QDomDocument();
}

LoadProjectTask::~LoadProjectTask() {
	if (proj) {
		delete proj;
	}
	delete xmlDoc;
}

void LoadProjectTask::run() {
    log.details(tr("Loading project from: %1").arg(url));
	loadXMLProjectModel(url, stateInfo, *xmlDoc);
}

Task::ReportResult LoadProjectTask::report() {
	if (!stateInfo.hasErrors()) {
        proj = createProjectFromXMLModel(url, *xmlDoc, stateInfo);
        if (proj!=NULL) {
            log.info(tr("Project loaded: %1").arg(url));
        }
	}
	return Task::ReportResult_Finished;
}


static QVariantMap string2Map(const QString& string, bool emptyMapIfError) {
    Q_UNUSED(emptyMapIfError);

	QDataStream s(QByteArray::fromBase64(string.toAscii()));
	QVariant res(QVariant::Map);
	s >> res;
    if (res.type() == QVariant::Map) {
        return res.toMap();
    }
    assert(emptyMapIfError);
    return QVariantMap();
}

void LoadProjectTask::loadXMLProjectModel(const QString& url, TaskStateInfo& si, QDomDocument& doc) {
	assert(doc.isNull());

	QFile f(url);
	if (!f.open(QIODevice::ReadOnly)) {
		si.setError(  tr("error_open_file %1").arg(url) );
		return;
	}
	QByteArray  xmlData = f.readAll();
	f.close();

	bool res = doc.setContent(xmlData);
	if (!res) {
		si.setError(  tr("error_invalid_content %1").arg(url) );
		doc.clear();
	}
	if (doc.doctype().name()!="GB2PROJECT") {
		si.setError(  tr("error_invalid_content %1").arg(url) );
		doc.clear();
	}
}

Project* LoadProjectTask::createProjectFromXMLModel(const QString& pURL, const QDomDocument& xmlDoc, TaskStateInfo& si) {
    Q_UNUSED(si); //TODO: report about errors using si!!!

    
	QDomElement projectElement = xmlDoc.documentElement();
	QString name = projectElement.attribute("name");

	ProjectImpl* pi = new ProjectImpl(name, pURL);
	
	//read all documents
	QDomNodeList documentList = projectElement.elementsByTagName("document");
	for(int i=0; i < documentList.size(); i++) {
		QDomNode dn = documentList.item(i);
		if (!dn.isElement()) {
			continue;
		}
		QDomElement docElement = dn.toElement();
		QString ioAdapterId = docElement.attribute("io-adapter");
		QString docURL = docElement.attribute("url");
		DocumentFormatId format = docElement.attribute("format");
        bool readonly = docElement.attribute("readonly").toInt() != 0;
        bool instanceLock = docElement.attribute("format-lock").toInt() != 0;
		IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioAdapterId);
		DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(format);
		assert(df!=NULL);
        QVariantMap fs = string2Map(docElement.text(), true);
        
        QList<UnloadedObjectInfo> unloadedObjects;
        QDomNodeList objectList = docElement.elementsByTagName("object");
        QSet<QString> objNames;
        for(int j=0; j < objectList.size(); j++) {
            QDomNode on = objectList.item(j);
            if (!on.isElement()) {
                continue;
            }
            QDomElement objElement = on.toElement();
            UnloadedObjectInfo info;
            info.name = objElement.attribute("name");
            info.type = objElement.attribute("type");
            info.hints = string2Map(objElement.text(), true);
            if (info.isValid() && !objNames.contains(info.name)) {
                unloadedObjects.append(info);
                objNames.insert(info.name);
            }
        }
        Document* d = new Document(df, iof, docURL, unloadedObjects, fs, instanceLock ? tr("last_loaded_state_was_locked_by_format") : QString());
        d->setUserModLock(readonly);
		pi->addDocument(d);
	}

	// read all saved views
	QDomNodeList viewElements = projectElement.elementsByTagName("view");
	for(int i=0;i<viewElements.size(); i++) {
		QDomNode n = viewElements.item(i);
		assert(n.isElement());
		if (!n.isElement()) {
			continue;
		}
		QDomElement viewElement = n.toElement();
		GObjectViewFactoryId id = viewElement.attribute("factory");
		QString viewName = viewElement.attribute("viewName");
		QString stateName = viewElement.attribute("stateName");
		QVariantMap map  = string2Map(viewElement.text(), false);
		GObjectViewState* state = new GObjectViewState(id, viewName, stateName, map);
		pi->addState(state);
	}
	pi->setModified(false);
    return pi;
}

RegisterProjectServiceTask::RegisterProjectServiceTask(Project* _proj) 
: Task(tr("Register project"), TaskFlag_NoRun), proj(_proj)
{
}

void RegisterProjectServiceTask::prepare() {
    ProjectServiceImpl* ps = new ProjectServiceImpl(proj);
    addSubTask(AppContext::getServiceRegistry()->registerServiceTask(ps));
}

OpenProjectTaskAndExecDialog::OpenProjectTaskAndExecDialog(QDialog *_dialog)
:OpenProjectTask(QString(), true)
, dialog(_dialog){

};

Task::ReportResult OpenProjectTaskAndExecDialog::report(){
    dialog->exec();
    delete dialog;
    return ReportResult_Finished;
}

} //namespace
