/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GUrl.h"

#include <QtCore/QDir>
#include <QtCore/QFileInfo>

namespace GB2 {

static QString makeFilePathCanonical(const QString& originalUrl) {
	//ensure that name is canonical
	QString result = originalUrl;
	if (result.startsWith("file://")) {
		result = result.mid(QString("file://").length());
	}
    
#ifdef Q_OS_WIN 
    if (result.startsWith("/")) {
        result = result.mid(1);
    }
#endif

    if ((!result.startsWith("/") && (result.length() <= 1 || result.at(1) != ':')) || result.contains('\\')) { //not absolute
		result = QFileInfo(result).absoluteFilePath();
    }
    
	if (result.length() > 1 && result.at(1) == ':') { //make drive letter upper case
		result = result[0].toUpper() + result.mid(1);
	}

    
    //replace all '//' pairs with single '/' character
#ifdef Q_OS_WIN //check if to keep shared folder prefix on windows
    bool keepSharedPrefix = result.startsWith("//");
#else 
    bool keepSharedPrefix = false;
#endif

    while (result.contains("//")) {
        result.replace("//", "/");
    }
    if (keepSharedPrefix) {
        assert(result.startsWith("/"));
        result = "/" + result;
    }


    if (result.contains("..")) {
		QString canonicalPath = QFileInfo(result).canonicalFilePath();
		if (!canonicalPath.isEmpty()) {
			result = canonicalPath;				
		} else {
			//TODO: try to make it canonical manually
		}
	}
	return result;
}

GUrlType GUrl::getURLType(const QString& rawUrl) {
	GUrlType result = GUrl_File;
	if (rawUrl.startsWith("http://") || rawUrl.startsWith("https://")) {
		result = GUrl_Http;
	} else if (rawUrl.startsWith("ftp://")) {
		result = GUrl_Ftp;
    } else if( rawUrl.startsWith( VirtualFileSystem::URL_PREFIX ) ) {
        result = GUrl_VFSFile;
    }
	return result;
}

// constructs url specified by string. The type is parsed
GUrl::GUrl(const QString& _urlString) {
	urlString = _urlString;
	type = getURLType(urlString);
	if (type == GUrl_File) {
		urlString = makeFilePathCanonical(urlString);
	}
}


// constructs url specified by string. The type provided as param
GUrl::GUrl(const QString& _urlString, const GUrlType _type) {
	urlString = _urlString;
	type = _type;
	if (type == GUrl_File) {
		urlString = makeFilePathCanonical(urlString);
	}
}

GUrl::GUrl(const GUrl& anotherUrl) {
	urlString = anotherUrl.getURLString();
	type = anotherUrl.getType();
}

bool GUrl::operator ==(const GUrl& url) const {
	return urlString == url.getURLString();
}

bool GUrl::operator !=(const GUrl& url) const {
	return !(*this == url);
}

static QString path(const GUrl* url) {
	//TODO: parse HTTP and other formats for path part
    QString result;
    if( url->isVFSFile() ) {
        return result;
    }
    result = url->getURLString();
	return result;
}

QString GUrl::dirPath() const {
	QString result;
    if( isVFSFile() ) {
        return result;
    }
    result = QFileInfo(path(this)).absoluteDir().absolutePath();
	return result;
}

QString GUrl::fileName() const {
    QString result;
    if( isVFSFile() ) {
        return result;
    }
	result = QFileInfo(path(this)).fileName();
	return result;
}

QString GUrl::baseFileName() const {
    QString result;
    if( isVFSFile() ) {
        QStringList args = urlString.split( VirtualFileSystem::URL_NAME_SEPARATOR, QString::SkipEmptyParts, Qt::CaseSensitive );
        if( 2 == args.size() ) {
            result = QFileInfo( args.at( 1 ) ).baseName();
        }
    } else {
        result = QFileInfo(path(this)).baseName();
    }
	return result;
}

QString GUrl::lastFileSuffix() const {
    QString result;
    if( isVFSFile() ) {
        return result;
    }
    result = QFileInfo(path(this)).suffix();
	return result;
}

QString GUrl::completeFileSuffix() const {
    QString result;
    if( isVFSFile() ) {
        return result;
    }
    result = QFileInfo(path(this)).completeSuffix();
	return result;
}


static bool registerMetas()  {
	qRegisterMetaType<GUrl>("GUrl");
	qRegisterMetaTypeStreamOperators<GUrl>("GB2::GUrl");

	return true;
}

QDataStream &operator<<(QDataStream &out, const GUrl &myObj) {
	out << myObj.getURLString() << myObj.getType();
	return out;
}

QDataStream &operator>>(QDataStream &in, GUrl &myObj) {
	QString urlString; in >> urlString;
	int t; in >> t; 
	GUrlType type = (GUrlType)t;
	myObj = GUrl(urlString, type);
	return in;
}


bool GUrl::registerMeta  = registerMetas();


}//namespace
