/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QUrl>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include "NetworkConfiguration.h"

namespace GB2
{

static const char * SETTINGS_HTTP_PROXY_HOST = "network_settings/http_proxy/host";
static const char * SETTINGS_HTTP_PROXY_PORT = "network_settings/http_proxy/port";
static const char * SETTINGS_HTTP_PROXY_USER = "network_settings/http_proxy/user";
static const char * SETTINGS_HTTP_PROXY_PASSWORD = "network_settings/http_proxy/password";
static const char * SETTINGS_HTTP_PROXY_ENABLED = "network_settings/http_proxy/enabled";
static const char * SETTINGS_PROXY_EXCEPTED_URLS = "network_settings/proxy_exc_urls";
static const char * SETTINGS_PROXY_EXCEPTED_URLS_ENABLED = "network_settings/proxy_exc_urls_enabled";

NetworkConfiguration::NetworkConfiguration() {
    Settings * s = AppContext::getSettings();
    pc.excepted_addr_enabled = s->getValue( SETTINGS_PROXY_EXCEPTED_URLS_ENABLED ).toBool();
    pc.excepted_addr = s->getValue( SETTINGS_PROXY_EXCEPTED_URLS ).toStringList();

    QString httpProxyHost = s->getValue( SETTINGS_HTTP_PROXY_HOST ).toString();
    int httpProxyPort = s->getValue( SETTINGS_HTTP_PROXY_PORT ).toInt();
    
    if( !httpProxyHost.isEmpty() && httpProxyPort ) {
        QNetworkProxy httpProxy( QNetworkProxy::HttpProxy, httpProxyHost, httpProxyPort );

        QString user = s->getValue( SETTINGS_HTTP_PROXY_USER ).toString();
        if (!user.isEmpty())  {
            QByteArray passwordEncoded = s->getValue( SETTINGS_HTTP_PROXY_PASSWORD ).toByteArray();
            QString passwordDecoded = QByteArray::fromBase64(passwordEncoded);
            httpProxy.setUser(user);
            httpProxy.setPassword(passwordDecoded);
        }

        pc.proxyz[QNetworkProxy::HttpProxy] = httpProxy;
        pc.proxyz_usage[QNetworkProxy::HttpProxy] = s->getValue( SETTINGS_HTTP_PROXY_ENABLED ).toBool();
    }
}

NetworkConfiguration::~NetworkConfiguration() {
    Settings * s = AppContext::getSettings();
    s->setValue( SETTINGS_PROXY_EXCEPTED_URLS_ENABLED, pc.excepted_addr_enabled );
    s->setValue( SETTINGS_PROXY_EXCEPTED_URLS, pc.excepted_addr );

    QNetworkProxy httpP = getProxy( QNetworkProxy::HttpProxy );
    
    if( !httpP.hostName().isEmpty() ) {
        s->setValue( SETTINGS_HTTP_PROXY_HOST, httpP.hostName() );
        s->setValue( SETTINGS_HTTP_PROXY_PORT, httpP.port() );
        s->setValue( SETTINGS_HTTP_PROXY_USER, httpP.user());
        s->setValue( SETTINGS_HTTP_PROXY_PASSWORD, httpP.password().toAscii().toBase64());
        s->setValue( SETTINGS_HTTP_PROXY_ENABLED, isProxyUsed(QNetworkProxy::HttpProxy) );
        
    }
}

int NetworkConfiguration::addProxy( const QNetworkProxy & p )
{
    int ret = !pc.proxyz.contains( p.type() );
    pc.proxyz.insert( p.type(), p );
    pc.proxyz_usage.insert( p.type(), false ); //needs explicit enabling
    return ret;
}

QNetworkProxy NetworkConfiguration::getProxyByUrl( const QUrl & url ) const
{
    Proxy_t prtype = url2type( url );
    if( pc.proxyz.contains( prtype ) ) {
        assert( pc.proxyz_usage.contains(prtype) );
        if( pc.proxyz_usage[prtype] ) {
            return ( pc.excepted_addr_enabled && pc.excepted_addr.contains( url.toString() ) ? 
                QNetworkProxy() : pc.proxyz[prtype] );
        }
    }
    return QNetworkProxy();
}

QNetworkProxy NetworkConfiguration::getProxy( Proxy_t prtype ) const {
    return (pc.proxyz.contains( prtype ) ? pc.proxyz[prtype] : QNetworkProxy() );
}

void NetworkConfiguration::removeProxy( Proxy_t prtype ) {
    pc.proxyz.remove( prtype );
}

bool NetworkConfiguration::isProxyUsed( Proxy_t prtype ) const {
    return pc.proxyz_usage.contains( prtype ) ? pc.proxyz_usage[prtype] : false;
}

void NetworkConfiguration::setExceptionsList( const QStringList & exc_addr ) {
    pc.excepted_addr = exc_addr;
}

void NetworkConfiguration::setProxyUsed( Proxy_t prtype, bool flag ) {
    if( pc.proxyz_usage.contains(prtype) ) {
        pc.proxyz_usage[prtype] = flag;
    }
}

Proxy_t NetworkConfiguration::url2type( const QUrl & url ) {
    if( "http" == url.scheme() ) {
        return QNetworkProxy::HttpProxy;
    } 
    if( "ftp" == url.scheme() ) {
        return QNetworkProxy::FtpCachingProxy;
    }
    assert( false );
    return QNetworkProxy::NoProxy;
}

void NetworkConfiguration::copyFrom(const NetworkConfiguration& image) {
    pc = image.pc;
}

} //namespace
