/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_SUBST_MATRIX_H
#define _GB2_SUBST_MATRIX_H

#include <core_api/core_api.h>
#include <core_api/DNAAlphabet.h>

#include <QtCore/QVarLengthArray>

namespace GB2 {

class GB2_COREAPI_EXPORT SScore {
public:
    char c1;
    char c2;
    float score;
    SScore(char _c1, char _c2, float _score) : c1(_c1), c2(_c2), score(_score){}
};

// Substitution, Scoring or Weight matrix model. 
// Example: Blosum70, PAM200, VTML200
class GB2_COREAPI_EXPORT SMatrix {	
public:
    SMatrix(const QString& name, DNAAlphabet* alphabet, 
                const QList<SScore>& rawMatrix, 
                const QString& description = QString());

    //constructs empty anonymous matrix
    SMatrix() { alphabet = NULL, minChar = 0; maxChar = 0; charsInRow = 0;}

    bool isEmpty() const {return scores.size() == 0;}

    float getScore(char c1, char c2) const;
    
    void setScore(char c1, char c2, float score);

    const QString& getName()  const {return name;}

    const QString& getDescription()  const {return description;}

    DNAAlphabet* getAlphabet() const {return alphabet;}

    float getMinScore() const {return minScore;}

    float getMaxScore() const {return maxScore;}

    //TODO: make this class serializable
    QVariant toQVariant() const;

    static SMatrix fromQVariant(const QVariant& v);

private:
    int getScoreIdx(char c1, char c2) const;
    void copyCharValues(char src, char dst);

    QString                 name;
    QString                 description;

    DNAAlphabet*            alphabet;
    QVarLengthArray<float>  scores;          //TODO: make scores integer ?
    char                    minChar;   // used for optimization of scores size. Minimal character in the alphabet.
    char                    maxChar;   // used for optimization of scores size. Maximum character in the alphabet.
    int                     charsInRow;
    float                   minScore;
    float                   maxScore;
    QByteArray              validCharacters; // used only for debugging now. Use array but not Set since number of characters is low
};


inline float SMatrix::getScore(char c1, char c2) const {
    int idx = getScoreIdx(c1, c2);
    return scores[idx];
}

inline void SMatrix::setScore(char c1, char c2, float score) {
    int idx = getScoreIdx(c1, c2);
    scores[idx] = score;
}

inline int SMatrix::getScoreIdx(char c1, char c2) const {
    assert(validCharacters.contains(c1));
    assert(validCharacters.contains(c2));
    int d1 = c1 - minChar;
    int d2 = c2 - minChar;
    assert(d1 < charsInRow && d2 < charsInRow && d1 >= 0 && d2 >= 0);
    return d1 * charsInRow + d2;
}

} // namespace

#endif
