/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <datatype/DNASequence.h>
#include <core_api/DNAAlphabet.h>
#include <datatype/MAlignment.h>

#include "RemoteMachine.h"
#include "SerializeUtils.h"

#include <QtCore/QFile>

namespace GB2 {

// remote machine settings

const QString SerializeUtils::REMOTE_MACHINE_VERSION_1_0 = "Remote machine v1.0: ";

QString SerializeUtils::serializeRemoteMachineSettings( RemoteMachineSettings * machine ) {
    assert( NULL != machine );
    return REMOTE_MACHINE_VERSION_1_0 + machine->getProtocolId() + " " + machine->serialize();
}

bool SerializeUtils::deserializeRemoteMachineSettingsFromFile( const QString & machinePath, RemoteMachine ** machine ) {
    QFile file( machinePath );
    if( !file.open( QIODevice::ReadOnly ) ) {
        return NULL;
    }
    
    QString data = file.readAll();
    if( !deserializeRemoteMachineSettings( data, machine ) || NULL == machine ) {
        return false;
    }
    return true;
}

bool SerializeUtils::deserializeRemoteMachineSettingsFromFile( const QString & machinePath, RemoteMachineSettings ** settings ) {
    QFile file( machinePath );
    if( !file.open( QIODevice::ReadOnly ) ) {
        return NULL;
    }
    if( !deserializeRemoteMachineSettings( file.readAll(), settings ) || NULL == settings ) { return false; }
    return true;
}

static QString getProtocolId( const QString & data ) {
    QString res;
    QList< ProtocolInfo* > infos = AppContext::getProtocolInfoRegistry()->getProtocolInfos();
    foreach( ProtocolInfo * pi, infos ) {
        assert( NULL != pi );
        if( data.startsWith( pi->getId() ) ) {
            res = pi->getId();
            break;
        }
    }
    return res;
}

bool SerializeUtils::deserializeRemoteMachineSettings( const QString & data, RemoteMachineSettings ** ret, QString * retProtoId ) {
    if( NULL == ret ) {
        return false;
    }
    *ret = NULL;
    
    if( data.startsWith( REMOTE_MACHINE_VERSION_1_0 ) ) {
        QString machineData = data.mid( REMOTE_MACHINE_VERSION_1_0.size() );
        QString protoId = getProtocolId( machineData );
        ProtocolInfo * protoInfo = AppContext::getProtocolInfoRegistry()->getProtocolInfo( protoId );
        if( NULL == protoInfo ) {
            return false;
        }
        
        *ret = protoInfo->getRemoteMachineFactory()->createSettings( machineData.mid( protoId.size() + 1 ) );
        if( NULL != retProtoId ) {
            *retProtoId = protoId;
        }
        return NULL == *ret ? false : true;
    } else { /* there is only one version of remote machines in this version of ugene */
        return false;
    }
    
    return false; // unreachable code
}

bool SerializeUtils::deserializeRemoteMachineSettings( const QString & data, RemoteMachine ** machine ) {
    if( NULL == machine ) {
        return false;
    }
    *machine = NULL;
    
    RemoteMachineSettings * settings = NULL;
    QString protoId;
    if( !deserializeRemoteMachineSettings( data, &settings, &protoId ) ) {
        assert( NULL == settings );
        return false;
    }
    assert( NULL != settings );
    *machine = AppContext::getProtocolInfoRegistry()->getProtocolInfo( protoId )->getRemoteMachineFactory()->createInstance( settings );
    delete settings;
    
    return NULL == *machine ? false : true;
}

} // GB2
