/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNAAlphabetRegistryImpl.h"
#include "DNATranslationImpl.h"

#include <util_text/TextUtils.h>

#include <core_api/AppContext.h>
#include <core_api/Settings.h>

#include <QtCore/QDir>
#include <QtCore/QFile>
#include <QtCore/QTextStream>

#include <QtCore/QXmlStreamReader>

namespace GB2 {

const QString BaseDNATranslationIds::NUCL_DNA_DEFAULT_COMPLEMENT("NUCL_DNA_DEFAULT_COMPLEMENT");
const QString BaseDNATranslationIds::NUCL_RNA_DEFAULT_COMPLEMENT("NUCL_RNA_DEFAULT_COMPLEMENT");

const QString BaseDNATranslationIds::NUCL_DNA_EXTENDED_COMPLEMENT("NUCL_DNA_EXTENDED_COMPLEMENT");
const QString BaseDNATranslationIds::NUCL_RNA_EXTENDED_COMPLEMENT("NUCL_RNA_EXTENDED_COMPLEMENT");

#define DATA_DIR_KEY                QString("back_translation")
#define DATA_FILE_KEY               QString("back_translation/lastFile")
#define DEFAULT_ORGANISM_FILE       QString("tables.xml")

static void fill3To1(QList<Mapping3To1<char> >& map, QMap<DNATranslationRole,QList<Triplet> >& codons, 
    const DNAAlphabet* srcAl, const DNAAlphabet* dstAl,
    const char* amino, const char* role, const char* n1, const char* n2, const char* n3)
{
    assert(srcAl->isNucleic()); Q_UNUSED(srcAl);
    assert(dstAl->isAmino()); Q_UNUSED(dstAl);

    int len = strlen(amino);
    assert(len == (int)strlen(role) && len == (int)strlen(n1) && len == (int)strlen(n2) && len == (int)strlen(n3));
    for(int i=0; i<len; i++) {
        char res = amino[i];
        char c1 = n1[i];
        char c2 = n2[i];
        char c3 = n3[i];
        assert(dstAl->contains(res));
        assert(srcAl->contains(c1));
        assert(srcAl->contains(c2));
        assert(srcAl->contains(c3));
        Triplet t(c1, c2, c3);
        Mapping3To1<char> m(t, res);
        map.append(m);
        if (role[i] == 'M') codons[DNATranslationRole_Start].append(t);
        else if (role[i] == 'L') codons[DNATranslationRole_Start_Alternative].append(t);
        else if (amino[i] == '*') codons[DNATranslationRole_Stop].append(t);
    }
}

static void fill1To3(BackTranslationRules& map, 
    const DNAAlphabet* srcAl, const DNAAlphabet* dstAl,
    const char* amino, const int* prob, 
    const char* n1, const char* n2, const char* n3)
{
    assert(srcAl->isAmino()); Q_UNUSED(srcAl);
    assert(dstAl->isNucleic()); Q_UNUSED(dstAl);

    TripletP t('N', 'N', 'N', 100);
    map.map.append(t);
    map.index['-'] = map.map.size();
    TripletP dash('-', '-', '-', 100);
    map.map.append(dash);
    int len = strlen(amino);
    assert(len == (int)strlen(n1) && len == (int)strlen(n2) && len == (int)strlen(n3));

    QByteArray alph = srcAl->getAlphabetChars();
    QList<TripletP> v;
    int sump;
    foreach (char c, alph) {
        v.clear();
        sump = 0;
        for(int i=0; i<len; i++) {
            if (amino[i] != c) continue;
            char c1 = n1[i];
            char c2 = n2[i];
            char c3 = n3[i];
            int p = prob[i];
            sump += p;
#ifdef DEBUG
            char src = amino[i];
            assert(srcAl->contains(src));
            assert(dstAl->contains(c1));
            assert(dstAl->contains(c2));
            assert(dstAl->contains(c3));
            assert((0 <= p) && (p <= 100));
#endif
            TripletP t(c1, c2, c3, p);
            v.append(t);
        }
        if (v.empty()) {
            if (c != '-') {
                map.index[c] = 0;
                map.index[c] = 1;
            }
        } else {
            map.index[c] = map.map.size();
            qSort(v);
            v.first().p += (100 - sump);
            foreach (TripletP t, v) {
                map.map.append(t);
            }
        }
    }
}

static bool parseCutFile(const QString& url, char* amino, int* prob, char* n1, char* n2, char* n3) {
    QFile organismFile(url);
    if (organismFile.open(QFile::ReadOnly)) {
        QTextStream data(&organismFile);
        QString line;
        QStringList parsedData;
        QByteArray buf(4, '\0');
        int pos = 0;
        bool ok = true;
        do {
            line = data.readLine();
            if (line.isEmpty() || line.startsWith("#")) continue;
            parsedData = line.split(QRegExp("\\s"), QString::SkipEmptyParts);
            if ( parsedData.size() != 5 ) return false;
            if ( parsedData[0].length() != 3 ) return false;
            buf = parsedData[0].toAscii();
            n1[pos] = buf[0];
            n2[pos] = buf[1];
            n3[pos] = buf[2];
            if ( parsedData[1].length() != 1 ) return false;
            buf = parsedData[1].toAscii();
            amino[pos] = buf[0];
            double pr = parsedData[2].toDouble(&ok);
            if (!ok) return false;
            prob[pos] = qRound(pr);
            pos++;
            amino[pos] = n1[pos] = n2[pos] = n3[pos] = '\0';
        } while (!line.isNull());
        return true;
    }
    return false;
}

void DNAAlphabetRegistryImpl::reg4tables(const char* amino, const char* role, const char* n1, const char* n2, const char* n3,
    const QString& id, const QString& name) 
{
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT);
        DNAAlphabet* dstAlphabet = findById(BaseDNAAlphabetIds::AMINO_DEFAULT);

        QList<Mapping3To1<char> > map;
        QMap<DNATranslationRole,QList<Triplet> > codons;
        fill3To1(map, codons, srcAlphabet, dstAlphabet, amino, role, n1, n2, n3);

        DNATranslation* t = new DNATranslation3to1Impl(id, name, 
            srcAlphabet, dstAlphabet, map, 'X', codons);
        treg->registerDNATranslation(t);
    }

    //extended NUCL DNA to AMINO -> all extended symbols lead to "unknown"
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_DNA_EXTENDED);
        DNAAlphabet* dstAlphabet = findById(BaseDNAAlphabetIds::AMINO_DEFAULT);

        QList<Mapping3To1<char> > map;
        QMap<DNATranslationRole,QList<Triplet> > codons;
        fill3To1(map, codons, srcAlphabet, dstAlphabet, amino, role, n1, n2, n3);

        DNATranslation* t = new DNATranslation3to1Impl(id, name, 
            srcAlphabet, dstAlphabet, map, 'X', codons);
        treg->registerDNATranslation(t);
    }
    QByteArray an1(n1);
    const char* rn1 = an1.replace('T', 'U');
    QByteArray an2(n2);
    const char* rn2 = an2.replace('T', 'U');
    QByteArray an3(n3);
    const char* rn3 = an3.replace('T', 'U');
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_RNA_DEFAULT);
        DNAAlphabet* dstAlphabet = findById(BaseDNAAlphabetIds::AMINO_DEFAULT);

        QList<Mapping3To1<char> > map;
        QMap<DNATranslationRole,QList<Triplet> > codons;
        fill3To1(map, codons, srcAlphabet, dstAlphabet, amino, role, rn1, rn2, rn3);

        DNATranslation* t = new DNATranslation3to1Impl(id, name, 
            srcAlphabet, dstAlphabet, map, 'X', codons);
        treg->registerDNATranslation(t);
    }

    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_RNA_EXTENDED);
        DNAAlphabet* dstAlphabet = findById(BaseDNAAlphabetIds::AMINO_DEFAULT);

        QList<Mapping3To1<char> > map;
        QMap<DNATranslationRole,QList<Triplet> > codons;
        fill3To1(map, codons, srcAlphabet, dstAlphabet, amino, role, rn1, rn2, rn3);

        DNATranslation* t = new DNATranslation3to1Impl(id, name, 
            srcAlphabet, dstAlphabet, map, 'X', codons);
        treg->registerDNATranslation(t);
    }
}

void DNAAlphabetRegistryImpl::regPtables(const char* amino, const int* prob, const char* n1, const char* n2, const char* n3,
    const QString& id, const QString& name) 
{
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::AMINO_DEFAULT);
        DNAAlphabet* dstAlphabet = findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT);

        BackTranslationRules map;
        fill1To3(map, srcAlphabet, dstAlphabet, amino, prob, n1, n2, n3);

        DNATranslation* t = new DNATranslation1to3Impl(id, name, 
            srcAlphabet, dstAlphabet, map);
        treg->registerDNATranslation(t);
    }
}

#define CASE_OFFSET ('a'-'A')
#define MAP(a, b) \
map[int(a)] = b; \
if (!srcAlphabet->isCaseSensitive()) { \
map[int(a) + CASE_OFFSET]=(b)+CASE_OFFSET; \
}


void DNAAlphabetRegistryImpl::initBaseTranslations() {

    //default NUCL DNA complement
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT);
        DNAAlphabet* dstAlphabet = srcAlphabet;

        QByteArray map = TextUtils::createMap(srcAlphabet->getMap(), 'N');

        MAP('A','T');
        MAP('C','G');
        MAP('G','C');
        MAP('T','A');

        DNATranslation* t = new DNATranslation1to1Impl(BaseDNATranslationIds::NUCL_DNA_DEFAULT_COMPLEMENT, 
            tr("nucl_dna_default_to_nucl_compl"), 
            srcAlphabet, dstAlphabet, map);
        treg->registerDNATranslation(t);
    }

    //default NUCL RNA complement
    {
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_RNA_DEFAULT);
        DNAAlphabet* dstAlphabet = srcAlphabet;

        QByteArray map = TextUtils::createMap(srcAlphabet->getMap(), 'N');

        MAP('A','U');
        MAP('C','G');
        MAP('G','C');
        MAP('U','A');

        DNATranslation* t = new DNATranslation1to1Impl(BaseDNATranslationIds::NUCL_RNA_DEFAULT_COMPLEMENT,
            tr("nucl_rna_default_to_nucl_compl"), 
            srcAlphabet, dstAlphabet, map);
        treg->registerDNATranslation(t);
    }


    //extended NUCL DNA complement
    {
        //source: http://www.geneinfinity.org/sp_nucsymbols.html
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_DNA_EXTENDED);
        DNAAlphabet* dstAlphabet = srcAlphabet;

        QByteArray map = TextUtils::createMap(srcAlphabet->getMap(), 'N');
        MAP('A','T');
        MAP('C','G');
        MAP('G','C');
        MAP('T','A');
        MAP('M','K');
        MAP('R','Y');
        MAP('W','W');
        MAP('S','S');
        MAP('Y','R');
        MAP('K','M');
        MAP('V','B');
        MAP('H','D');
        MAP('D','H');
        MAP('B','V');

        DNATranslation* t = new DNATranslation1to1Impl(BaseDNATranslationIds::NUCL_DNA_EXTENDED_COMPLEMENT, 
            tr("nucl_dna_extended_to_nucl_compl"), 
            srcAlphabet, dstAlphabet, map);
        treg->registerDNATranslation(t);
    }


    //extended NUCL RNA complement
    {
        //source: http://www.geneinfinity.org/sp_nucsymbols.html
        DNAAlphabet* srcAlphabet = findById(BaseDNAAlphabetIds::NUCL_RNA_EXTENDED);
        DNAAlphabet* dstAlphabet = srcAlphabet;

        QByteArray map = TextUtils::createMap(srcAlphabet->getMap(), 'N');
        MAP('A','U');
        MAP('C','G');
        MAP('G','C');
        MAP('U','A');
        MAP('M','K');
        MAP('R','Y');
        MAP('W','W');
        MAP('S','S');
        MAP('Y','R');
        MAP('K','M');
        MAP('V','B');
        MAP('H','D');
        MAP('D','H');
        MAP('B','V');

        DNATranslation* t = new DNATranslation1to1Impl(BaseDNATranslationIds::NUCL_RNA_EXTENDED_COMPLEMENT,
            tr("nucl_rna_extended_to_nucl_compl"), 
            srcAlphabet, dstAlphabet, map);
        treg->registerDNATranslation(t);
    }

    // source: http://www.ncbi.nlm.nih.gov/Taxonomy/Utils/wprintgc.cgi
    // 1. The Standard Code (transl_table=1)
    reg4tables(
        "FFLLSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "---L---------------L---------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(1),
        tr("dna2amino 1 Standard"));

    //2. The Vertebrate Mitochondrial Code (transl_table=2)
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSS**VVVVAAAADDEEGGGG",
        "--------------------------------LLLM---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(2),
        tr("dna2amino 2 Vertebrate Mitochondrial"));

    //3. The Yeast Mitochondrial Code (transl_table=3)
    reg4tables(
        "FFLLSSSSYY**CCWWTTTTPPPPHHQQRRRRIIMMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "----------------------------------LM----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(3),
        tr("dna2amino 3 Yeast Mitochondrial"));

    //4. The Mold, Protozoan, and Coelenterate Mitochondrial Code and the Mycoplasma/Spiroplasma Code (transl_table=4) 
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "--LL---------------L------------LLLM---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(4),
        tr("dna2amino 4 Protozoan Mitochondrial"));

    //5. The Invertebrate Mitochondrial Code (transl_table=5) 
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSSSSVVVVAAAADDEEGGGG",
        "---L----------------------------LLLM---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(5),
        tr("dna2amino 5 Invertebrate Mitochondrial"));

    //6. The Ciliate, Dasycladacean and Hexamita Nuclear Code (transl_table=6) 
    reg4tables(
        "FFLLSSSSYYQQCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(6),
        tr("dna2amino 6 Ciliate Nuclear"));

    //9. The Echinoderm and Flatworm Mitochondrial Code (transl_table=9) 
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
        "-----------------------------------M---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(9),
        tr("dna2amino 9 Flatworm Mitochondrial"));

    //10. The Euplotid Nuclear Code (transl_table=10) 
    reg4tables(
        "FFLLSSSSYY**CCCWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(10),
        tr("dna2amino 10 Euplotid Nuclear"));

    //11. The Bacterial and Plant Plastid Code (transl_table=11)
    reg4tables(
        "FFLLSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "---L---------------L------------LLLM---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(11),
        tr("dna2amino 11 Plant Plastid"));

    //12. The Alternative Yeast Nuclear Code (transl_table=12) 
    reg4tables(
        "FFLLSSSSYY**CC*WLLLSPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-------------------L---------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(12),
        tr("dna2amino 12 Alternative Yeast Nuclear")); 

    //13. The Ascidian Mitochondrial Code (transl_table=13) 
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSSGGVVVVAAAADDEEGGGG",
        "---L------------------------------LM---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(13),
        tr("dna2amino 13 Ascidian Mitochondrial"));

    //14. The Alternative Flatworm Mitochondrial Code (transl_table=14)
    reg4tables(
        "FFLLSSSSYYY*CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(14),
        tr("dna2amino 14 Alternative Flatworm Mitochondrial"));

    //15. Blepharisma Nuclear Code (transl_table=15) 
    reg4tables(
        "FFLLSSSSYY*QCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(15),
        tr("dna2amino 15 Blepharisma Nuclear")); 


    //16. Chlorophycean Mitochondrial Code (transl_table=16) 
    reg4tables(
        "FFLLSSSSYY*LCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(16),
        tr("dna2amino 16 Chlorophycean Mitochondrial"));

    //21. Trematode Mitochondrial Code (transl_table=21) 
    reg4tables(
        "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
        "-----------------------------------M---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(21),
        tr("dna2amino 21 Trematode Mitochondrial"));

    //22. Scenedesmus obliquus mitochondrial Code (transl_table=22) 
    reg4tables(
        "FFLLSS*SYY*LCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "-----------------------------------M----------------------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(22),
        tr("dna2amino 22 Scenedesmus obliquus Mitochondrial"));


    //23. Thraustochytrium Mitochondrial Code (transl_table=23) 
    reg4tables(
        "FF*LSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
        "--------------------------------L--M---------------L------------",
        "TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG",
        "TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG",
        "TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG",
        DNATranslationID(23),
        tr("dna2amino 23 Thraustochytrium Mitochondrial"));

    char amino[65], n1[65], n2[65], n3[65];
    int prob[64];

    QString dir = QDir::searchPaths( PATH_PREFIX_DATA ).first() + "/back_translation/";
    QString lastOrganismFile = AppContext::getSettings()->getValue(DATA_FILE_KEY).toString();
    if (lastOrganismFile.isEmpty() || !QFile::exists(lastOrganismFile)) {
        lastOrganismFile = dir + DEFAULT_ORGANISM_FILE;
        AppContext::getSettings()->setValue(DATA_FILE_KEY, lastOrganismFile);
    }
    QXmlStreamReader xml;
    QFile file(lastOrganismFile);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        return;
    }
    xml.setDevice(&file);
    QString currentType;
    QString currentName;
    QString fileName;
    QString readableName;

    while (!xml.atEnd() && !xml.hasError()) {
        QXmlStreamReader::TokenType tt = xml.readNext();
        if (tt == QXmlStreamReader::StartDocument) {
            continue;
        }
        if (tt == QXmlStreamReader::StartElement) {
            if (xml.name() == "CodonFrequencyTableList") {
                continue;
            }
            if (xml.name() == "Folder" && xml.attributes().value("name") == "Codon Frequency Tables") {
                continue;
            }
            if (xml.name() == "Folder") {
                currentType = xml.attributes().value("name").toString();
                continue;
            }
            if (xml.name() == "CodonFrequencyTable") {
                currentName = xml.attributes().value("name").toString();
                QStringList buf = currentName.split(".");
                buf.removeLast();
                readableName = buf.join(".").replace("_", " ");
                fileName = currentType + "/" + currentName;
                if (parseCutFile(dir + fileName, amino, prob, n1, n2, n3))
                    regPtables(amino, prob, n1, n2, n3, "BackTranslation/" + fileName, readableName);
            }
        }
    } 
}

} //namespace
