/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_PDB_FORMAT_H_
#define _GB2_PDB_FORMAT_H_

#include <core_api/DocumentFormats.h>
#include <core_api/DocumentModel.h>
#include <QtCore/QSharedDataPointer>

namespace GB2 {

class IOAdapter;
class BioStruct3D;
class AnnotationTableObject;
class DNASequenceObject;
class AtomData;
typedef QSharedDataPointer<AtomData> SharedAtom;


class GB2_COREAPI_EXPORT  PDBFormat : public DocumentFormat {
	Q_OBJECT
public:
	PDBFormat(QObject* p);
	virtual DocumentFormatId getFormatId() const {return BaseDocumentFormats::PLAIN_PDB;}
	virtual const QString& getFormatName() const {return formatName;}

	virtual Document* loadDocument(IOAdapter* io, TaskStateInfo& ti, const QVariantMap& fs, DocumentLoadMode mode = DocumentLoadMode_Whole);

	virtual bool checkRawData(const QByteArray& rawData) const;
    static int getElementNumberByName(const QByteArray& elementName);
    inline static double getAtomCovalentRadius(int atomicNumber);
    static QChar getAcronymByName(const QByteArray& name);
    static QHash<QByteArray, int> createAtomNumMap();
    static void calculateBonds(BioStruct3D& bioStruct);
    static Document* createDocumentFromBioStruct3D(BioStruct3D &bioStruct, DocumentFormat* format, IOAdapterFactory* iof, const GUrl& url, TaskStateInfo& ti, const QVariantMap& fs );


private:
    static const int NUM_ELEMENTS = 120; 
    QString formatName;
	static QHash<QByteArray,int> atomNumMap;
	static QHash<QByteArray, QChar> acronymNameMap;
	static double atomRadiusTable[NUM_ELEMENTS];
    
    void initUtilityMaps();
    void fillBioStruct3DAnnotationTable(AnnotationTableObject* ao, const BioStruct3D& bioStruct);
	
	
	class PDBParser { 
	private:
		// Data
        IOAdapter *io;
		QString currentPDBLine;
		QChar currentChainId;
		int currentMoleculeId;
        int currentModelId;
        bool flagMultipleModels;
		bool flagAtomRecordPresent, flagSequenceRecordPresent;
		QHash<QChar, int> chainIndexMap;
        QMap<int, int> residueStartIndexMap;
        QMap<int, int> unreferencedResidueStartIndexMap;
        QVariantMap currentMoleculeDescr;
		// Methods
        QByteArray getSpecValue(const QByteArray& specLine, const QByteArray& valueName);
        int getChainIndexByName(const QChar& chainId);
        void parseHeader(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseCompound(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseDBRef(BioStruct3D& biostruct, TaskStateInfo& ti);
        void parseSequence(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseSecondaryStructure(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseHet(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseAtomConnections(BioStruct3D& biostruct, TaskStateInfo& ti);
		void parseAtom(BioStruct3D& biostruct, TaskStateInfo& ti);
        void parseModel(BioStruct3D& biostruct, TaskStateInfo& ti);
        QByteArray getNextSpecLine();
	public:
		PDBParser(IOAdapter* io);
		void parseBioStruct3D(BioStruct3D& biostruct, TaskStateInfo& ts);
	};

};

double PDBFormat::getAtomCovalentRadius( int atomicNumber )
{
    Q_ASSERT(atomicNumber < NUM_ELEMENTS);
    return atomRadiusTable[atomicNumber];

}

}//namespace

#endif
