/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QAction>
#include <QtGui/QMenu>
#include <QtGui/QMessageBox>

#include <core_api/AppContext.h>
#include <core_api/DocumentFormats.h>
#include <core_api/DocumentModel.h>
#include <core_api/DocumentUtils.h>
#include <core_api/GUrlUtils.h>
#include <core_api/MainWindow.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/DnaAssemblyAlgRegistry.h>
#include <core_api/ProjectModel.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <selection/GObjectSelection.h>
#include <util_ov_msaedit/MSAEditorFactory.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/AddDocumentTask.h>
#include <document_format/DocumentFormatUtils.h>

#include "DnaAssemblyUtils.h"
#include "DnaAssemblyDialog.h"


namespace GB2 {

DnaAssemblySupport::DnaAssemblySupport()
{
    QAction* showDlgAction = new QAction( tr("DNA Assembly"), this );
    showDlgAction->setIcon(QIcon(":core/images/align.png"));
    connect( showDlgAction, SIGNAL( triggered() ), SLOT( sl_showDnaAssemblyDialog() ) );
    AppContext::getMainWindow()->getMenuManager()->getTopLevelMenu( MWMENU_TOOLS )->addAction( showDlgAction );
}

void DnaAssemblySupport::sl_showDnaAssemblyDialog() 
{
    DnaAssemblyAlgRegistry* registry = AppContext::getDnaAssemblyAlgRegistry();
    if (registry->getTaskNameList().isEmpty()) {
        QMessageBox::information(QApplication::activeWindow(), tr("DNA Assembly"),
            tr("There are no algorithms for DNA assembly available.\nPlease, check your plugin list.") );
        return;
    }
        
    DnaAssemblyDialog dlg(registry, QApplication::activeWindow());
    if (dlg.exec()) {
        DnaAssemblyToRefTaskSettings s;
        s.refSeqUrl = dlg.getRefSeqUrl();
        s.algName = dlg.getAlgorithmName();
        s.resultFileName = dlg.getResultFileName();
        s.setCustomSettings( dlg.getCustomSettings() );
        Task* assemblyTask = new DnaAssemblyMultiTask( s, dlg.getShortReadUrls(), true );
        AppContext::getTaskScheduler()->registerTopLevelTask(assemblyTask);
    }
 
}

//////////////////////////////////////////////////////////////////////////

DnaAssemblyLoadShortReadsTask::DnaAssemblyLoadShortReadsTask( DnaAssemblyToRefTaskSettings& s, const QList<GUrl>& sr )
: Task(tr("PrepareAssemblyToRefTask"), TaskFlags_NR_FOSCOE), settings(s), shortReads(sr)
{
    
}

void DnaAssemblyLoadShortReadsTask::prepare()
{
     foreach(const GUrl& url, shortReads) {
        QList<DocumentFormat*> detectedFormats = DocumentFormatUtils::detectFormat(url);    
        if (!detectedFormats.isEmpty()) {
            IOAdapterFactory* factory = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
            DocumentFormat* format = detectedFormats.first();
            Task* loadTask = new LoadDocumentTask(format->getFormatId(), url, factory);
            addSubTask(loadTask);
        }  
     }
}

Task::ReportResult DnaAssemblyLoadShortReadsTask::report()
{
    if (settings.shortReads.isEmpty()) {
        setError("No short reads are loaded");
    }
    return ReportResult_Finished;
}

QList<Task*> DnaAssemblyLoadShortReadsTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> res;
    if (!subTask->hasErrors()) {
        LoadDocumentTask* task = qobject_cast<LoadDocumentTask*> (subTask);
        if (task == NULL) {
            return res;
        }
        Document* doc = task->getDocument();
        QList<GObject*> seqObjects = doc->findGObjectByType(GObjectTypes::SEQUENCE);

        foreach(GObject* obj, seqObjects) {
            DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
            assert(dnaObj != NULL);
            //TODO: check for common alphabet?
            settings.shortReads.append(dnaObj->getDNASequence());
        }
    }
    return res;
}

//////////////////////////////////////////////////////////////////////////

DnaAssemblyMultiTask::DnaAssemblyMultiTask( const DnaAssemblyToRefTaskSettings& s, const QList<GUrl>& shortReads, bool view )
: Task("DnaAssemblyMultiTask", TaskFlags_NR_FOSCOE), settings(s), shortReadUrls(shortReads), openView(view)
{
    setMaxParallelSubtasks(1);
}

void DnaAssemblyMultiTask::prepare()
{
    // load short reads and prepare settings
    addSubTask(new DnaAssemblyLoadShortReadsTask(settings, shortReadUrls));
    // perform assembly
    QString algName = settings.algName;
    DnaAssemblyToRefTaskFactory* factory = AppContext::getDnaAssemblyAlgRegistry()->getTaskFactory(algName);
    assert(factory);
    if (factory == NULL) {
        setError(QString("Algorithm %1 is not found").arg(algName));
        return;
    }
    assemblyToRefTask = factory->createTaskInstance(settings);
    addSubTask(assemblyToRefTask);
}

Task::ReportResult DnaAssemblyMultiTask::report()
{
    return ReportResult_Finished;
}

QList<Task*> DnaAssemblyMultiTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> subTasks;
    if (subTask->hasErrors()) {
        return subTasks;
    }

    if ( (subTask == assemblyToRefTask) && openView) {
        GUrl refSeqUrl = settings.refSeqUrl.getURLString();
        assert(!refSeqUrl.isEmpty());
        
        QString alnFileName = settings.resultFileName.getURLString();
        if (alnFileName.isEmpty()) {
            alnFileName = GUrlUtils::rollFileName(refSeqUrl.dirPath() + "/" + refSeqUrl.baseFileName() + ".sam", DocumentUtils::getNewDocFileNameExcludesHint());
        }
        
        MAlignmentObject *newObj = new MAlignmentObject(assemblyToRefTask->getResult());
        DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::SAM);
        IOAdapterFactory *iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
        doc = new Document(df, iof, alnFileName, QList<GObject*>()<<newObj);
        Project* proj = AppContext::getProject();
        if (proj == NULL) {
            QList<GUrl> emptyList;
            subTasks.append(  AppContext::getProjectLoader()->openProjectTask(emptyList, false) );
        }
        addDocumentTask = new AddDocumentTask(doc);
        subTasks.append(addDocumentTask);
    } else if ( subTask == addDocumentTask) {
        GObjectViewFactoryRegistry* reg = AppContext::getObjectViewFactoryRegistry();
        GObjectViewFactory* f = reg->getFactoryById(MSAEditorFactory::ID);
        assert(f != NULL);
        const QList<GObject*>& docObjects = doc->getObjects();
        GObjectSelection os; os.addToSelection(docObjects);
        MultiGSelection ms; ms.addSelection(&os);
        assert(f->createViewTask(ms));
        Task* openViewTask = f->createViewTask(ms);
        subTasks.append(openViewTask);
    }

    return subTasks;
}

const MAlignment& DnaAssemblyMultiTask::getAssemblyResult()
{
    Q_ASSERT(assemblyToRefTask != NULL);
    return assemblyToRefTask->getResult();
}

} // GB2

