/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemovePartFromSequenceDialogController.h"

#include <QtGui/QMessageBox>
#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtGui/QFileDialog>

#include <document_format/GenbankLocationParser.h>

#include <core_api/DocumentFormats.h>
#include <core_api/Log.h>
#include <core_api/AppContext.h>

#include <datatype/AnnotationData.h>


namespace GB2{

static LogCategory log(ULOG_CAT_USER_INTERFACE);

RemovePartFromSequenceDialogController::RemovePartFromSequenceDialogController(LRegion _toDelete, LRegion _source, QWidget *p )
:QDialog(p), filter(""), toDelete(_toDelete), source(_source){
    setupUi(this);

    connect(browseButton, SIGNAL(clicked()), SLOT(sl_browseButtonClicked()));
    AnnotationData ad;
    ad.location.append(toDelete);
    removeLocationEdit->setText(Genbank::LocationParser::buildLocationString(&ad));

    connect(formatBox, SIGNAL(currentIndexChanged(int)), this, SLOT(sl_indexChanged(int)));

    formatBox->addItem("FASTA", BaseDocumentFormats::PLAIN_FASTA);
    formatBox->addItem("Genbank", BaseDocumentFormats::PLAIN_GENBANK);
    connect(mergeAnnotationsBox, SIGNAL(toggled(bool)), this, SLOT(sl_mergeAnnotationsToggled(bool)));
    sl_indexChanged(0);
}

void RemovePartFromSequenceDialogController::accept(){
    QString genbankRegion = removeLocationEdit->text();
    QList<LRegion> locs;
    bool b;
    Genbank::LocationParser::parseLocation(genbankRegion.toAscii().constData(), genbankRegion.length(), b, locs);
    if(locs.size() > 1){
        QMessageBox::critical(this, this->windowTitle(), tr("There must be only one region to delete"));
        return;
    }    
    if(locs.size() < 1){
        QMessageBox::critical(this, this->windowTitle(), tr("Unable to parse region to delete"));
        return;
    }
    toDelete = locs.first();

    if (toDelete == source) {
        QMessageBox::critical(this, this->windowTitle(), tr("Cannot remove the whole sequence"));
        return;
    }
    
    if (toDelete.startPos < source.startPos || toDelete.endPos() > source.endPos()){
        QMessageBox::critical(this, this->windowTitle(), tr("Region to delete is out of sequence bounds"));
        return;
    }



    this->close();
    QDialog::accept();
}

void RemovePartFromSequenceDialogController::sl_browseButtonClicked(){
    LastOpenDirHelper h;
    
    h.url = QFileDialog::getSaveFileName(this, tr("Select file to save..."), h.dir, filter);
    filepathEdit->setText(h.url);
    sl_indexChanged(formatBox->currentIndex());
}

RemovePartFromSequenceTask::AnnotationStrategyForRemove RemovePartFromSequenceDialogController::getStrategy(){
    if(removeRB->isChecked()){
        return RemovePartFromSequenceTask::AnnotationStrategyForRemove_Remove;
    }else{
        assert(resizeRB->isChecked());
        return RemovePartFromSequenceTask::AnnotationStrategyForRemove_Resize;
    }
}

void RemovePartFromSequenceDialogController::sl_indexChanged( int index){
    DocumentFormatId currentId = (formatBox->itemData(index)).toString();
    filter = DialogUtils::prepareDocumentsFileFilter(currentId, false);
    DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(currentId);
    QString newExt = df->getSupportedDocumentFileExtensions().first();
    QString filepath = filepathEdit->text();
    if (filepath.isEmpty()){
        return;
    }
    QFileInfo fi(filepath);
    filepathEdit->setText(fi.absoluteDir().absolutePath() + "/" + fi.baseName() + "." + newExt);
}

void RemovePartFromSequenceDialogController::sl_mergeAnnotationsToggled( bool ){
    if(mergeAnnotationsBox->isChecked()){
        formatBox->removeItem(formatBox->findText("FASTA"));
    }else{
        formatBox->addItem("FASTA", BaseDocumentFormats::PLAIN_FASTA);
    }
    sl_indexChanged(formatBox->findText("Genbank"));
}


}//ns
