/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationsTreeView.h"

#include "AnnotatedDNAView.h"
#include "ADVConstants.h"
#include "ADVSequenceObjectContext.h"

#include "EditAnnotationDialogController.h"

#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <core_api/Settings.h>
#include <core_api/Timer.h>
#include <core_api/DBXRefRegistry.h>

#include <document_format/GenbankLocationParser.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/DNASequenceObject.h>
#include <selection/AnnotationSelection.h>
#include <util_gui/ProjectTreeController.h>
#include <util_gui/ProjectTreeItemSelectorDialog.h>
#include <util_gui/GUIUtils.h>
#include <util_gui/EditQualifierDialog.h>
#include <util_tasks/LoadRemoteDocumentTask.h>

#include <QtCore/QFileInfo>
#include <QtGui/QVBoxLayout>
#include <QtGui/QPainter>
#include <QtGui/QMenu>
#include <QtGui/QClipboard>
#include <QtGui/QToolTip>
#include <QtGui/QMessageBox>
#include <QtGui/QHeaderView>
#include <QtGui/QLineEdit>

/* TRANSLATOR GB2::AnnotationsTreeView */

namespace GB2 {

class TreeSorter {
public:
    TreeSorter(AnnotationsTreeView* t) : w(t) {
        w->setSortingEnabled(false);
    }
    virtual ~TreeSorter() {
        w->setSortingEnabled(true);
    }
    AnnotationsTreeView* w;
};

#define SETTINGS_ROOT QString("view_adv/annotations_tree_view/")
#define COLUMN_SIZES QString("columnSizes")

const QString AnnotationsTreeView::annotationMimeType = "application/x-annotation-features";

AnnotationsTreeView::AnnotationsTreeView(AnnotatedDNAView* _ctx) : ctx(_ctx){
    lastMB = Qt::NoButton;
    lastClickedColumn = 0;

    tree = new QTreeWidget(this);
    tree->setObjectName("tree_widget");

    tree->setSortingEnabled(true);
    tree->sortItems(0, Qt::AscendingOrder);
    
    tree->setColumnCount(2);
    headerLabels << tr("annotation_name_label") << tr("qualifier_value_label");

    tree->setHeaderLabels(headerLabels);
    tree->setUniformRowHeights(true);
    tree->setSelectionMode(QAbstractItemView::ExtendedSelection);
    tree->viewport()->installEventFilter(this);
    tree->setMouseTracking(true);

    connect(tree, SIGNAL(itemEntered(QTreeWidgetItem*, int)), SLOT(sl_itemEntered(QTreeWidgetItem*, int)));
    connect(tree, SIGNAL(itemClicked(QTreeWidgetItem*, int)), SLOT(sl_itemClicked(QTreeWidgetItem*, int)));
    connect(tree, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)), SLOT(sl_itemDoubleClicked(QTreeWidgetItem*, int)));
    connect(tree, SIGNAL(itemExpanded(QTreeWidgetItem*)), SLOT(sl_itemExpanded(QTreeWidgetItem*)));
    //connect(tree, SIGNAL(itemActivated(QTreeWidgetItem*, int)), SLOT(sl_itemActivated(QTreeWidgetItem*, int)));

    QVBoxLayout *layout = new QVBoxLayout();
    layout->setMargin(0);
    layout->addWidget(tree);
    setLayout(layout);
    
    restoreWidgetState();

    connect(ctx, SIGNAL(si_buildPopupMenu(GObjectView*, QMenu*)), SLOT(sl_onBuildPopupMenu(GObjectView*, QMenu*)));
    connect(ctx, SIGNAL(si_annotationObjectAdded(AnnotationTableObject*)), SLOT(sl_onAnnotationObjectAdded(AnnotationTableObject*)));
    connect(ctx, SIGNAL(si_annotationObjectRemoved(AnnotationTableObject*)), SLOT(sl_onAnnotationObjectRemoved(AnnotationTableObject*)));
    foreach(AnnotationTableObject* obj, ctx->getAnnotationObjects()) {
        sl_onAnnotationObjectAdded(obj);
    }
    connectAnnotationSelection();
    connectAnnotationGroupSelection();
    connect(tree, SIGNAL(itemSelectionChanged()), SLOT(sl_onItemSelectionChanged()));

    connect(AppContext::getAnnotationsSettingsRegistry(),
        SIGNAL(si_annotationSettingsChanged(const QStringList&)),
        SLOT(sl_onAnnotationSettingsChanged(const QStringList&)));

#define SORT_INTERVAL 500
    sortTimer.setInterval(SORT_INTERVAL);
    sortTimer.setSingleShot(true);
    connect(&sortTimer, SIGNAL(timeout()), SLOT(sl_sortTree()));

    addColumnIcon = QIcon(":core/images/add_column.png");
    removeColumnIcon = QIcon(":core/images/remove_column.png");
    
    addAnnotationObjectAction = new QAction(tr("add_annotation_object"), this);
    connect(addAnnotationObjectAction, SIGNAL(triggered()), SLOT(sl_onAddAnnotationObjectToView()));

    removeObjectsFromViewAction = new QAction(tr("Selected object from view"), this);
    removeObjectsFromViewAction->setShortcut(QKeySequence(Qt::SHIFT| Qt::Key_Delete));
    removeObjectsFromViewAction->setShortcutContext(Qt::WidgetShortcut);
    connect(removeObjectsFromViewAction, SIGNAL(triggered()), SLOT(sl_removeObjectFromView()));
    tree->addAction(removeObjectsFromViewAction);

    removeAnnsAndQsAction = new QAction(tr("Selected annotations and qualifiers"), this);
    removeAnnsAndQsAction->setShortcut(QKeySequence(Qt::Key_Delete));
    removeAnnsAndQsAction->setShortcutContext(Qt::WindowShortcut);
    connect(removeAnnsAndQsAction, SIGNAL(triggered()), SLOT(sl_removeAnnsAndQs()));
    tree->addAction(removeAnnsAndQsAction);

    copyQualifierAction = new QAction(tr("copy_qualifier_value"), this);
    connect(copyQualifierAction, SIGNAL(triggered()), SLOT(sl_onCopyQualifierValue()));

    copyQualifierURLAction = new QAction(tr("copy_qualifier_url"), this);
    connect(copyQualifierURLAction, SIGNAL(triggered()), SLOT(sl_onCopyQualifierURL()));

    toggleQualifierColumnAction = new QAction(tr("toggle_qual_column"), this);
    connect(toggleQualifierColumnAction, SIGNAL(triggered()), SLOT(sl_onToggleQualifierColumn()));

    removeColumnByHeaderClickAction = new QAction(tr("hide_qual_column"), this);
    removeColumnByHeaderClickAction->setIcon(removeColumnIcon);
    connect(removeColumnByHeaderClickAction, SIGNAL(triggered()), SLOT(sl_onRemoveColumnByHeaderClick()));

    copyColumnTextAction = new QAction(tr("copy_column_text"), this);
    connect(copyColumnTextAction, SIGNAL(triggered()), SLOT(sl_onCopyColumnText()));

    copyColumnURLAction = new QAction(tr("copy_column_url"), this);
    connect(copyColumnURLAction, SIGNAL(triggered()), SLOT(sl_onCopyColumnURL()));

    renameAction = new QAction(tr("Edit item"), this);
    renameAction->setShortcut(QKeySequence(Qt::Key_F2));
    renameAction->setShortcutContext(Qt::WidgetShortcut);
    connect(renameAction, SIGNAL(triggered()), SLOT(sl_rename()));
    tree->addAction(renameAction);

    editAction = new QAction(tr("Edit qualifier"), this);
    editAction->setShortcut(QKeySequence(Qt::Key_F4));
    editAction->setShortcutContext(Qt::WidgetShortcut);
    connect(editAction, SIGNAL(triggered()), SLOT(sl_edit()));
    tree->addAction(editAction);

    viewAction = new QAction(tr("View qualifier"), this);
    viewAction->setShortcut(QKeySequence(Qt::Key_F3));
    viewAction->setShortcutContext(Qt::WidgetShortcut);
    connect(viewAction, SIGNAL(triggered()), SLOT(sl_edit()));
    tree->addAction(viewAction);

    addQualifierAction = new QAction(tr("Qualifier"), this);
    addQualifierAction->setShortcut(QKeySequence(Qt::Key_Insert));
    addQualifierAction->setShortcutContext(Qt::WidgetShortcut);
    connect(addQualifierAction, SIGNAL(triggered()), SLOT(sl_addQualifier()));
    tree->addAction(addQualifierAction);

/*    cutAnnotationsAction = new QAction(tr("cut_annotations"), this);
//    cutAnnotationsAction->setShortcut(QKeySequence(Qt::K));
    connect(cutAnnotationsAction, SIGNAL(triggered()), SLOT(sl_cutAnnotations));
    tree->addAction(cutAnnotationsAction);

    copyAnnotationsAction = new QAction(tr("copy_annotations"), this);
//    copyAnnotationsAction->setShortcut(QKeySequence(Qt::K));
    connect(copyAnnotationsAction, SIGNAL(triggered()), SLOT(sl_copyAnnotations));
    tree->addAction(copyAnnotationsAction);

    pasteAnnotationsAction = new QAction(tr("paste_annotations"), this);
//    pasteAnnotationsAction->setShortcut(QKeySequence(Qt::K));
    connect(pasteAnnotationsAction, SIGNAL(triggered()), SLOT(sl_pasteAnnotations));
    tree->addAction(pasteAnnotationsAction);*/

    updateState();

    tree->setAcceptDrops(true);
}

void AnnotationsTreeView::restoreWidgetState() { 
    QStringList geom = AppContext::getSettings()->getValue(SETTINGS_ROOT + COLUMN_SIZES, QStringList()).toStringList();
    if (geom.isEmpty()) {
        tree->setColumnWidth(0, 300);
        tree->setColumnWidth(1, 300);
    } else {
        for (int i=0;i<geom.size(); i++) {
            const QString& w = geom.at(i);
            bool ok = false;
            int width  = w.toInt(&ok);
            if (ok) {
                tree->setColumnWidth(i, width);
            }
        }
    }
}

void AnnotationsTreeView::saveWidgetState() {
    QStringList geom;
    for (int i=0; i < tree->columnCount(); i++) {
        QString s = QString::number(tree->columnWidth(i));
        geom.append(s);
    }
    AppContext::getSettings()->setValue(SETTINGS_ROOT+COLUMN_SIZES, geom);
}


AVGroupItem* AnnotationsTreeView::findGroupItem(const AnnotationGroup* g) const {
    GTIMER(c2,t2,"AnnotationsTreeView::findGroupItem");
    if (g->getParentGroup() == NULL) {
        for (int i=0, n = tree->topLevelItemCount(); i<n; i++) {
            AVItem* item = static_cast<AVItem*>(tree->topLevelItem(i));
            assert(item->type == AVItemType_Group);
            AVGroupItem* groupItem = static_cast<AVGroupItem*>(item);
            if (groupItem->group == g) {
                return groupItem;
            }
        }
    } else {
        AVGroupItem* parentGroupItem = findGroupItem(g->getParentGroup());
        if (parentGroupItem != NULL) {
            for(int i = 0, n = parentGroupItem->childCount(); i < n; i++) {
                AVItem* item = static_cast<AVItem*>(parentGroupItem->child(i));
                if (item->type != AVItemType_Group) {
                    continue;
                }
                AVGroupItem* gItem = static_cast<AVGroupItem*>(item);
                if (gItem->group == g) {
                    return gItem;
                }
            }
        }
    }
    return NULL;
}

AVAnnotationItem* AnnotationsTreeView::findAnnotationItem(const AVGroupItem* groupItem, const Annotation* a) const {
    GTIMER(c2,t2,"AnnotationsTreeView::findAnnotationItem");
    for(int i = 0, n = groupItem->childCount(); i < n; i++) {
        AVItem* item = static_cast<AVItem*>(groupItem->child(i));
        if (item->type != AVItemType_Annotation) {
            continue;
        }
        AVAnnotationItem* aItem = static_cast<AVAnnotationItem*>(item);
        if (aItem->annotation == a) {
            return aItem;
        }
    }
    return NULL;
}

AVAnnotationItem* AnnotationsTreeView::findAnnotationItem(const AnnotationGroup* g, const Annotation* a) const {
    AVGroupItem* groupItem = findGroupItem(g);
    if (groupItem == NULL) {
        return NULL;
    }
    return findAnnotationItem(groupItem, a);
}

QList<AVAnnotationItem*> AnnotationsTreeView::findAnnotationItems(const Annotation* a) const {
    QList<AVAnnotationItem*> res;
    foreach(AnnotationGroup* g, a->getGroups()) {
        AVGroupItem* gItem = findGroupItem(g);
        AVAnnotationItem* aItem = findAnnotationItem(gItem, a);
        res.append(aItem);
    }
    return res;
}

void AnnotationsTreeView::connectAnnotationSelection() {
    connect(ctx->getAnnotationsSelection(), 
        SIGNAL(si_selectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>& )), 
        SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>&)));
}

void AnnotationsTreeView::connectAnnotationGroupSelection() {
    connect(ctx->getAnnotationsGroupSelection(), 
        SIGNAL(si_selectionChanged(AnnotationGroupSelection*, const QList<AnnotationGroup*>&, const QList<AnnotationGroup*>& )), 
        SLOT(sl_onAnnotationGroupSelectionChanged(AnnotationGroupSelection*, const QList<AnnotationGroup*>&, const QList<AnnotationGroup*>&)));
}

void AnnotationsTreeView::sl_onItemSelectionChanged() {
    AnnotationSelection* as = ctx->getAnnotationsSelection();
    as->disconnect(this);
    as->clear();

    AnnotationGroupSelection* ags = ctx->getAnnotationsGroupSelection();
    ags->disconnect(this);
    ags->clear();


    QList<QTreeWidgetItem*> items = tree->selectedItems();
    foreach(QTreeWidgetItem* i, items) {
        AVItem* item  = static_cast<AVItem*>(i);
        if (item->type == AVItemType_Annotation) {
            AVAnnotationItem* aItem = static_cast<AVAnnotationItem*>(item);
            assert(aItem->annotation!=NULL);
            assert(aItem->annotation->getGObject()!=NULL);
            as->addToSelection(aItem->annotation);
        } else if (item->type == AVItemType_Group) {
            AVGroupItem* gItem = static_cast<AVGroupItem*>(item);
            assert(gItem->group!=NULL);
            ags->addToSelection(gItem->group);
        }
    }
    connectAnnotationSelection();
    connectAnnotationGroupSelection();
    updateState();
}

void AnnotationsTreeView::sl_onAnnotationSelectionChanged(AnnotationSelection* s, const QList<Annotation*>& added, const QList<Annotation*>& removed) {
    Q_UNUSED(s);
    tree->disconnect(this, SIGNAL(sl_onItemSelectionChanged()));

    foreach(Annotation* a, removed) {
        foreach(AnnotationGroup* g, a->getGroups()) {
            AVAnnotationItem* item = findAnnotationItem(g, a);
            if (item->isSelected()) {
                item->setSelected(false);
            }
        }
    }
    AVAnnotationItem* toVisible = NULL;
    foreach(Annotation* a, added) {
        foreach(AnnotationGroup* g, a->getGroups()) {
            AVAnnotationItem* item = findAnnotationItem(g, a);
            if (!item->isSelected()) {
                item->setSelected(true);
                for (QTreeWidgetItem* p = item->parent(); p!=NULL; p = p->parent()) {
                    if (!p->isExpanded()) {
                        p->setExpanded(true);
                    }
                }
            }
            toVisible = item;
        }
    }
    
    connect(tree, SIGNAL(itemSelectionChanged()), SLOT(sl_onItemSelectionChanged ()));

    //make item visible if special conditions are met
    if (toVisible!=NULL && added.size() == 1) {
        tree->scrollToItem(toVisible, QAbstractItemView::EnsureVisible);
    }
    updateState();
}


void AnnotationsTreeView::sl_onAnnotationGroupSelectionChanged(AnnotationGroupSelection* s, 
                                                               const QList<AnnotationGroup*>& added, 
                                                               const QList<AnnotationGroup*>& removed) 
{
    Q_UNUSED(s);

    foreach(AnnotationGroup* g, removed) {
        AVGroupItem* item = findGroupItem(g);
        if (item->isSelected()) {
            item->setSelected(false);
        }
    }

    foreach(AnnotationGroup* g, added) {
        AVGroupItem* item = findGroupItem(g);
        if (!item->isSelected()) {
            item->setSelected(true);
        }
    }

    if (added.size() == 1) {
        AVGroupItem* item = findGroupItem(added.first());
        tree->scrollToItem(item, QAbstractItemView::EnsureVisible);
    }
}


void AnnotationsTreeView::sl_onAnnotationObjectAdded(AnnotationTableObject* obj) {
    GTIMER(c2,t2,"AnnotationsTreeView::sl_onAnnotationObjectAdded");
    TreeSorter ts(this);
    
    assert(findGroupItem(obj->getRootGroup()) == NULL);
    AVGroupItem* groupItem = buildGroupTree(NULL, obj->getRootGroup());
    tree->addTopLevelItem(groupItem);
    connect(obj, SIGNAL(si_onAnnotationsAdded(const QList<Annotation*>&)), SLOT(sl_onAnnotationsAdded(const QList<Annotation*>&)));
    connect(obj, SIGNAL(si_onAnnotationsRemoved(const QList<Annotation*>&)), SLOT(sl_onAnnotationsRemoved(const QList<Annotation*>&)));
    connect(obj, SIGNAL(si_onAnnotationModified(const AnnotationModification&)), SLOT(sl_onAnnotationModified(const AnnotationModification&)));

    connect(obj, SIGNAL(si_onGroupCreated(AnnotationGroup*)), SLOT(sl_onGroupCreated(AnnotationGroup*)));
    connect(obj, SIGNAL(si_onGroupRemoved(AnnotationGroup*, AnnotationGroup*)), SLOT(sl_onGroupRemoved(AnnotationGroup*, AnnotationGroup*)));
    connect(obj, SIGNAL(si_onGroupRenamed(AnnotationGroup*, const QString& )), SLOT(sl_onGroupRenamed(AnnotationGroup*, const QString& )));

    connect(obj, SIGNAL(si_modifiedStateChanged()), SLOT(sl_annotationObjectModifiedStateChanged()));
}

void AnnotationsTreeView::sl_onAnnotationObjectRemoved(AnnotationTableObject* obj) {
    TreeSorter ts(this);

    AVGroupItem* groupItem = findGroupItem(obj->getRootGroup());
    assert(groupItem!=NULL);
    delete groupItem;
    
    obj->disconnect(this);
}


void AnnotationsTreeView::sl_onAnnotationsAdded(const QList<Annotation*>& as) {
    GTIMER(c1,t1,"AnnotationsTreeView::sl_onAnnotationsAdded");
    TreeSorter ts(this);

    QSet<AVGroupItem*> toUpdate;
    foreach(Annotation* a, as) {
        foreach(AnnotationGroup* ag, a->getGroups()) {
            AVGroupItem* gi = findGroupItem(ag);
            if (gi!=NULL) {
                //if (findAnnotationItem(gi, a)== NULL) 
                {
                    buildAnnotationTree(gi, a);
                }
            } else {
                AnnotationGroup* childGroup = ag;
                while(true) {
                    gi = findGroupItem(childGroup->getParentGroup());
                    if (gi != NULL) {
                        break;
                    }
                    childGroup = childGroup->getParentGroup();
                }
                assert(gi!=NULL && childGroup!=NULL);
                buildGroupTree(gi, childGroup);
            }
            assert(gi!=NULL);
            toUpdate.insert(gi);
        }
    }
    GTIMER(c2,t2,"AnnotationsTreeView::sl_onAnnotationsAdded [updateVisual]");
    while (!toUpdate.isEmpty()) {
        AVGroupItem* i= *toUpdate.begin();
        toUpdate.remove(i);
        i->updateVisual();
        AVGroupItem* p = static_cast<AVGroupItem*>(i->parent());
        if (p!=NULL) {
            toUpdate.insert(p);
        }
    }
}

void AnnotationsTreeView::sl_onAnnotationsRemoved(const QList<Annotation*>& as) {
    TreeSorter ts(this);
    
    tree->disconnect(this, SIGNAL(sl_onItemSelectionChanged()));

    QSet<AVGroupItem*> groupsToUpdate;
    foreach(Annotation* a, as) {
        QList<AVAnnotationItem*> aItems = findAnnotationItems(a);
        assert(!aItems.isEmpty());
        foreach(AVAnnotationItem* ai, aItems) {
            groupsToUpdate.insert(static_cast<AVGroupItem*>(ai->parent()));
            delete ai;
        }
    }
    foreach(AVGroupItem* g, groupsToUpdate) {
        g->updateVisual();
    }

    connect(tree, SIGNAL(itemSelectionChanged()), SLOT(sl_onItemSelectionChanged ()));
    
    sl_onItemSelectionChanged();
}

void AnnotationsTreeView::sl_onAnnotationModified(const AnnotationModification& md) {
    switch(md.type) {
        case AnnotationModification_NameChanged: 
        case AnnotationModification_LocationChanged:
            {
                QList<AVAnnotationItem*> aItems = findAnnotationItems(md.annotation);
                assert(!aItems.isEmpty());
                foreach(AVAnnotationItem* ai, aItems) {
                    ai->updateVisual(ATVAnnUpdateFlag_BaseColumns);
                }
            }
            break;

        case AnnotationModification_QualifierRemoved:
            {
                const QualifierModification& qm = (const QualifierModification&)md;
                QList<AVAnnotationItem*> aItems  = findAnnotationItems(qm.annotation);
                foreach(AVAnnotationItem* ai, aItems) {
                    ai->removeQualifier(qm.qualifier);
                }
            }
            break;
        case AnnotationModification_QualifierAdded:
            {
                const QualifierModification& qm = (const QualifierModification&)md;
                QList<AVAnnotationItem*> aItems  = findAnnotationItems(qm.annotation);
                foreach(AVAnnotationItem* ai, aItems) {
                    ai->addQualifier(qm.qualifier);
                }
            }
            break;
        case AnnotationModification_AddedToGroup:
            {
                const AnnotationGroupModification& gmd = (const AnnotationGroupModification&)md;
                AVGroupItem* gi = findGroupItem(gmd.group);
                assert(gi!=NULL);
                buildAnnotationTree(gi, gmd.annotation);
            }
            break;

        case AnnotationModification_RemovedFromGroup:
            {
                const AnnotationGroupModification& gmd = (const AnnotationGroupModification&)md;
                AVAnnotationItem* ai = findAnnotationItem(gmd.group, gmd.annotation);
                assert(ai!=NULL);
                delete ai;

            }
            break;
    }
}

void AnnotationsTreeView::sl_onGroupCreated(AnnotationGroup* g) {
    AVGroupItem* pi = g->getParentGroup()== NULL ? NULL : findGroupItem(g->getParentGroup());
    buildGroupTree(pi, g);
}

void AnnotationsTreeView::sl_onGroupRemoved(AnnotationGroup* parent, AnnotationGroup* g) {
    AVGroupItem* pg  = findGroupItem(parent);
    assert(parent!=NULL && pg!=NULL);
    
    for(int i = 0, n = pg->childCount(); i < n; i++) {
        AVItem* item = static_cast<AVItem*>(pg->child(i));
        if (item->type == AVItemType_Group && (static_cast<AVGroupItem*>(item))->group == g) {
            delete item;
            break;
        }
    }
    pg->updateVisual();
}

void AnnotationsTreeView::sl_onGroupRenamed(AnnotationGroup* g, const QString& oldName) {
    Q_UNUSED(oldName);
    AVGroupItem* gi = findGroupItem(g);
    assert(gi!=NULL);
    gi->updateVisual();
}

AVGroupItem* AnnotationsTreeView::buildGroupTree(AVGroupItem* parentGroupItem, AnnotationGroup* g) {
    AVGroupItem* groupItem = new AVGroupItem(this, parentGroupItem, g);
    const QList<AnnotationGroup*>& subgroups = g->getSubgroups();
    foreach(AnnotationGroup* subgroup, subgroups) {
        buildGroupTree(groupItem, subgroup);
    }
    const QList<Annotation*>& annotations = g->getAnnotations();
    foreach(Annotation* a, annotations) {
        buildAnnotationTree(groupItem, a);
    }
    return groupItem;
}

AVAnnotationItem* AnnotationsTreeView::buildAnnotationTree(AVGroupItem* parentGroup, Annotation* a) {
    AVAnnotationItem* annotationItem = new AVAnnotationItem(parentGroup, a);
    const QVector<Qualifier>& qualifiers = a->getQualifiers();
    if (!qualifiers.isEmpty()) {
        annotationItem->setChildIndicatorPolicy(QTreeWidgetItem::ShowIndicator);
    }
    return annotationItem;
}

void AnnotationsTreeView::populateAnnotationQualifiers(AVAnnotationItem* ai){
    assert(ai->childIndicatorPolicy() == QTreeWidgetItem::ShowIndicator);
    assert(ai->childCount() == 0);
    const QVector<Qualifier>& qualifiers = ai->annotation->getQualifiers();
    foreach(const Qualifier& q, qualifiers) {
        AVQualifierItem* curQualifierItem = new AVQualifierItem(ai, q);
        Q_UNUSED(curQualifierItem);
    }
    ai->setChildIndicatorPolicy(QTreeWidgetItem::DontShowIndicatorWhenChildless);
}

void AnnotationsTreeView::sl_onAnnotationSettingsChanged(const QStringList& changedSettings) {
    foreach (const QString& name , changedSettings) {
        // clear icons cache with color associated with this name
        AVAnnotationItem::getIconsCache().remove(name);

        //update all top-level groups with the same annotation name -> track visible state
        for (int i=0; i<tree->topLevelItemCount(); i++) {
            AVGroupItem* top = static_cast<AVGroupItem*>(tree->topLevelItem(i));
            for (int j = 0; j < top->childCount(); j++) {
                AVItem* item = static_cast<AVItem*>(top->child(j));
                if (item->type == AVItemType_Group) {
                    AVGroupItem* level1 = static_cast<AVGroupItem*>(item);
                    if (level1->group->getGroupName() == name) {
                        level1->updateVisual();
                    }
                }
            }
            top->updateAnnotations(name, ATVAnnUpdateFlag_BaseColumns);
        }

        //update all annotations with this name -> track visible state and color
    }
}

void AnnotationsTreeView::updateColumnContextActions(AVItem* item, int col) {
    copyColumnTextAction->setEnabled(item!=NULL && (col >= 2 || (item->type == AVItemType_Annotation && col == 1)) && !item->text(col).isEmpty());
    copyColumnURLAction->setEnabled(item!=NULL && col >= 2 && item->isColumnLinked(col));
    if (!copyColumnTextAction->isEnabled()) {
        copyColumnTextAction->setText(tr("copy_column_text"));
    } else {
        QString colName;
        if (col >= 2) {
            assert(qColumns.size() > col - 2);
            colName = qColumns[col - 2];
            copyColumnTextAction->setText(tr("copy_column_'%1'_text").arg(colName));
        } else {
            AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(item);
            copyColumnTextAction->setText(tr("copy_'%1'_annotation_location").arg(ai->annotation->getAnnotationName()));
        }
    }

    if (!copyColumnURLAction->isEnabled()) {
        copyColumnURLAction->setText(tr("copy_column_url"));
    } else {
        assert(qColumns.size() > col - 2);
        QString colName = qColumns[col - 2];
        copyColumnURLAction->setText(tr("copy_column_'%1'_url").arg(colName));
    }
}

void AnnotationsTreeView::sl_onBuildPopupMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    
    // Add actions that not depends on the point popup is called
    adjustMenu(m);
    
    QPoint globalPos = QCursor::pos();
    QPoint treePos = tree->mapFromGlobal(globalPos);
    if (!tree->rect().contains(treePos)) {
        return;
    }
    
    // Check is popup is called for column header
    QHeaderView* header = tree->header();
    QPoint headerPoint = header->mapFromGlobal(globalPos);
    if (header->rect().contains(headerPoint)) {
        int idx = header->logicalIndexAt(headerPoint);
        if (idx >= 2) {
            assert(idx - 2 < qColumns.size());
            lastClickedColumn = idx;
            removeColumnByHeaderClickAction->setText(tr("hide_'%1'_column").arg(qColumns[lastClickedColumn-2]));
            QAction* first = m->actions().first();
            m->insertAction(first, removeColumnByHeaderClickAction);
            m->insertSeparator(first);
        }
        return;
    }

    //Ensure that item clicked is in the tree selection. Do not destroy multi-selection if present
    QList<QTreeWidgetItem*> selItems = tree->selectedItems();
    QPoint viewportPos = tree->viewport()->mapFromGlobal(globalPos);
    if (selItems.size() <= 1) {
        QTreeWidgetItem* item = tree->itemAt(viewportPos);
        if (item!=NULL) {
            if (selItems.size() == 1 && selItems.first()!=item) {
                tree->setItemSelected(selItems.first(), false);
            }
            tree->setItemSelected(item, true);
        }
    }
    
    //Update column sensitive actions that appears only in context menu
    selItems = tree->selectedItems();
    lastClickedColumn = tree->columnAt(viewportPos.x());
    updateColumnContextActions(selItems.size() == 1 ? static_cast<AVItem*>(selItems.first()) : static_cast<AVItem*>(NULL), lastClickedColumn);

    //Add active context actions to the top level menu
    QList<QAction*> contextActions;
    contextActions << copyQualifierAction << copyQualifierURLAction 
        << toggleQualifierColumnAction << copyColumnTextAction 
        << copyColumnURLAction << editAction
;//        << cutAnnotationsAction << copyAnnotationsAction << pasteAnnotationsAction;
    
    QMenu* copyMenu = GUIUtils::findSubMenu(m, ADV_MENU_COPY);
    foreach(QAction* a, contextActions) {
        if (a->isEnabled()) {
            copyMenu->addAction(a);
        }
    }
    int nActive = 0;
    QAction* first = m->actions().first();
    foreach(QAction* a, contextActions) {
        if (a->isEnabled()) {
            nActive++;
            m->insertAction(first, a);
        }
    }
    if (nActive > 0) {
        m->insertSeparator(first);
    }
}

void AnnotationsTreeView::adjustMenu(QMenu* m) const {
    QMenu* addMenu = GUIUtils::findSubMenu(m, ADV_MENU_ADD);
    assert(addMenu!=NULL);
    addMenu->addAction(addAnnotationObjectAction);
    addMenu->addAction(addQualifierAction);

    QMenu* removeMenu = GUIUtils::findSubMenu(m, ADV_MENU_REMOVE);
    assert(removeMenu!=NULL);
    removeMenu->addAction(removeObjectsFromViewAction);
    removeMenu->addAction(removeAnnsAndQsAction);
}

void AnnotationsTreeView::sl_onAddAnnotationObjectToView() {
    ProjectTreeControllerModeSettings s;
    s.objectTypesToShow.append(GObjectTypes::ANNOTATION_TABLE);
    AnnotationTableObjectConstraints ac;
    ac.sequenceSizeToFit = 0;//TODO: ctx->getDNASequenceContext()->getSequence().length();
    s.objectConstraints.append(&ac);
    s.groupMode = ProjectTreeGroupMode_Flat;
    foreach(GObject* o, ctx->getObjects()) {
        s.excludeObjectList.append(o);
    }
    QList<GObject*> objs = ProjectTreeItemSelectorDialog::selectObjects(s, this);
    foreach(GObject* obj, objs) {
        assert(obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE);
        ctx->addObject(obj);
    } 
}



static QList<AVGroupItem*> selectGroupItems(const QList<QTreeWidgetItem*>& items, TriState readOnly, TriState rootOnly) {
    QList<AVGroupItem*> res;
    foreach(QTreeWidgetItem* i, items) {
        AVItem* item = static_cast<AVItem*>(i);
        if (item->type == AVItemType_Group) {
            AVGroupItem* gItem = static_cast<AVGroupItem*>(item);
            if (rootOnly != TriState_Unknown) {
                bool groupIsRoot = gItem->parent() == NULL;
                if ( (rootOnly == TriState_Yes && !groupIsRoot) || (rootOnly==TriState_No && groupIsRoot)) {
                    continue;
                }
            }
            if (readOnly != TriState_Unknown) {
                bool groupReadOnly = gItem->isReadonly();
                if ( (readOnly == TriState_Yes && !groupReadOnly) || (readOnly==TriState_No && groupReadOnly)) {
                    continue;
                }
            }
            res.append(gItem);
        }
    }
    return res;
}

static QList<AVAnnotationItem*> selectAnnotationItems(const QList<QTreeWidgetItem*>& items, TriState readOnly) {
    QList<AVAnnotationItem*> res;
    foreach(QTreeWidgetItem* i, items) {
        AVItem* item = static_cast<AVItem*>(i);
        if (item->type == AVItemType_Annotation) {
            AVAnnotationItem* aItem = static_cast<AVAnnotationItem*>(item);
            if (readOnly != TriState_Unknown) {
                bool aReadOnly= aItem->isReadonly();
                if ( (readOnly == TriState_Yes && !aReadOnly) || (readOnly==TriState_No && aReadOnly)) {
                    continue;
                }
            }
            res.append(aItem);
        }
    }
    return res;
}

static QList<AVQualifierItem*> selectQualifierItems(const QList<QTreeWidgetItem*>& items, TriState readOnly) {
    QList<AVQualifierItem*> res;
    foreach(QTreeWidgetItem* i, items) {
        AVItem* item = static_cast<AVItem*>(i);
        if (item->type == AVItemType_Qualifier) {
            AVQualifierItem* qItem = static_cast<AVQualifierItem*>(item);
            if (readOnly != TriState_Unknown) {
                bool qReadOnly= qItem->isReadonly();
                if ( (readOnly == TriState_Yes && !qReadOnly) || (readOnly==TriState_No && qReadOnly)) {
                    continue;
                }
            }
            res.append(qItem);
        }
    }
    return res;
}

void AnnotationsTreeView::sl_removeObjectFromView() {
    QList<AVGroupItem*> topLevelGroups = selectGroupItems(tree->selectedItems(), TriState_Unknown, TriState_Yes);
    QList<GObject*> objects;
    foreach(AVGroupItem* gItem, topLevelGroups) {
        objects.append(gItem->group->getGObject());
    }
    foreach(GObject* obj, objects) {
        assert(obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE);
        ctx->removeObject(obj);
    }
}

static bool groupDepthInverseComparator(const AVGroupItem* i1, const AVGroupItem* i2) {
    int depth1 = i1->group->getGroupDepth();
    int depth2 = i2->group->getGroupDepth();
    return depth1 > depth2;
}

void AnnotationsTreeView::sl_removeAnnsAndQs() {
    //remove selected qualifiers first (cache them, since different qualifier items with equal name/val are not distinguished)
    QList<AVQualifierItem*> qualifierItemsToRemove = selectQualifierItems(tree->selectedItems(), TriState_No);
    QVector<Qualifier>  qualsToRemove(qualifierItemsToRemove.size());
    QVector<Annotation*>  qualAnnotations(qualifierItemsToRemove.size());
    for(int i=0, n = qualifierItemsToRemove.size(); i<n ; i++) {
        AVQualifierItem* qi = qualifierItemsToRemove[i];
        AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(qi->parent());
        qualAnnotations[i] = ai->annotation;
        qualsToRemove[i] = Qualifier(qi->qName, qi->qValue);
    }
    for(int i=0, n = qualifierItemsToRemove.size(); i<n ; i++) {
        Annotation* a = qualAnnotations.at(i);
        const Qualifier& q = qualsToRemove.at(i);
        a->removeQualifier(q);
    }
    

    //remove selected annotations now
    QList<AVAnnotationItem*> annotationItemsToRemove = selectAnnotationItems(tree->selectedItems(), TriState_No);
    QMultiMap<AnnotationGroup*, Annotation*> annotationsByGroup;
    foreach(AVAnnotationItem* aItem, annotationItemsToRemove) {
        assert(!aItem->annotation->getGObject()->isStateLocked());
        AnnotationGroup* ag = (static_cast<AVGroupItem*>(aItem->parent())->group);
        annotationsByGroup.insert(ag, aItem->annotation);
    }

    QList<AnnotationGroup*> agroups = annotationsByGroup.uniqueKeys();
    foreach(AnnotationGroup* ag, agroups)  {
        QList<Annotation*> annotations = annotationsByGroup.values(ag);
        ag->removeAnnotations(annotations);
    }


    //now remove selected groups
    QList<AVGroupItem*> groupItemsToRemove = selectGroupItems(tree->selectedItems(), TriState_No, TriState_No); 
    
    qSort(groupItemsToRemove.begin(), groupItemsToRemove.end(), groupDepthInverseComparator);
    //now remove all groups
    foreach(AVGroupItem* gi, groupItemsToRemove) {
        AnnotationGroup* pg = gi->group->getParentGroup();
        pg->removeSubgroup(gi->group);
    }
}

void AnnotationsTreeView::updateState() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    
    QList<AVGroupItem*> topLevelGroups = selectGroupItems(items, TriState_Unknown, TriState_Yes);
    removeObjectsFromViewAction->setEnabled(!topLevelGroups.isEmpty());
    
    QList<AVGroupItem*> nonRootModGroups = selectGroupItems(items, TriState_No, TriState_No);
    QList<AVAnnotationItem*> modAnnotations = selectAnnotationItems(items, TriState_No);
    QList<AVQualifierItem*>  modQualifiers = selectQualifierItems(items, TriState_No);
    removeAnnsAndQsAction->setEnabled(!nonRootModGroups.isEmpty() || !modAnnotations.isEmpty() || !modQualifiers.isEmpty());

    bool hasOnly1QualifierSelected = items.size() == 1 && (static_cast<AVItem*>(items.first()))->type == AVItemType_Qualifier;
    QString qName = hasOnly1QualifierSelected ? (static_cast<AVQualifierItem*>(items.first()))->qName : QString("");

    copyQualifierAction->setEnabled(hasOnly1QualifierSelected);
    copyQualifierAction->setText(hasOnly1QualifierSelected ? tr("copy_qual_'%1'_value").arg(qName) : tr("copy_qualifier_value"));
    
    bool hasOnly1QualifierSelectedWithURL = hasOnly1QualifierSelected && (static_cast<AVItem*>(items.first()))->isColumnLinked(1);
    copyQualifierURLAction->setEnabled(hasOnly1QualifierSelectedWithURL);
    copyQualifierURLAction->setText(hasOnly1QualifierSelectedWithURL ? tr("copy_qual_'%1'_url").arg(qName) : tr("copy_qualifier_url"));

    
    toggleQualifierColumnAction->setEnabled(hasOnly1QualifierSelected);
    bool hasColumn = qColumns.contains(qName);
    toggleQualifierColumnAction->setText(!hasOnly1QualifierSelected ? tr("toggle_qual_column")
        : (qColumns.contains(qName) ? tr("hide_'%1'_column"): tr("add_'%1'_column")).arg(qName));

    toggleQualifierColumnAction->setIcon(hasOnly1QualifierSelected ? (hasColumn ? removeColumnIcon : addColumnIcon) : QIcon());

    AVItem* ci = static_cast<AVItem*>(tree->currentItem());
    bool editableItemSelected = items.size() == 1 && ci!=NULL && ci == items.first() && !ci->isReadonly();
    renameAction->setEnabled(editableItemSelected);
    editAction->setEnabled(hasOnly1QualifierSelected && editableItemSelected);
    viewAction->setEnabled(hasOnly1QualifierSelected);
    
    bool hasEditableAnnotationContext = editableItemSelected && (ci->type == AVItemType_Annotation || ci->type == AVItemType_Qualifier);
    addQualifierAction->setEnabled(hasEditableAnnotationContext);
}

static bool isReadOnly(QTreeWidgetItem *item) {
    for (; item; item = item->parent()) {
        AVItem *itemi = dynamic_cast<AVItem*>(item);
        AnnotationTableObject *obj;
        switch (itemi->type) {
            case AVItemType_Group: obj = dynamic_cast<AVGroupItem*>(itemi)->group->getGObject(); break;
            case AVItemType_Annotation: obj = dynamic_cast<AVAnnotationItem*>(itemi)->annotation->getGObject(); break;
            default: continue;
        }
        if (obj->isStateLocked())
            return true;
    }
    return false;
}

bool AnnotationsTreeView::eventFilter(QObject* o, QEvent* e) {
    static bool copyOnly = false;
    static QList<int> frozen;
    static QList<QTreeWidgetItem*> selItems;
    if (o != tree->viewport()) {
        return false;
    }
    switch (e->type()) {
        case QEvent::ToolTip: {
            QHelpEvent* he = (QHelpEvent*)e;
            QPoint globalPos = he->globalPos();
            QPoint viewportPos = tree->viewport()->mapFromGlobal(globalPos);
            QTreeWidgetItem* item = tree->itemAt(viewportPos);
            if (item != NULL) {
                AVItem* avi = static_cast<AVItem*>(item);
                if (avi->type == AVItemType_Annotation) {
                    AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(avi);
                    ADVSequenceObjectContext* sc = ctx->getSequenceContext(ai->getAnnotationTableObject());
                    QString tip = ai->annotation->getQualifiersTip(15, 
                        sc?sc->getSequenceObject():NULL,
                        sc?sc->getComplementTT():NULL);
                    if (!tip.isEmpty()) {
                        QToolTip::showText(he->globalPos(), tip);
                        return true;
                    }
                }
            }
            return false;
        }
        case QEvent::MouseButtonRelease:
            lastMB = ((QMouseEvent*)e)->button();
            return false;
        case QEvent::MouseButtonPress: {
            QMouseEvent *me = dynamic_cast<QMouseEvent*>(e);
            if (me->modifiers() == Qt::NoModifier && me->button() == Qt::LeftButton) {
                QTreeWidgetItem *item = tree->itemAt(dragStartPos = me->pos());
                if (item != NULL) {
                    AVItemType type = dynamic_cast<AVItem*>(item)->type;
                    if (type == AVItemType_Annotation || type == AVItemType_Group) {
                        if (!tree->selectedItems().contains(item))
                            tree->setCurrentItem(item);
                        isDragging = true;
                        return false;
                    }
                }
            }
            isDragging = false;
            return false;
        }
        case QEvent::MouseMove: {
            QMouseEvent *me = dynamic_cast<QMouseEvent*>(e);
            if (!(me->buttons() & Qt::LeftButton) || !isDragging)
                return false;

            copyOnly = false;
            selItems = tree->selectedItems();
            for (int i = 0; i < selItems.size(); ++i) {
                assert(i >= 0 && i < selItems.size());
                AVItem *itemi = dynamic_cast<AVItem*>(selItems[i]);
                if (itemi->type == AVItemType_Annotation) {
                    for (QTreeWidgetItem *cur = selItems[i]->parent(); cur; cur = cur->parent()) {
                        if (selItems.contains(cur)) {
                            selItems.removeAt(i--);
                            break;
                        }
                    }
                }
                if (itemi->type != AVItemType_Annotation && itemi->type != AVItemType_Group) {
                    selItems[i]->setSelected(false);
                    selItems.removeAt(i--);
                } else {
                    if (!copyOnly && isReadOnly(itemi))
                        copyOnly = true;
                }
            }
            for (int i = 0; i < selItems.size(); ++i) {
                if (selItems[i]->parent() == NULL) {
                    QTreeWidgetItem *item = selItems[i];
                    selItems.removeAt(i);
                    for (int j = 0, s = item->childCount(); j < s; ++j, ++i)
                        selItems.insert(i, item->child(j));
                    --i;
                }
            }
            if (!selItems.size())
                return false;

            if ((me->pos() - dragStartPos).manhattanLength() < QApplication::startDragDistance())
                return true;
            isDragging = false;

            QByteArray itemData;
            QDataStream dataStream(&itemData, QIODevice::WriteOnly);
            int s = selItems.size();
            dataStream << s;
            for (int i = 0; i < s; ++i) {
                AVItem *itemi = dynamic_cast<AVItem*>(selItems[i]);
                dataStream << (itemi->type == AVItemType_Group);
                if (itemi->type == AVItemType_Annotation)
                    dataStream << *dynamic_cast<const AVAnnotationItem*>(itemi)->annotation->data();
                else
                    dataStream << *dynamic_cast<const AVGroupItem*>(itemi)->group;
            }
            QMimeData *mimeData = new QMimeData;
            mimeData->setData(annotationMimeType, itemData);
            QDrag *drag = new QDrag(this);
            drag->setMimeData(mimeData);
            frozen.clear();
            if (drag->exec(Qt::MoveAction | Qt::CopyAction, Qt::CopyAction) == Qt::MoveAction) {
                for (int i = 0, s = selItems.size(); i < s; ++i) {
                    if (frozen.contains(i))
                        continue;
                    QTreeWidgetItem *pitem = selItems[i]->parent();
                    assert(pitem);
                    AVItem *pitemi = dynamic_cast<AVItem*>(pitem);
                    assert(pitemi->type == AVItemType_Group);
                    AnnotationGroup *g = dynamic_cast<AVGroupItem*>(pitemi)->group;
                    if (dynamic_cast<AVItem*>(selItems[i])->type == AVItemType_Annotation) {
                        Annotation *a = dynamic_cast<AVAnnotationItem*>(selItems[i])->annotation;
                        g->removeAnnotation(a);
                    } else {
                        AnnotationGroup *sg = dynamic_cast<AVGroupItem*>(selItems[i])->group;
                        g->removeSubgroup(sg);
                    }
                }
            }
            selItems.clear();

            return true;
        }
        case QEvent::DragEnter: {
            QDragEnterEvent *de = dynamic_cast<QDragEnterEvent*>(e);
            if (de->mimeData()->hasFormat(annotationMimeType)) {
                de->acceptProposedAction();
                return true;
            }
            return false;
        }
        case QEvent::DragMove: {
            QDragMoveEvent *de = dynamic_cast<QDragMoveEvent*>(e);
            if (de->mimeData()->hasFormat(annotationMimeType)) {
                QTreeWidgetItem *item = tree->itemAt(de->pos());
                if (item) {// || !item->isDisabled()) {
                    if (isReadOnly(item)) {
                        de->ignore();
                        return true;
                    }

                    if (!(de->keyboardModifiers() & Qt::ShiftModifier)) {
                        if (copyOnly) {
                            de->ignore();
                            return true;
                        }
                        for (AVItem *itemi = dynamic_cast<AVItem*>(item); itemi != NULL; itemi = dynamic_cast<AVItem*>(itemi->parent())) {
                            if (itemi->type == AVItemType_Group) {
                                for (int i = 0, s = selItems.size(); i < s; ++i) {
                                    if (selItems[i] == itemi) {
                                        de->ignore();
                                        return true;
                                    }
                                }
                            }
                        }
                        de->setDropAction(Qt::MoveAction);
                    } else
                        de->setDropAction(Qt::CopyAction);
                    de->accept();
                } else
                    de->ignore();
                return true;
            }
            return false;
        }
        case QEvent::Drop: {
            QDropEvent *de = dynamic_cast<QDropEvent*>(e);
            const QMimeData *mime = de->mimeData();
            if (mime->hasFormat(annotationMimeType)) {
                QTreeWidgetItem *itemw = tree->itemAt(de->pos());
                if (itemw != NULL) {
                    AVItem *itemi = dynamic_cast<AVItem*>(itemw);
                    while (itemi != NULL && itemi->type != AVItemType_Group)
                        itemi = dynamic_cast<AVItem*>(itemi->parent());
                    if (itemi != NULL) {
                        QByteArray itemData = mime->data(annotationMimeType);
                        QDataStream dataStream(&itemData, QIODevice::ReadOnly);
                        AVGroupItem *itemg = dynamic_cast<AVGroupItem*>(itemi);

                        int s;
                        dataStream >> s;
                        for (int i = 0; i < s; ++i) {
                            bool isGroup;
                            dataStream >> isGroup;
                            if (isGroup && selItems.size() > i && selItems[i]->parent() == itemi)
                                frozen.append(i);
                            if (!isGroup && itemi->parent() == NULL)
                                frozen.append(i);
                            if (!isGroup) {
                                AnnotationData *adata = new AnnotationData;
                                dataStream >> *adata;
                                if (itemg->parent() != NULL) {
                                    Annotation *a = new Annotation(QSharedDataPointer<AnnotationData>(adata));
                                    itemg->group->addAnnotation(a);
                                } else
                                    delete adata;
                            } else {
                                if (isGroup && frozen.contains(i))
                                    itemg->group->removeSubgroup(dynamic_cast<AVGroupItem*>(selItems[i])->group);
                                dataStream >> itemg->group;
                            }
                        }

                        if (de->keyboardModifiers() & Qt::ShiftModifier)
                            de->setDropAction(Qt::CopyAction);
                        else
                            de->setDropAction(Qt::MoveAction);
                        de->accept();
                        return true;
                    }
                }
            } else
                return false;

            de->ignore();
            return true;
        }
        default:
            return false;
    }
}

void AnnotationsTreeView::sl_itemEntered(QTreeWidgetItem * i, int column) {
    AVItem* item = static_cast<AVItem*>(i);
    Qt::CursorShape newShape = Qt::ArrowCursor;
    Qt::CursorShape currentShape = tree->cursor().shape();
    if (item != NULL) {
        if (item->isColumnLinked(column)) {
            newShape = Qt::PointingHandCursor;
        }
    }
    if (newShape == Qt::PointingHandCursor || ((newShape == Qt::ArrowCursor && currentShape == Qt::PointingHandCursor))) {
        tree->setCursor(newShape);
    }
}

void AnnotationsTreeView::sl_itemDoubleClicked(QTreeWidgetItem *i, int) {
    AVItem* item = static_cast<AVItem*>(i);
    if (item->type == AVItemType_Qualifier) {
        editItem(item);
    }
}

void AnnotationsTreeView::sl_itemClicked(QTreeWidgetItem * i, int column) {
    AVItem* item = static_cast<AVItem*>(i);
    if (lastMB != Qt::LeftButton || item==NULL || !item->isColumnLinked(column)) {
        return;
    }
    QString fileUrl = item->getFileUrl(column);
    if (!fileUrl.isEmpty()) {
        Task* task = new LoadRemoteDocumentAndOpenViewTask(fileUrl);
        AppContext::getTaskScheduler()->registerTopLevelTask(task);
    } else {
        GUIUtils::runWebBrowser(item->buildLinkURL(column));
    }
}

void AnnotationsTreeView::sl_itemExpanded(QTreeWidgetItem* qi) {
    AVItem* i = static_cast<AVItem*>(qi);
    if (i->type != AVItemType_Annotation) {
        return;
    }
    AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(i);
    if (ai->childCount() == 0 && !ai->annotation->getQualifiers().isEmpty()) {
        assert(ai->childIndicatorPolicy() == QTreeWidgetItem::ShowIndicator);
        populateAnnotationQualifiers(ai);
    } else {
        assert(ai->childIndicatorPolicy() == QTreeWidgetItem::DontShowIndicatorWhenChildless);
    }
}

void AnnotationsTreeView::sl_onCopyQualifierValue() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    AVItem* item = static_cast<AVItem*>(items.first());
    assert(item->type == AVItemType_Qualifier);
    AVQualifierItem* qi = static_cast<AVQualifierItem*>(item);
    QApplication::clipboard()->setText(qi->qValue);
}

void AnnotationsTreeView::sl_onCopyQualifierURL() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    AVItem* item = static_cast<AVItem*>(items.first());
    if (item->isColumnLinked(1)) {
        QApplication::clipboard()->setText(item->buildLinkURL(1));
    }
}

void AnnotationsTreeView::sl_onCopyColumnText() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    AVItem* item = static_cast<AVItem*>(items.first());
    QApplication::clipboard()->setText(item->text(lastClickedColumn));
}

void AnnotationsTreeView::sl_onCopyColumnURL() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    AVItem* item = static_cast<AVItem*>(items.first());
    QApplication::clipboard()->setText(item->buildLinkURL(lastClickedColumn));
}

void AnnotationsTreeView::sl_onToggleQualifierColumn() {
    QList<QTreeWidgetItem*> items = tree->selectedItems();
    assert(items.size() == 1);
    AVItem* item = static_cast<AVItem*>(items.first());
    assert(item->type == AVItemType_Qualifier);
    AVQualifierItem* qi = static_cast<AVQualifierItem*>(item);
    if (qColumns.contains(qi->qName)) {
        removeQualifierColumn(qi->qName);
    } else {
        addQualifierColumn(qi->qName);
    }
}

void AnnotationsTreeView::sl_onRemoveColumnByHeaderClick() {
    assert(lastClickedColumn >= 2);
    assert(lastClickedColumn-2 <= qColumns.size());
    removeQualifierColumn(qColumns[lastClickedColumn-2]);
}


void AnnotationsTreeView::updateAllAnnotations(ATVAnnUpdateFlags flags) {
    QString emptyFilter;
    for(int i=0; i<tree->topLevelItemCount(); i++) {
        AVGroupItem* gi = static_cast<AVGroupItem*>(tree->topLevelItem(i));
        gi->updateAnnotations(emptyFilter, flags);
    }
}

void AnnotationsTreeView::addQualifierColumn(const QString& q) {
    TreeSorter ts(this);

    qColumns.append(q);
    int nColumns = headerLabels.size() + qColumns.size();
    tree->setColumnCount(nColumns);
    tree->setHeaderLabels(headerLabels + qColumns);
    tree->setColumnWidth(nColumns-2, nColumns - 2 == 1 ? 200 : 100);
    updateAllAnnotations(ATVAnnUpdateFlag_QualColumns);
    
    updateState();
}


void AnnotationsTreeView::removeQualifierColumn(const QString& q) {
    bool ok = qColumns.removeOne(q);
    if (!ok) {
        return;
    }

    TreeSorter ts(this);

    tree->setColumnCount(headerLabels.size() + qColumns.size());
    tree->setHeaderLabels(headerLabels + qColumns);
    updateAllAnnotations(ATVAnnUpdateFlag_QualColumns);
    
    updateState();
}

#define COLUMN_NAMES "ATV_COLUMNS"

void AnnotationsTreeView::saveState(QVariantMap& map) const {
    map.insert(COLUMN_NAMES, QVariant(qColumns));
    
    QStringList columns = map.value(COLUMN_NAMES).toStringList();
    assert(columns == qColumns);
}

void AnnotationsTreeView::updateState(const QVariantMap& map) {
    QStringList columns = map.value(COLUMN_NAMES).toStringList();
    //QByteArray geom = map.value(COLUMNS_GEOM).toByteArray();
    
    if (columns != qColumns && !columns.isEmpty()) {
        TreeSorter ts(this);
        foreach(QString q, qColumns) {
            removeQualifierColumn(q);
        }
        foreach(QString q, columns) {
            addQualifierColumn(q);
        }
    }
    /*if (columns == qColumns && !geom.isEmpty()) {
        tree->header()->restoreState(geom);
    }*/
}

void AnnotationsTreeView::setSortingEnabled(bool v) {
    if (sortTimer.isActive()) {
        sortTimer.stop();
    }
    if (v) {
        sortTimer.start();
    } else {
        tree->setSortingEnabled(false);
    }
}

void AnnotationsTreeView::sl_sortTree() {
    tree->setSortingEnabled(true);
}

void AnnotationsTreeView::sl_rename() {
    AVItem* item = static_cast<AVItem*>(tree->currentItem());
    renameItem(item);
}

void AnnotationsTreeView::sl_edit() {
    AVItem* item = static_cast<AVItem*>(tree->currentItem());
    if (item != NULL) {
        editItem(item);
    }
}

void AnnotationsTreeView::editItem(AVItem* item) {
    //warn: item could be readonly here -> used just for viewing advanced context
    if (item->type == AVItemType_Annotation) {





    }
       
    
    if (item->type == AVItemType_Qualifier) {
        AVQualifierItem* qi  = static_cast<AVQualifierItem*>(item);
        AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(qi->parent());
        Qualifier q;
        bool ro = qi->isReadonly();
        bool ok = editQualifierDialogHelper(qi, ro, q);
        if (!ro && ok && (q.getQualifierName()!=qi->qName || q.getQualifierValue()!=qi->qValue)) {
            Annotation* a = (static_cast<AVAnnotationItem*>(qi->parent()))->annotation;
            a->removeQualifier(qi->qName, qi->qValue);
            a->addQualifier(q);
            AVQualifierItem* qi = ai->findQualifierItem(q.getQualifierName(), q.getQualifierValue());
            tree->setCurrentItem(qi);
            tree->scrollToItem(qi);
        }  
    }
}

void AnnotationsTreeView::moveDialogToItem(QTreeWidgetItem* item, QDialog& d) {
    if (item == NULL) {
        return;
    }
    tree->scrollToItem(item);

    //try place dialog right below or right above the item
    d.layout()->update();
    QRect itemRect = tree->visualItemRect(item).translated(tree->viewport()->mapToGlobal(QPoint(0, 0)));
    QSize dialogSize = d.layout()->minimumSize();
    QRect dialogRect(0, 0, dialogSize.width(), dialogSize.height() + 35); //+35 -> estimation for a title bar
    QRect widgetRect = rect().translated(mapToGlobal(QPoint(0, 0)));
    QRect finalDialogRect = dialogRect.translated(itemRect.bottomLeft());
    if (!widgetRect.contains(finalDialogRect)) {
        finalDialogRect = dialogRect.translated(itemRect.topLeft()).translated(QPoint(0, -dialogRect.height()));
    }
    if (widgetRect.contains(finalDialogRect)) {
        d.move(finalDialogRect.topLeft());
    }
}

QString AnnotationsTreeView::renameDialogHelper(AVItem* i, const QString& defText, const QString& title) {
    QDialog d(this);
    d.setWindowTitle(title);
    QVBoxLayout* l = new QVBoxLayout();
    d.setLayout(l);

    QLineEdit* edit = new QLineEdit(&d);
    edit->setText(defText);
    edit->setSelection(0, defText.length());
    connect(edit, SIGNAL(returnPressed()), &d, SLOT(accept()));
    l->addWidget(edit);

    moveDialogToItem(i, d);

    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return defText;
    }
    return edit->text();
}

bool AnnotationsTreeView::editQualifierDialogHelper(AVQualifierItem* i, bool ro, Qualifier& q) {
    EditQualifierDialog d(this, Qualifier(i == NULL ? "new_qualifier" : i->qName , i == NULL ? "" : i->qValue), ro);
    moveDialogToItem(i == NULL ? tree->currentItem() : i, d);
    int rc = d.exec();
    q = d.getModifiedQualifier();
    return rc == QDialog::Accepted;
}

void AnnotationsTreeView::renameItem(AVItem* item) {
    if (item->isReadonly()) {
        return;
    }
    if (item->type == AVItemType_Group) {
        AVGroupItem* gi = static_cast<AVGroupItem*>(item);
        assert(gi->group->getParentGroup()!=NULL); //not a root group
        QString oldName = gi->group->getGroupName();
        QString newName = renameDialogHelper(item, oldName, tr("Rename group"));
        if (newName != oldName && AnnotationGroup::isValidGroupName(newName, false) 
            && gi->group->getParentGroup()->getSubgroup(newName, false) == NULL) 
        {
            gi->group->setGroupName(newName);
            gi->updateVisual();
        }
    } else if (item->type == AVItemType_Annotation) {
        AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(item);
        QList<LRegion> l = ai->annotation->getLocation();
        QString annName = ai->annotation->getAnnotationName();
        bool isCompl = ai->annotation->isOnComplementStrand();
		QList<ADVSequenceObjectContext*> soList = ctx->findRelatedSequenceContexts(ai->annotation->getGObject());
		assert(soList.size() == 1);
		ADVSequenceObjectContext* so = soList.first();
		LRegion seqRange = so->getSequenceObject()->getSequenceRange();
		EditAnnotationDialogController dialog(ai->annotation, seqRange, this);
        moveDialogToItem(ai, dialog);
        int result = dialog.exec();
        if(result == QDialog::Accepted){

            QString newName = dialog.getName();
            if (newName!=ai->annotation->getAnnotationName()) {
				ai->annotation->setAnnotationName(newName);
				QList<AVAnnotationItem*> ais = findAnnotationItems(ai->annotation);
				foreach(AVAnnotationItem* a, ais) {
					a->updateVisual(ATVAnnUpdateFlag_BaseColumns);
				}
            }
            QList<LRegion> newRegions = dialog.getRegionList();
            if( !newRegions.isEmpty() && l != newRegions){
				ai->annotation->replaceLocationRegions(newRegions);
            }
            bool newCompl = dialog.isCompliment();
            if(newCompl != isCompl){
                ai->annotation->setOnComplementStrand(newCompl);                
            }       
        }
    } else {
        assert(item->type == AVItemType_Qualifier);
        AVQualifierItem* qi = static_cast<AVQualifierItem*>(item);
        AVAnnotationItem* ai = static_cast<AVAnnotationItem*>(qi->parent());
        QString newName = renameDialogHelper(item, qi->qName, tr("Rename qualifier"));
        if (newName != qi->qName) {
            Annotation* a = (static_cast<AVAnnotationItem*>(qi->parent()))->annotation;
            QString val = qi->qValue;
            a->removeQualifier(qi->qName, val);
            a->addQualifier(newName, val);
            AVQualifierItem* qi = ai->findQualifierItem(newName, val);
            tree->setCurrentItem(qi);
            tree->scrollToItem(qi);
        }  
    }
}

void AnnotationsTreeView::sl_addQualifier() {
    AVItem* item = static_cast<AVItem*>(tree->currentItem());
    if (item->isReadonly() || item->type == AVItemType_Group) {
        return;
    }
    Qualifier q;
    bool ok = editQualifierDialogHelper(NULL, false, q);
    if (ok) {
        assert(!q.getQualifierName().isEmpty());
        AVAnnotationItem* ai = item->type == AVItemType_Annotation ? static_cast<AVAnnotationItem*>(item) : static_cast<AVAnnotationItem*>(item->parent());
        Annotation* a = ai->annotation;
        a->addQualifier(q);
        ai->setExpanded(true);
        AVQualifierItem* qi = ai->findQualifierItem(q.getQualifierName(), q.getQualifierValue());
        tree->setCurrentItem(qi);
        tree->scrollToItem(qi);
    }
}

/*void AnnotationsTreeView::sl_cutAnnotations() {
    QByteArray itemData;
    QDataStream dataStream(&itemData, QIODevice::WriteOnly);
    QList<QTreeWidgetItem*> selItems = tree->selectedItems();
    int s = selItems.size();
    dataStream << true;
    dataStream << s;
    for (int i = 0; i < s; ++i) {
        AVItem *itemi = dynamic_cast<AVItem*>(selItems[i]);
        dataStream << (itemi->type == AVItemType_Group);
        if (itemi->type == AVItemType_Annotation)
            dataStream << *dynamic_cast<const AVAnnotationItem*>(itemi)->annotation->data();
        else
            dataStream << *dynamic_cast<const AVGroupItem*>(itemi)->group;
    }
    QMimeData *mimeData = new QMimeData;
    mimeData->setData(annotationMimeType, itemData);
    QClipboard *clipboard = QApplication::clipboard();
    clipboard->setMimeData(mimeData);
}

void AnnotationsTreeView::sl_copyAnnotations() {
    QByteArray itemData;
    QDataStream dataStream(&itemData, QIODevice::WriteOnly);
    QList<QTreeWidgetItem*> selItems = tree->selectedItems();
    int s = selItems.size();
    dataStream << false;
    dataStream << s;
    for (int i = 0; i < s; ++i) {
        AVItem *itemi = dynamic_cast<AVItem*>(selItems[i]);
        dataStream << (itemi->type == AVItemType_Group);
        if (itemi->type == AVItemType_Annotation)
            dataStream << *dynamic_cast<const AVAnnotationItem*>(itemi)->annotation->data();
        else
            dataStream << *dynamic_cast<const AVGroupItem*>(itemi)->group;
    }
    QMimeData *mimeData = new QMimeData;
    mimeData->setData(annotationMimeType, itemData);
    QClipboard *clipboard = QApplication::clipboard();
    clipboard->setMimeData(mimeData);
}

void AnnotationsTreeView::sl_pasteAnnotations() {
}*/

void AnnotationsTreeView::sl_annotationObjectModifiedStateChanged() {
    AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(sender());
    assert(ao!=NULL);
    AVGroupItem* gi = findGroupItem(ao->getRootGroup());
    assert(gi!=NULL);
    gi->updateVisual();
}

AVItem* AnnotationsTreeView::currentItem(){
    return static_cast<AVItem*>(tree->currentItem());
}
//////////////////////////////////////////////////////////////////////////
/// Tree model
bool AVItem::processLinks(const QString& qName, const QString& qValue, int col) {
    bool linked = false;
    if (qName == "db_xref") {
        QStringList l = qValue.split(":");
        QString dbName = l[0];
        QString dbId = l.size() > 1 ? l[1] : "";
        DBXRefInfo info = AppContext::getDBXRefRegistry()->getRefByKey(dbName);
        linked = !info.url.isEmpty();
        setToolTip(col, info.comment);
        if (linked) {
            setData(col, Qt::UserRole, true);
        }
    } 

    if (linked) {
        QFont f = font(col);
        f.setUnderline(true);
        setFont(col, f);
        setForeground(col, Qt::blue);
    }
    return linked;
}

bool AVItem::isColumnLinked(int col) const {
    return data(col, Qt::UserRole).type() == QVariant::Bool;
}

QString AVItem::buildLinkURL(int col) const {
    assert(isColumnLinked(col));
    QString qValue = text(col);
    QStringList split = qValue.split(":");
    QString type = split.first();
    QString id = split.size() < 2 ? QString("") : split[1];
    QString url = AppContext::getDBXRefRegistry()->getRefByKey(type).url.arg(id);
    return url;
}

QString AVItem::getFileUrl(int col) const 
{
    assert(isColumnLinked(col));
    QStringList split = text(col).split(":");
    QString type = split.first();
    QString fileUrl = AppContext::getDBXRefRegistry()->getRefByKey(type).fileUrl;
    if (!fileUrl.isEmpty())
    {
        QString id = split.size() < 2 ? QString("") : split[1];
        return fileUrl.arg(id);
    }
    return fileUrl;

}

AVGroupItem::AVGroupItem(AnnotationsTreeView* _atv, AVGroupItem* parent, AnnotationGroup* g) : AVItem(parent, AVItemType_Group), group(g), atv(_atv) 
{
    updateVisual();
}

AVGroupItem::~AVGroupItem() {
    group = NULL;
}

const QIcon& AVGroupItem::getGroupIcon() {
    static QIcon groupIcon(":/core/images/group_green_active.png");
    return groupIcon;
}

const QIcon& AVGroupItem::getDocumentIcon() {
    static QIcon groupIcon(":/core/images/gobject.png");
    return groupIcon;
}

void AVGroupItem::updateVisual() {
    if (group->getParentGroup() == group->getGObject()->getRootGroup()) {
        const AnnotationSettings* as  = AppContext::getAnnotationsSettingsRegistry()->getAnnotationSettings(group->getGroupName());
        Qt::ItemFlags f = flags();
        if (as->visible) {
            f|=Qt::ItemIsEnabled | Qt::ItemIsSelectable;
        } else {
            f&= (!Qt::ItemIsEnabled) | Qt::ItemIsSelectable;
        }
        setFlags(f);
    }
    if (parent() == NULL) {
        AnnotationTableObject* aobj  = group->getGObject();
        QString docShortName = aobj->getDocument()->getName();
        assert(!docShortName.isEmpty());
        QString text = group->getGObject()->getGObjectName() + " ["+docShortName+"]";
        if (aobj->isTreeItemModified()) { 
            text+=" *";
        }
        setText(0, text);
        setIcon(0, getDocumentIcon());
    } else {
        int na = group->getAnnotations().size();
        int ng = group->getSubgroups().size();
        QString nameString = group->getGroupName() + "  " + QString("(%1, %2)").arg(ng).arg(na);
        setText(0, nameString);
        setIcon(0, getGroupIcon());
    }
}

void AVGroupItem::updateAnnotations(const QString& nameFilter, ATVAnnUpdateFlags f) {
    bool noFilter = nameFilter.isEmpty();
    for (int j = 0; j < childCount(); j++) {
        AVItem* item = static_cast<AVItem*>(child(j));
        if (item->type == AVItemType_Group) {
            AVGroupItem* level1 = static_cast<AVGroupItem*>(item);
            if (noFilter || level1->group->getGroupName() == nameFilter) {
                level1->updateAnnotations(nameFilter, f);
            }
        } else {
            assert(item->type == AVItemType_Annotation);
            AVAnnotationItem* aItem= static_cast<AVAnnotationItem*>(item);
            if (noFilter || aItem->annotation->getAnnotationName() == nameFilter) {
                aItem->updateVisual(f);
            }
        }
    }
}

bool AVGroupItem::isReadonly() const {
    //documents names are not editable
    return group->getParentGroup() == NULL ? true: group->getGObject()->isStateLocked();
}

AnnotationTableObject* AVGroupItem::getAnnotationTableObject() const {
    return group->getGObject();
}

AnnotationGroup* AVGroupItem::getAnnotationGroup() const {
    return group;
}

AVAnnotationItem::AVAnnotationItem(AVGroupItem* parent, Annotation* a) : AVItem(parent, AVItemType_Annotation), annotation(a)
{
    updateVisual(ATVAnnUpdateFlags(ATVAnnUpdateFlag_BaseColumns) | ATVAnnUpdateFlag_QualColumns);    
    hasNumericQColumns = false;
}

AVAnnotationItem::~AVAnnotationItem() {
    annotation = NULL;
}

#define MAX_ICONS_CACHE_SIZE 500

QMap<QString, QIcon>& AVAnnotationItem::getIconsCache() {
    static QMap<QString, QIcon> iconsCache;
    return iconsCache;
}

void AVAnnotationItem::updateVisual(ATVAnnUpdateFlags f) {
    if (f.testFlag(ATVAnnUpdateFlag_BaseColumns)) {
        const QString& name = annotation->getAnnotationName();

        const AnnotationSettings* as = AppContext::getAnnotationsSettingsRegistry()->getAnnotationSettings(name);
        Qt::ItemFlags f2 = flags();
        if (as->visible) {
            f2|= Qt::ItemIsEnabled | Qt::ItemIsSelectable;
        } else {
            f2&= (!Qt::ItemIsEnabled) | Qt::ItemIsSelectable;
        }
        setFlags(f2);

        QMap<QString, QIcon>& cache = getIconsCache();
        QIcon icon = cache.value(name);
        if (icon.isNull()) {
            icon = GUIUtils::createSquareIcon(as->color, 9);
            if (cache.size() > MAX_ICONS_CACHE_SIZE) {
                cache.clear();
            }
            cache[name] = icon;
        }
        assert(!icon.isNull());

        setIcon(0, icon);
        setText(0, annotation->getAnnotationName());
        locationString = Genbank::LocationParser::buildLocationString(annotation->data());
    }

    if (f.testFlag(ATVAnnUpdateFlag_QualColumns)) {
        //setup custom qualifiers columns
        AnnotationsTreeView* atv = getAnnotationTreeView();
        assert(atv!=NULL);
        const QStringList& colNames = atv->getQualifierColumnNames();
        hasNumericQColumns = false;
        for (int i=0, n = colNames.size(); i < n ;i++) {
            int col = 2+i;
            QString colName = colNames[i];
            QString colText = annotation->findFirstQualifierValue(colName);
            setText(2+i, colText);
            bool linked = processLinks(colName, colText,  col);
            if (!linked) {
                bool ok  = false;
                double d = colText.toDouble(&ok);
                if (ok) {
                    setData(col, Qt::UserRole, d);
                    hasNumericQColumns = true;
                }
            }
        }
    }
}

QVariant AVAnnotationItem::data( int col, int role ) const {
    if (col == 1 && role == Qt::DisplayRole) {
        if (locationString.isEmpty()) {
            locationString = Genbank::LocationParser::buildLocationString(annotation->data());
        }
        return locationString;
    }

    return QTreeWidgetItem::data(col, role);
}

bool AVAnnotationItem::operator<(const QTreeWidgetItem & other) const {
    int col = treeWidget()->sortColumn();
    const AVItem& avItem = (const AVItem&)other;
    if (avItem.type != AVItemType_Annotation) {
        return text(col) < other.text(col);
    }
    const AVAnnotationItem& ai = (const AVAnnotationItem&)other;
    if (col == 0) {
        return annotation->getAnnotationName() < ai.annotation->getAnnotationName();
    }
    if (col == 1 || (isColumnNumeric(col) && ai.isColumnNumeric(col))) {
        double oval = ai.getNumericVal(col);
        double mval = getNumericVal(col);
        return mval < oval;
    }
    return text(col) < other.text(col);
}

bool AVAnnotationItem::isColumnNumeric(int col) const {
    if (col == 0) {
        return false;
    } 
    if (col == 1) {
        return true;
    }
    if (!hasNumericQColumns) {
        return false;
    }
    return data(col, Qt::UserRole).type() == QVariant::Double;
}

double AVAnnotationItem::getNumericVal(int col) const {
    if (col == 1) {
        LRegion r = annotation->getLocation().first();
        return r.startPos;
    }
    bool ok  = false;
    double d = data(col, Qt::UserRole).toDouble(&ok);
    assert(ok);
    return d;
}

void AVAnnotationItem::removeQualifier(const Qualifier& q) {
    for(int i=0, n = childCount(); i < n; i++) {
        AVQualifierItem* qi = static_cast<AVQualifierItem*>(child(i));
        if (qi->qName == q.getQualifierName() && qi->qValue == q.getQualifierValue()) {
            delete qi;
            break;
        }
    }
    updateVisual(ATVAnnUpdateFlag_QualColumns);
}

void AVAnnotationItem::addQualifier(const Qualifier& q) {
    AVQualifierItem* qi = new AVQualifierItem(this, q); Q_UNUSED(qi);
    updateVisual(ATVAnnUpdateFlag_QualColumns);
}


AVQualifierItem* AVAnnotationItem::findQualifierItem(const QString& name, const QString& val) const {
    for(int i=0, n = childCount(); i < n; i++) {
        AVQualifierItem* qi = static_cast<AVQualifierItem*>(child(i));
        if (qi->qName == name && qi->qValue == val) {
            return qi;
        }
    }
    return NULL;
}

AVQualifierItem::AVQualifierItem(AVAnnotationItem* parent, const Qualifier& q) 
: AVItem(parent, AVItemType_Qualifier), qName(q.getQualifierName()), qValue(q.getQualifierValue())
{
    setText(0, qName);
    setText(1, qValue);

    processLinks(qName, qValue, 1);
}

}//namespace
