/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/ProjectModel.h>
#include <core_api/NetworkConfiguration.h>
#include <core_api/Counter.h>
#include <core_api/DBXRefRegistry.h>
#include <document_format/DocumentFormatUtils.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_tasks/CopyDataTask.h>
#include <util_tasks/LoadDocumentTask.h>


#include <QtCore/QFileInfo>
#include <QtCore/QFile>
#include <QtCore/QUrl>

#include <QtNetwork/QNetworkProxy>



#include "LoadRemoteDocumentTask.h"

#define DB_ATTR "database"
#define DOC_ID_ATTR "document_id"
#define EXPECTED_DOC_ATTR "expected_document"

namespace GB2 {

LogCategory log(ULOG_CAT_LOAD_REMOTE_RESOURCE);

const QString NCBI_ESEARCH_URL("http://eutils.ncbi.nlm.nih.gov/entrez/eutils/esearch.fcgi?db=%1&term=%2&tool=UGENE");
const QString NCBI_EFETCH_URL("http://eutils.ncbi.nlm.nih.gov/entrez/eutils/efetch.fcgi?db=%1&id=%2&retmode=text&rettype=%3&tool=UGENE");
const QString GENBANK_DNA("NCBI GenBank (DNA sequence)");
const QString GENBANK_PROTEIN("NCBI protein sequence database");

// Entrez tools variables
#define GENBANK_NUCLEOTIDE_ID "nucleotide"
#define GENBANK_PROTEIN_ID "protein"
#define GENBANK_FORMAT "gb"
#define FASTA_FORMAT "fasta"


LoadRemoteDocumentTask::LoadRemoteDocumentTask( const GUrl& fileUrl) : 
    Task("LoadRemoteDocument", TaskFlags_NR_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), copyDataTask(NULL), loadDocumentTask(NULL), doc(NULL)
{
    sourceUrl = fileUrl;
    fileName = sourceUrl.fileName();
    GCOUNTER( cvar, tvar, "LoadRemoteDocumentTask" );
}

LoadRemoteDocumentTask::LoadRemoteDocumentTask( const QString& accId, const QString& dbNm) :
    Task("LoadRemoteDocument", TaskFlags_NR_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), copyDataTask(NULL), loadDocumentTask(NULL), doc(NULL), accNumber(accId), dbName(dbNm)

{
    sourceUrl = GUrl(RemoteDBRegistry::getRemoteDBRegistry().getURL(accId, dbName));
    
    if (sourceUrl.isLocalFile()) {
        QString dbId = RemoteDBRegistry::getRemoteDBRegistry().getDbEntrezName(dbName);
        if (dbId == GENBANK_NUCLEOTIDE_ID || dbId == GENBANK_PROTEIN_ID) {
            format = GENBANK_FORMAT;
        } else {
            format = FASTA_FORMAT;
        }
        fileName = accNumber + "." + format;
    } else {
        fileName = sourceUrl.fileName();

    }
    
    GCOUNTER( cvar, tvar, "LoadRemoteDocumentTask" );
}

void LoadRemoteDocumentTask::prepare()
{
    if (fileName.isEmpty()) {
        stateInfo.setError("Incorrect key identifier!");
        return;
    }
    
    // Check if the file has already been downloaded
    RecentlyDownloadedCache* cache = AppContext::getRecentlyDownloadedCache();
    if (cache->contains(fileName)) {
        fullPath = cache->getFullPath(fileName);
        if ( initLoadDocumentTask() ) {
            addSubTask(loadDocumentTask);
        } 
        return;   
    }
    
    QString path = AppContext::getAppSettings()->getUserAppsSettings()->getDownloadDirPath();
    fullPath = path + "/" + fileName;
    QDir dir;
    if (!dir.exists(path)) {
        // Creating directory if it doesn't exist
        bool res = dir.mkdir(path);
        if (!res) {
            stateInfo.setError(QString("Cannot create directory %1").arg(path));
            return;
        }
    }
    
    if (sourceUrl.isHyperLink()) {
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::HTTP_FILE);
        IOAdapterFactory * iow = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
        copyDataTask = new CopyDataTask(iof, sourceUrl, iow, fullPath);
        addSubTask(copyDataTask);
    } else {
		assert(sourceUrl.isLocalFile());
        QString dbId = RemoteDBRegistry::getRemoteDBRegistry().getDbEntrezName(dbName);
        loadDataFromEntrezTask = new LoadDataFromEntrezTask(dbId, accNumber,format,fullPath);
        addSubTask(loadDataFromEntrezTask);
    }
}

Task::ReportResult LoadRemoteDocumentTask::report()
{
    return ReportResult_Finished;
}

QList<Task*> LoadRemoteDocumentTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> subTasks;
    if (subTask->hasErrors()) {
        return subTasks;
    }
    if (subTask == copyDataTask || subTask == loadDataFromEntrezTask) {
        if (initLoadDocumentTask()) { 
           subTasks.append(loadDocumentTask); 
           AppContext::getRecentlyDownloadedCache()->append(fullPath);
       }
    } else  if ( subTask == loadDocumentTask) {
        doc = loadDocumentTask->getDocument();
    }
    return subTasks;
}

bool LoadRemoteDocumentTask::initLoadDocumentTask(  )
{
    Q_ASSERT(!fullPath.isEmpty());

    // Check if the document has been loaded 
    Project* proj = AppContext::getProject();
    if (proj != NULL) {
        Document* foundDoc = proj->findDocumentByURL(fullPath);
        if (foundDoc != NULL) {
            doc = foundDoc;
            return false;
        }
    }

    // Detect format
    if (formatId.isEmpty()) {
        QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(fullPath);
        if (formats.isEmpty()) {
            stateInfo.setError("Unknown file format!");
            return false;
        } else {
            formatId = formats.first()->getFormatId();
        }
    }
    IOAdapterFactory * iow = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    loadDocumentTask =  new LoadDocumentTask(formatId, fullPath, iow);

    return true;
}

Document* LoadRemoteDocumentTask::getDocument()
{
    return doc;
}
////////////////////////////////////////////////////////////////////////////////////////////////////


RecentlyDownloadedCache::RecentlyDownloadedCache()
{
    QStringList fileNames = AppContext::getAppSettings()->getUserAppsSettings()->getRecentlyDownloadedFileNames();
    foreach (const QString& path, fileNames) {
        QFileInfo info(path);
        if (info.exists()) {
            append(path);
        }
    }
}

void RecentlyDownloadedCache::append( const QString& fileName )
{
    QFileInfo info(fileName);
    urlMap.insert(info.fileName(), fileName);
}

QString RecentlyDownloadedCache::getFullPath( const QString& fileName )
{
    Q_ASSERT(urlMap.contains(fileName));
    return urlMap.value(fileName);
}

RecentlyDownloadedCache::~RecentlyDownloadedCache()
{
    //TODO: cache depends on AppSettings! get rid of this dependency!
    QStringList fileNames = urlMap.values();
    AppSettings* settings = AppContext::getAppSettings();
    Q_ASSERT(settings != NULL);
    UserAppsSettings* us = settings->getUserAppsSettings();
    Q_ASSERT(us != NULL);
    us->setRecentlyDownloadedFileNames(fileNames);
}

//////////////////////////////////////////////////////////////////////////


LoadDataFromEntrezTask::LoadDataFromEntrezTask( const QString& dbId, const QString& accNum, const QString& retType, const QString& path )
: Task("LoadDataFromEntrez", TaskFlags_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), db(dbId), accNumber(accNum), fullPath(path), format(retType)

{

}

void LoadDataFromEntrezTask::run()
{
    stateInfo.progress = 0;
    log.trace("Load data from Entrez started...");
    loop = new QEventLoop;

    networkManager = new QNetworkAccessManager(this);
    connect(networkManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(sl_replyFinished(QNetworkReply*)));
    
    // Step one: search for global Entrez index
    log.trace("Request for global entrez index info...");
    QString traceSearchUrl= QString(NCBI_ESEARCH_URL).arg(db).arg(accNumber);
    log.trace(traceSearchUrl);

    QUrl request( NCBI_ESEARCH_URL.arg(db).arg(accNumber) );
    NetworkConfiguration* nc = AppContext::getAppSettings()->getNetworkConfiguration();
    QNetworkProxy proxy = nc->getProxyByUrl(request);
    networkManager->setProxy(proxy);
    searchReply = networkManager->get(QNetworkRequest( request ));
    connect(searchReply, SIGNAL(error(QNetworkReply::NetworkError)),
        this, SLOT(sl_onError(QNetworkReply::NetworkError)));
    loop->exec();
    
    if (resultIndex.isEmpty()) {
        stateInfo.setError("Result not found");
    }
    if (stateInfo.hasErrors()  ) {
        return;
    }
    
    log.trace(QString("Global index is %1").arg(resultIndex));
    log.trace("Downloading file...");
    // Step two: download the file    
    QString traceFetchUrl = QString(NCBI_EFETCH_URL).arg(db).arg(resultIndex).arg(format);
    log.trace(traceFetchUrl);
    QUrl requestUrl(NCBI_EFETCH_URL.arg(db).arg(resultIndex).arg(format));
    downloadReply = networkManager->get(QNetworkRequest(requestUrl));
    connect(downloadReply, SIGNAL(error(QNetworkReply::NetworkError)),
        this, SLOT(sl_onError(QNetworkReply::NetworkError)));
    connect( downloadReply, SIGNAL(uploadProgress( qint64, qint64 )),
        this, SLOT(sl_uploadProgress(qint64,qint64)) );

    loop->exec();
    log.trace("Download finished.");
    
    QFile downloadedFile(fullPath);
    if (!downloadedFile.open(QIODevice::WriteOnly)) {
        stateInfo.setError("Cannot open file to write!");
        return;
    }

    downloadedFile.write(downloadReply->readAll());
    
}

void LoadDataFromEntrezTask::sl_replyFinished( QNetworkReply* reply )
{
    if (reply == searchReply) {
        QXmlInputSource source(reply);
        ESearchResultHandler* handler = new ESearchResultHandler;
        xmlReader.setContentHandler(handler);
        xmlReader.setErrorHandler(handler);
        bool ok = xmlReader.parse(source);
        if (!ok) {
            assert(0);
            stateInfo.setError("Parsing eSearch result failed");
        } else {
            resultIndex =  handler->getResultIndex();
        }
        delete handler;

    } else if (reply == downloadReply) {
        
    }
    loop->exit();
    
}

void LoadDataFromEntrezTask::sl_onError( QNetworkReply::NetworkError error )
{
    stateInfo.setError(QString("NetworkReply error %1").arg(error));
    loop->exit();
}

void LoadDataFromEntrezTask::sl_uploadProgress( qint64 bytesSent, qint64 bytesTotal )
{
    stateInfo.progress = bytesSent/ bytesTotal * 100;
}

//////////////////////////////////////////////////////////////////////////

bool ESearchResultHandler::startElement( const QString &namespaceURI, const QString &localName, const QString &qName, const QXmlAttributes &attributes )
{
    Q_UNUSED(namespaceURI); Q_UNUSED(localName); Q_UNUSED(attributes);

    if (!metESearchResult && qName != "eSearchResult") {
        errorStr = QObject::tr("This is not ESearch result!");
        return false;
    }
    if ("eSearchResult" == qName) {
        metESearchResult = true;
    } 
    curText.clear();
    return true;
}

bool ESearchResultHandler::endElement( const QString &namespaceURI, const QString &localName, const QString &qName )
{
    Q_UNUSED(namespaceURI); Q_UNUSED(localName);
    if ("Id" == qName) {
        index = curText;
    }
    return true;
}

ESearchResultHandler::ESearchResultHandler()
{

    metESearchResult = false;
}

bool ESearchResultHandler::characters( const QString &str )
{
    curText += str;
    return true;
}

bool ESearchResultHandler::fatalError( const QXmlParseException &exception )
{
    Q_UNUSED(exception);
    assert(0);
//     QMessageBox::information( NULL, QObject::tr("SAX Bookmarks"),
//         QObject::tr("Parse error at line %1, column %2:\n"
//         "%3")
//         .arg(exception.lineNumber())
//         .arg(exception.columnNumber())
//         .arg(exception.message()));
    return false;

}

//////////////////////////////////////////////////////////////////////////

RemoteDBRegistry::RemoteDBRegistry()
{
    queryDBs.insert(GENBANK_DNA,  GENBANK_NUCLEOTIDE_ID);
    queryDBs.insert(GENBANK_PROTEIN, GENBANK_PROTEIN_ID);
    
    const QMap<QString,DBXRefInfo>& entries = AppContext::getDBXRefRegistry()->getEntries();
    foreach(const DBXRefInfo& info, entries.values()) {
        if (!info.fileUrl.isEmpty()) {
            httpDBs.insert(info.name, info.fileUrl);
        }
    }

    hints.insert(GENBANK_DNA, QObject::tr("Use Genbank DNA accession number. For example: NC_001363 or D11266"));
    hints.insert(GENBANK_PROTEIN, QObject::tr("Use Genbank protein accession number. For example: AAA59172.1"));
    hints.insert("PDB", QObject::tr("Use PDB molecule four-letter identifier. For example: 3INS or 1CRN"));
}

RemoteDBRegistry& RemoteDBRegistry::getRemoteDBRegistry()
{
    static RemoteDBRegistry registry;
    return registry;
}


QList<QString> RemoteDBRegistry::getDBs()
{
    return  ( queryDBs.keys() + httpDBs.keys() );
}

QString RemoteDBRegistry::getURL( const QString& accId, const QString& dbName )
{
    QString result("");
    if (httpDBs.contains(dbName)) {
        result = QString(httpDBs.value(dbName)).arg(accId);
    }    
    return result; 
}

QString RemoteDBRegistry::getDbEntrezName( const QString& dbName )
{
    return queryDBs.value(dbName);
}

QString RemoteDBRegistry::getHint( const QString& dbName )
{
    if (hints.contains(dbName)) {
        return hints.value(dbName);
    } else {
        return QObject::tr("Use %1 unique identifier.").arg(dbName);
    }

}
//////////////////////////////////////////////////////////////////////////

void GTest_LoadRemoteDocumentTask::init(XMLTestFormat *tf, const QDomElement& el){
    Q_UNUSED(tf);

	dbName.clear();
	docId.clear();
	expectedDoc.clear();
	t = NULL;

	QString tmp = el.attribute(DB_ATTR);
	
	if(tmp.isEmpty()){
		failMissingValue(DB_ATTR);
		return;
	}
	dbName = tmp;
	
	tmp = el.attribute(DOC_ID_ATTR);
	if(tmp.isEmpty()){
		failMissingValue(DB_ATTR);
		return;
	}
	docId = tmp;

	tmp = el.attribute(EXPECTED_DOC_ATTR);
	if(tmp.isEmpty()){
		failMissingValue(EXPECTED_DOC_ATTR);
		return;
	}
	expectedDoc = env->getVar("COMMON_DATA_DIR") + "/" + tmp;
}

void GTest_LoadRemoteDocumentTask::prepare(){
	RemoteDBRegistry& registry = RemoteDBRegistry::getRemoteDBRegistry();
	const QList<QString> dataBases = registry.getDBs();
	bool checked = false;
	foreach(const QString& db, dataBases) {
		if(dbName == db){
			checked = true;
		}
	}
	if(!checked){
		setError(tr("Given database name %1 not present in database registry").arg(dbName));
		return;
	}
	t = new LoadRemoteDocumentTask(docId, dbName);
	addSubTask(t);
}

Task::ReportResult GTest_LoadRemoteDocumentTask::report(){
	if(t != NULL){
		if(!t->hasErrors()){
			QFile expectedFile(expectedDoc), actualFile(t->getLocalUrl());
			expectedFile.open(QIODevice::ReadOnly), actualFile.open(QIODevice::ReadOnly);
			QByteArray expectedContent(expectedFile.readAll()), actualContent(actualFile.readAll());
			if(expectedContent != actualContent){
				stateInfo.setError(GTest::tr("File %1 content not equal with expected").arg(t->getLocalUrl()));
			}
			expectedFile.close(), actualFile.close();
		}
		return ReportResult_Finished;		
	}
	return ReportResult_Finished;
}

void GTest_LoadRemoteDocumentTask::cleanup(){

}

QList<XMLTestFactory*> LoadRemoteDocumentTests::createTestFactories(){
	QList<XMLTestFactory*> res;
	res.append(GTest_LoadRemoteDocumentTask::createFactory());
	return res;
}

//////////////////////////////////////////////////////////////////////////

LoadRemoteDocumentAndOpenViewTask::LoadRemoteDocumentAndOpenViewTask( const QString& accId, const QString& dbName )
: Task("LoadRemoteDocumentAndOpenView", TaskFlags_NR_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), loadRemoteDocTask(NULL)
{
    accNumber = accId;
    databaseName = dbName;
}

LoadRemoteDocumentAndOpenViewTask::LoadRemoteDocumentAndOpenViewTask( const GUrl& url )
: Task("LoadRemoteDocumentAndOpenView", TaskFlags_NR_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), loadRemoteDocTask(NULL)
{
    docUrl = url;
}

void LoadRemoteDocumentAndOpenViewTask::prepare()
{
    if (docUrl.isEmpty()) {
        loadRemoteDocTask = new LoadRemoteDocumentTask(accNumber, databaseName);
    } else {
        loadRemoteDocTask = new LoadRemoteDocumentTask(docUrl); 
    }

    addSubTask(loadRemoteDocTask);


}

QList<Task*> LoadRemoteDocumentAndOpenViewTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> subTasks;
    if (subTask->hasErrors()) {
        return subTasks;
    }

    if (subTask == loadRemoteDocTask ) {
        QString fullPath = loadRemoteDocTask->getLocalUrl();
        Project* proj = AppContext::getProject();
        if (proj == NULL) {
            subTasks.append(  AppContext::getProjectLoader()->openProjectTask(fullPath, false));
        } else {
            Document* doc = loadRemoteDocTask->getDocument();
            assert(doc != NULL);
            if ( proj->getDocuments().contains(doc) ) {
                subTasks.append(new LoadUnloadedDocumentAndOpenViewTask(doc));
            } else {
                // Add document to project
                DocumentFormat* format = doc->getDocumentFormat(); 
                IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
                Document* clonedDoc = new Document(format, iof, fullPath);
                clonedDoc->loadFrom(doc); // doc was loaded in a separate thread -> clone all GObjects
                assert(!clonedDoc->isTreeItemModified());
                assert(clonedDoc->isLoaded());
                subTasks.append(new AddDocumentTask(clonedDoc));
                subTasks.append(new LoadUnloadedDocumentAndOpenViewTask(clonedDoc));

            }    
        }
    }

    return subTasks;
}
} //namespace
