/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <memory>
#include <cstdio>

#include <core_api/Log.h>
#include <core_api/Counter.h>

#include "PingTask.h"

#define PING_TASK_LOG_CAT "Ping task"

namespace GB2 {

static LogCategory log( PING_TASK_LOG_CAT );

static QString getLongString( const QString & str, int num ) {
    assert( !str.isEmpty() && 0 < num );
    QString res;
    for( int i = 0; i < num; ++i ) {
        res.append( str );
    }
    return res;
}

static bool isLongString( const QVariant & data, const QString & str, int num ) {
    assert( !str.isEmpty() && 0 < num );
    if( !data.canConvert( QVariant::String ) ) {
        return false;
    }
    QString dataStr = data.toString();
    if( num != dataStr.count( str ) ) {
        return false;
    }
    return true;
}

/************************************
 * PingTaskSettings
 ***********************************/

const QString PingTaskSettings::PING_SETTINGS_ITEM_STR = "12345678";

PingTaskSettings::PingTaskSettings() {
}

PingTaskSettings::~PingTaskSettings() {
}

QVariant PingTaskSettings::serialize() const {
    return getLongString( PING_SETTINGS_ITEM_STR, PING_SETTINGS_ITEMS_NUM );
}

bool PingTaskSettings::deserialize( const QVariant & data ) {
    return isLongString( data, PING_SETTINGS_ITEM_STR, PING_SETTINGS_ITEMS_NUM );
}

/************************************
* PingTaskResult
***********************************/

const QString PingTaskResult::PING_RESULT_ITEM_STR = "abcdefgh";

PingTaskResult::PingTaskResult() {
}

PingTaskResult::~PingTaskResult() {
}

QVariant PingTaskResult::serialize() const {
    return getLongString( PING_RESULT_ITEM_STR, PING_RESULT_ITEMS_NUM );
}

bool PingTaskResult::deserialize( const QVariant & data ) {
    return isLongString( data, PING_RESULT_ITEM_STR, PING_RESULT_ITEMS_NUM );
}

/************************************
* PingTask
***********************************/

PingTask::PingTask(const PingTaskSettings &/*settings*/)
: LocalTask( tr( "Ping task" ), TaskFlag_None ) {
    stateInfo.progress = 0;
    GCOUNTER( cvar, tvar, "PING_TASK" );
    log.message( LogLevel_INFO, tr( "Ping task with %1 id created" ).arg( QString::number( getTaskId() ) ) );
}

void PingTask::prepare() {
    log.message( LogLevel_INFO, tr( "Ping task with %1 id preparing" ).arg( QString::number( getTaskId() ) ) );
}

void PingTask::run() {
    log.message( LogLevel_INFO, tr( "Ping task with %1 id running" ).arg( QString::number( getTaskId() ) ) );
}

Task::ReportResult PingTask::report() {
    log.message( LogLevel_INFO, tr( "Ping task with %1 id reporting" ).arg( QString::number( getTaskId() ) ) );
    return ReportResult_Finished;
}

PingTask::~PingTask() {
    log.message( LogLevel_INFO, tr( "Ping task with %1 id destroyed" ).arg( QString::number( getTaskId() ) ) );
}

const LocalTaskResult * PingTask::getResult() const {
    return &result;
}

/************************************
* PingTaskFactory
***********************************/

const QString PingTaskFactory::ID = "Ping task";

PingTaskFactory::PingTaskFactory() {
}

PingTaskFactory::~PingTaskFactory() {
}

LocalTask * PingTaskFactory::createInstance( const QVariant & serializedSettings ) const {
    PingTaskSettings settings;
    if(!settings.deserialize(serializedSettings)) {
        return NULL;
    }
    return new PingTask(settings);
}

LocalTaskResult * PingTaskFactory::createResults( const QVariant & serializedResults ) const {
    std::auto_ptr<PingTaskResult> result(new PingTaskResult());
    if(!result->deserialize(serializedResults)) {
        return NULL;
    }
    return result.release();
}

QString PingTaskFactory::getId() const {
    return ID;
}

} // GB2
