/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SequenceWalkerTask.h"

#include <core_api/DNATranslation.h>
#include <util_text/TextUtils.h>

namespace GB2 {

SequenceWalkerConfig::SequenceWalkerConfig() 
: seq(NULL), seqSize(0), complTrans(NULL), aminoTrans(NULL), 
        chunkSize(0), lastChunkExtraLen(0), overlapSize(0), nThreads(MAX_PARALLEL_SUBTASKS_SERIAL) 
{
    strandToWalk = (complTrans!=NULL) ? StrandOption_Both : StrandOption_DirectOnly;
}


SequenceWalkerTask::SequenceWalkerTask(const SequenceWalkerConfig& c, SequenceWalkerCallback* cb, const QString& name, TaskFlags tf) 
: Task(name, tf), config(c), callback(cb)
{
    assert(config.chunkSize >= config.overlapSize);
    assert(cb != NULL);
    assert(config.strandToWalk == StrandOption_DirectOnly || config.complTrans!=NULL);

    maxParallelSubtasks = config.nThreads;
    QList<SequenceWalkerSubtask*> subs = prepareSubtasks();
    foreach(SequenceWalkerSubtask* sub, subs) {
        addSubTask(sub);
    }
}

QList<SequenceWalkerSubtask*> SequenceWalkerTask::prepareSubtasks() {
    QList<SequenceWalkerSubtask*> res;
    
    if (config.range.isEmpty()) {
        config.range.startPos = 0;
        config.range.len = config.seqSize;
    } else {
        LRegion wholeSeqReg(0, config.seqSize);
        assert(wholeSeqReg.contains(config.range));
        config.range = wholeSeqReg.intersect(config.range);
    }
    if (config.aminoTrans == NULL ) {
        //try walk direct and complement strands
        QList<LRegion> chunks = splitRange(config.range, config.chunkSize, config.overlapSize, config.lastChunkExtraLen, false);

        if (config.strandToWalk == StrandOption_Both || config.strandToWalk == StrandOption_DirectOnly) {
            QList<SequenceWalkerSubtask*> directTasks  = createSubs(chunks, false, false);
            res+=directTasks;
        }
        if (config.strandToWalk == StrandOption_Both || config.strandToWalk == StrandOption_ComplementOnly) {
            assert(config.complTrans!=NULL);
            QList<SequenceWalkerSubtask*> complTasks = createSubs(chunks, true, false);
            res+=complTasks;
        }
    } else {
        // try walk 3 direct and 3 complement translations
        if (config.strandToWalk == StrandOption_Both || config.strandToWalk == StrandOption_DirectOnly) {
            for (int i=0; i<3; i++) {
                LRegion strandRange(config.range.startPos + i, config.range.len - i);
                QList<LRegion> chunks = splitRange(strandRange, config.chunkSize, config.overlapSize, config.lastChunkExtraLen, false);
                QList<SequenceWalkerSubtask*> directTasks = createSubs(chunks, false, true);
                res+=directTasks;
            }
        }
        if (config.strandToWalk == StrandOption_Both || config.strandToWalk == StrandOption_ComplementOnly) {
            assert(config.complTrans!=NULL);
            for (int i=0; i<3; i++) {
                LRegion strandRange(config.range.startPos, config.range.len - i);
                QList<LRegion> chunks = splitRange(strandRange, config.chunkSize, config.overlapSize, config.lastChunkExtraLen, true);
                QList<SequenceWalkerSubtask*> complTasks = createSubs(chunks, true, true);
                res+=complTasks;
            }
        }
    }
    return res;
}

QList<SequenceWalkerSubtask*> SequenceWalkerTask::createSubs(const QList<LRegion>& chunks, bool doCompl, bool doAmino) {
    QList<SequenceWalkerSubtask*> res;
    for (int i=0, n = chunks.size(); i<n; i++) {
        const LRegion& chunk = chunks[i];
        bool lo = config.overlapSize > 0 && i > 0;
        bool ro = config.overlapSize > 0 && i+1 < n;
        SequenceWalkerSubtask* t = new SequenceWalkerSubtask(this, chunk, lo, ro, config.seq + chunk.startPos, chunk.len, doCompl, doAmino);
        res.append(t);
    }
    return res;
}

QList<LRegion> SequenceWalkerTask::splitRange(const LRegion& range, int chunkSize, int overlapSize, 
                                              int lastChunkExtraLen, bool reverseMode) 
{
    assert(chunkSize >= overlapSize);
    int stepSize = chunkSize - overlapSize;
    
    QList<LRegion> res;
    for (int pos = range.startPos, end = range.endPos(), lastPos = range.startPos; lastPos < end; pos+=stepSize) {
        int chunkLen = qMin(pos + chunkSize, end) - pos;
        if (end - chunkLen - pos <= lastChunkExtraLen) {
            chunkLen = end-pos;
        }
        lastPos = pos+chunkLen;
        res.append(LRegion(pos, chunkLen));
    }

    if (reverseMode) {
        QList<LRegion> revertedRegions;
        foreach(const LRegion& r, res) {
            LRegion rr(range.startPos + (range.endPos() - r.endPos()), r.len);
            revertedRegions.prepend(rr);
        }
        res = revertedRegions;
    }
    return res;
}

//////////////////////////////////////////////////////////////////////////
// subtask
SequenceWalkerSubtask::SequenceWalkerSubtask(SequenceWalkerTask* _t, const LRegion& glob, bool lo, bool ro, const char* _seq, int _len, bool _doCompl, bool _doAmino)
: Task(tr("sequence_walker_subtask"), TaskFlag_None), 
t(_t), globalRegion(glob), localSeq(_seq), originalLocalSeq(_seq),
localLen(_len), originalLocalLen(_len), doCompl(_doCompl), doAmino(_doAmino),
leftOverlap(lo), rightOverlap(ro)
{
    tpm = Task::Progress_Manual;
    
    // get resources
    QList< TaskResourceUsage > resources = t->getCallback()->getResources( this );
    foreach( const TaskResourceUsage & resource, resources ) {
        taskResources.append( resource );
    }
}

const char* SequenceWalkerSubtask::getRegionSequence() {
    if (needLocalRegionProcessing()) {
        prepareLocalRegion();
    }
    return localSeq;
}

int SequenceWalkerSubtask::getRegionSequenceLen() {
    if (needLocalRegionProcessing()) {
        prepareLocalRegion();
    }
    return localLen;
}

void SequenceWalkerSubtask::prepareLocalRegion() {
    assert(doAmino || doCompl);

    QByteArray res(localSeq, localLen);
    if (doCompl) {
        //do complement;
        assert(t->getConfig().complTrans!=NULL);
        const QByteArray& complementMap = t->getConfig().complTrans->getOne2OneMapper();
        TextUtils::translate(complementMap, res.data(), res.length());
        TextUtils::reverse(res.data(), res.length());
    }
    if (doAmino) {
        assert(t->getConfig().aminoTrans!=NULL && t->getConfig().aminoTrans->isThree2One());
        t->getConfig().aminoTrans->translate(res.data(), res.length(), res.data(), res.length());
        int newLen = res.length()/3;
        res.resize(newLen);
    }
    processedSeqImage = res;
    localLen = processedSeqImage.size();
    localSeq = processedSeqImage.constData();
}

void SequenceWalkerSubtask::run() {
    assert(!t->hasErrors());
    t->getCallback()->onRegion(this, stateInfo);
}

bool SequenceWalkerSubtask::intersectsWithOverlaps(const LRegion& reg) const {
    int overlap = getGlobalConfig().overlapSize;
    if (overlap == 0) {
        return false;
    }
    bool intersects = false;
    if (leftOverlap) {
        intersects = reg.intersects(LRegion(globalRegion.startPos, overlap));
    } 
    if (!intersects && rightOverlap) {
        intersects = reg.intersects(LRegion(globalRegion.endPos() - overlap, overlap));
    }
    return intersects;
}

} //namespace
