/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowIOTasks.h"

#include "SchemaSerializer.h"

#include <core_api/Log.h>
#include <core_api/L10n.h>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>

#include <QtCore/QFile>
#include <QtXml/qdom.h>

namespace GB2 {
using namespace Workflow;

static LogCategory log(ULOG_CAT_WD);


/************************************
* WorkflowIOUtils
************************************/
QDomDocument WorkflowIOUtils::schema2DomDocument( Schema * schema, const Metadata & meta ) {
    assert( NULL != schema );
    QDomDocument xmlDoc(SchemaSerializer::WORKFLOW_DOC);
    SchemaSerializer::schema2xml(*schema, xmlDoc);
    QDomElement elem = xmlDoc.documentElement();
    SchemaSerializer::saveIterations(schema->iterations, elem);
    SchemaSerializer::saveMeta(&meta, elem);
    return xmlDoc;
}

/************************************
 * SaveWorkflowTask
 ************************************/
const QString SaveWorkflowTask::SCHEMA_PATHS_SETTINGS_TAG = "workflow_settings/schema_paths";

SaveWorkflowTask::SaveWorkflowTask(Schema* schema, const Metadata& meta)
: Task(tr("Saving schema to file"), TaskFlag_None), url(meta.url) 
{
    if (schema) {
        rawData = WorkflowIOUtils::schema2DomDocument( schema, meta ).toByteArray();
    }
    
    // add ( name, path ) pair to settings. need for running schemas in cmdline by name
    Settings * settings = AppContext::getSettings();
    assert( settings != NULL );
    QVariantMap pathsMap = settings->getValue( SCHEMA_PATHS_SETTINGS_TAG ).toMap();
    pathsMap.insertMulti( meta.name, meta.url );
    settings->setValue( SCHEMA_PATHS_SETTINGS_TAG, pathsMap );
}

void SaveWorkflowTask::run() {
    log.details(tr("Saving schema to file: %1").arg(url));

    QFile f(url);
    if (f.open(QIODevice::WriteOnly | QIODevice::Truncate)) {
        qint64 s = f.write(rawData);
        f.close();
        if (s != rawData.size()) {
            stateInfo.setError(L10N::errorWritingFile(url));
        }
    } else {
        stateInfo.setError(L10N::errorOpeningFileWrite(url));
    }
}

/************************************
* LoadWorkflowTask
************************************/
LoadWorkflowTask::LoadWorkflowTask(Schema* s, Workflow::Metadata* meta, const QString& url):
Task(tr("Loading schema"), TaskFlag_None),
    url(url), schema(s), meta(meta) {
        xml = new QDomDocument();
}

void LoadWorkflowTask::run() {
    log.details(tr("Loading schema from file: %1").arg(url));

    QFile f(url);
    if (!f.open(QIODevice::ReadOnly)) {
        stateInfo.setError(  L10N::errorOpeningFileRead(url) );
        return;
    }
    QByteArray  xmlData = f.readAll();
    f.close();

    bool res = xml->setContent(xmlData);
    if (!res || xml->doctype().name() != SchemaSerializer::WORKFLOW_DOC) {
        stateInfo.setError( tr("Invalid content: %1").arg(url) );
        xml->clear();
    }
}

Task::ReportResult LoadWorkflowTask::report() {
    if (!stateInfo.hasErrors()) {
        stateInfo.setError(SchemaSerializer::xml2schema(xml->documentElement(), schema, remapping));
        if (stateInfo.hasErrors()) {
            schema->reset();
            if (meta) meta->reset();
        } else {
            if (meta) {
                SchemaSerializer::readMeta(meta, xml->documentElement());
                meta->url = url;
            }
            SchemaSerializer::readIterations(schema->iterations, xml->documentElement(), remapping);
        }
    }
    delete xml;

    return Task::ReportResult_Finished;
}

}//namespace
