/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include "DirectSocketRemoteTaskServer.h"
#include "DirectSocketRemoteMachine.h"
#include "DirectSocketScanner.h"

namespace GB2 {

const QString DirectSocketScanner::DIRECT_SOCKET_SCANNER_MSG = "scanner_request_msg ";

DirectSocketScanner::DirectSocketScanner() : scanNumber( 0 ) {
    udpSocket.bind( SCANNER_UDP_PORT, QUdpSocket::ShareAddress );
    connect( &udpSocket, SIGNAL( readyRead() ), SLOT( sl_readyRead() ) );
    
    // only ipv4 here!
    QList< QNetworkInterface > allInterfaces = QNetworkInterface::allInterfaces();
    foreach( const QNetworkInterface & ni, allInterfaces ) {
        addrEntries << ni.addressEntries();
    }
}

DirectSocketScanner::~DirectSocketScanner() {
    cleanup();
}

void DirectSocketScanner::cleanup() {
    qDeleteAll( scannedMachineSettings );
    scannedMachineSettings.clear();
}

void DirectSocketScanner::startScan() {
    cleanup();
    QByteArray data = ( DIRECT_SOCKET_SCANNER_MSG + QString::number( ++scanNumber ) ).toAscii();
    
    foreach( const QNetworkAddressEntry & entry, addrEntries ) {
        QHostAddress broadcastAddr = entry.broadcast();
        if( broadcastAddr.isNull() ) {
            continue;
        }
        udpSocket.writeDatagram( data, broadcastAddr, DirectSocketRemoteTaskServer::DIRECT_SOCKET_UDP_PORT );
    }
}

void DirectSocketScanner::sl_readyRead() {
    while( udpSocket.hasPendingDatagrams() ) {
        QByteArray data;
        data.resize( udpSocket.pendingDatagramSize() );
        
        QHostAddress addr;
        int ret = udpSocket.readDatagram( data.data(), data.size(), &addr );
        if( -1 == ret || !QString( data ).startsWith( DIRECT_SOCKET_SCANNER_MSG ) ) {
            continue;
        }
        
        QStringList words = QString( data ).split( " ", QString::SkipEmptyParts );
        if( 3 != words.size() || words[1].toInt() != scanNumber ) {
            continue;
        }
        
        bool ok = false;
        int remoteMachineTcpPort = words.at( 2 ).toInt( &ok );
        remoteMachineTcpPort = ok ? remoteMachineTcpPort : DirectSocketRemoteTaskServer::DIRECT_SOCKET_TCP_SERVER_DEFAULT_PORT;
        
        if( filterLocalHost && isLocalhostAddr( addr ) ) {
            continue;
        }
        
        QMutexLocker locker( &mtx );
        scannedMachineSettings << new DirectSocketRemoteMachineSettings( addr.toString(), remoteMachineTcpPort );
    }
}

QList< RemoteMachineSettings * > DirectSocketScanner::takeScanned() {
    QMutexLocker locker( &mtx );
    QList< RemoteMachineSettings * > ret = scannedMachineSettings;
    scannedMachineSettings.clear();
    return ret;
}

bool DirectSocketScanner::isLocalhostAddr( const QHostAddress & addr ) const {
    foreach( const QNetworkAddressEntry & entry, addrEntries ) {
        if( entry.ip() == addr ) {
            return true;
        }
    }
    return false;
}

} // GB2
