/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ImportAnnotationsFromCSVDialog.h"
#include "ImportAnnotationsFromCSVTask.h"
#include "CSVColumnConfigurationDialog.h"

#include <core_api/L10n.h>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>

#include <script/ScriptEditorDialog.h>

#include <gobjects/AnnotationTableObject.h>

#include <util_gui/DialogUtils.h>
#include <util_gui/SaveDocumentGroupController.h>

#include <QtCore/QFileInfo>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#include <memory>

//TODO: add complement token configuration
//TODO: autodetect numeric columns, propose using them as start/end/length positions
//TODO: option for negative len -> complement

namespace GB2 {

// TODO: unify with plugin settings
#define SETTINGS_ROOT       QString("dna_export/import_annotations_from_csv/")
#define A_NAME              QString("annotation_name")
#define T_SEPARATOR         QString("token_separator")
#define SKIP_LINES_COUNT    QString("skip_lines_count")
#define SKIP_LINES_PREFIX    QString("skip_lines_prefix")

ImportAnnotationsFromCSVDialog::ImportAnnotationsFromCSVDialog(QWidget* w) 
: QDialog(w)
{
	setupUi(this);
    connect(readFileButton, SIGNAL(clicked()), SLOT(sl_readFileClicked()));
    connect(previewButton, SIGNAL(clicked()), SLOT(sl_previewClicked()));
	connect(guessButton, SIGNAL(clicked()), SLOT(sl_guessSeparatorClicked()));
    connect(scriptButton, SIGNAL(clicked()), SLOT(sl_scriptSeparatorClicked()));
    connect(separatorEdit, SIGNAL(textChanged(const QString&)), SLOT(sl_separatorChanged(const QString&)));
    connect(prefixToSkipEdit, SIGNAL(textChanged(const QString&)), SLOT(sl_prefixToSkipChanged(const QString&)));
	connect(previewTable, SIGNAL(itemClicked(QTableWidgetItem*)), SLOT(sl_tableItemClicked(QTableWidgetItem*)));
	connect(previewTable->horizontalHeader(), SIGNAL(sectionClicked(int)), SLOT(sl_tableHeaderClicked(int)));
    connect(columnSeparatorRadioButton, SIGNAL(toggled(bool)), SLOT(sl_separatorRadioToggled(bool)));
    connect(scriptRadioButton, SIGNAL(toggled(bool)), SLOT(sl_scriptRadioToggled(bool)));
    
	SaveDocumentGroupControllerConfig conf;
    conf.dfc.addFlagToSupport(DocumentFormatFlag_SupportWriting);
    conf.dfc.supportedObjectTypes += GObjectTypes::ANNOTATION_TABLE;
    conf.fileDialogButton = saveFileButton;
    conf.fileNameEdit = saveFileName;
    conf.saveTitle = tr("Save imported annotations to");
    conf.parentWidget = this;
    conf.formatCombo = saveFormatCombo;

    saveGroupController = new SaveDocumentGroupController(conf, this);
    
    sl_separatorChanged(separatorEdit->text());
    sl_prefixToSkipChanged(prefixToSkipEdit->text());


    //restore last settings
    QString lastName = AppContext::getSettings()->getValue(SETTINGS_ROOT + A_NAME).toString();
    if (!lastName.isEmpty()) {
        defaultNameEdit->setText(lastName);
    }
    
    QString separator = AppContext::getSettings()->getValue(SETTINGS_ROOT + T_SEPARATOR).toString();
    if (!separator.isEmpty()) {
        separatorEdit->setText(separator);
    }

    int skipLinesCount = AppContext::getSettings()->getValue(SETTINGS_ROOT + SKIP_LINES_COUNT).toInt();
    linesToSkipBox->setValue(skipLinesCount);

    QString skipPrefix = AppContext::getSettings()->getValue(SETTINGS_ROOT + SKIP_LINES_PREFIX).toString();
    if (!skipPrefix.isEmpty()) {
        prefixToSkipEdit->setText(skipPrefix);
    }

	scriptHeader = tr("//The script parses input line\n // and returns an array of parsed elements as the result\nvar %1; //input line\nvar %2; //parsed line number")
		.arg(ReadCSVAsAnnotationsTask::LINE_VAR).arg(ReadCSVAsAnnotationsTask::LINE_NUM_VAR);
    
    columnSeparatorRadioButton->setChecked(true);
    sl_scriptRadioToggled(false);
}

void ImportAnnotationsFromCSVDialog::accept() {
	//check that position is OK
	int endPos = 0;
	int startPos = 0;
	int length = 0;
    int names = 0;
	foreach (const ColumnConfig& conf, columnsConfig) {
		switch(conf.role) {
			case ColumnRole_EndPos: endPos++; break;
			case ColumnRole_StartPos: startPos++; break;
			case ColumnRole_Length: length++; break;
            case ColumnRole_Name: names++; break;
            case ColumnRole_Ignore: break;
            case ColumnRole_Qualifier: break;
		}
	}
	if (endPos + startPos + length != 2  || endPos > 1 || startPos > 1 || length > 1) {
		QMessageBox::critical(this, L10N::errorTitle(), tr("Invalid start position/end position/length configuration!"));
		return;
	}
    if (names > 1 ) {
        QMessageBox::critical(this, L10N::errorTitle(), tr("Multiple columns are used as a name!"));
        return;
    }
	if (saveGroupController->getFormatToSave() == NULL) {
		QMessageBox::critical(this, L10N::errorTitle(), tr("Output file format is not set!"));
        saveFormatCombo->setFocus();
		return;
	}
	if (saveGroupController->getSaveFileName().isEmpty()) {
		QMessageBox::critical(this, L10N::errorTitle(), tr("Output file name is not set!"));
        saveFileName->setFocus();
		return;
	}
	if (separatorEdit->text().isEmpty()) {
		QMessageBox::critical(this, L10N::errorTitle(), tr("Column separator is not set!"));
        separatorEdit->setFocus();
		return;
	}
    if (!Annotation::isValidAnnotationName(defaultNameEdit->text())) {
        QMessageBox::critical(this, L10N::errorTitle(), tr("Invalid default annotation name!"));
        defaultNameEdit->setFocus();
        return;
    }

    //store last settings
    AppContext::getSettings()->setValue(SETTINGS_ROOT + A_NAME, defaultNameEdit->text());
    AppContext::getSettings()->setValue(SETTINGS_ROOT + T_SEPARATOR, separatorEdit->text());
    AppContext::getSettings()->setValue(SETTINGS_ROOT + SKIP_LINES_COUNT, linesToSkipBox->value());
    AppContext::getSettings()->setValue(SETTINGS_ROOT + SKIP_LINES_PREFIX, prefixToSkipEdit->text());

	return QDialog::accept();
}

void ImportAnnotationsFromCSVDialog::toParsingConfig(CSVParsingConfig& config) const {
	config.columns = columnsConfig;
	config.linesToSkip = linesToSkipBox->value();
	config.prefixToSkip = prefixToSkipEdit->text();
	config.keepEmptyParts = !separatorsModeCheck->isChecked();
	config.defaultAnnotationName = defaultNameEdit->text();
	
	config.parsingScript.clear();
    config.splitToken.clear();
    if( columnSeparatorRadioButton->isChecked() ) {
        config.splitToken = separatorEdit->text();
    } else if( scriptRadioButton->isChecked() ) {
        config.parsingScript = parsingScript;
    } else {
        assert(false);
    }
}

void ImportAnnotationsFromCSVDialog::toTaskConfig(ImportAnnotationsFromCSVTaskConfig& config) const {
	assert(result() == QDialog::Accepted);
	toParsingConfig(config.parsingOptions);
	config.csvFile = readFileName->text();
	config.df = saveGroupController->getFormatToSave();
	config.dstFile = saveGroupController->getSaveFileName();
	config.addToProject = addToProjectCheck->isChecked();
}

#define PREVIEW_BUFF_SIZE 8196
QString ImportAnnotationsFromCSVDialog::readFileHeader(const QString& fileName, bool silentFail) {
	IOAdapterId ioId = BaseIOAdapters::url2io(fileName);
	IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioId);
	std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
	bool ok = io->open(fileName, IOAdapterMode_Read);
	if (!ok) {
		if (!silentFail) {
			QMessageBox::critical(this, L10N::errorTitle(), L10N::errorOpeningFileRead(fileName));
			readFileName->setFocus();
		}
		return QString();
	}

	QByteArray previewBuff;
	previewBuff.resize(PREVIEW_BUFF_SIZE);
	qint64 bytesRead = io->readBlock(previewBuff.data(), PREVIEW_BUFF_SIZE);
	if (bytesRead == -1) {
		if (!silentFail) {
			QMessageBox::critical(this, L10N::errorTitle(), L10N::errorReadingFile(GUrl(fileName)));
			readFileName->setFocus();
		}
		return QString();

	}
	assert(bytesRead >= 0);
	previewBuff.resize(bytesRead);
	return QString::fromLocal8Bit(previewBuff);
}

void ImportAnnotationsFromCSVDialog::sl_guessSeparatorClicked() {
	guessSeparator(false);
}

void ImportAnnotationsFromCSVDialog::sl_scriptSeparatorClicked() {
    if (parsingScript.isEmpty()) {
        lastUsedSeparator = separatorEdit->text();
    }
    ScriptEditorDialog d(this, scriptHeader);
	if (!parsingScript.isEmpty()) {
		d.setScriptText(parsingScript);
	} else { //set sample script
		QString l1 = "var firstColumn = ["+ReadCSVAsAnnotationsTask::LINE_NUM_VAR+"];\n";
		QString l2 = "var otherColumns = "+ReadCSVAsAnnotationsTask::LINE_VAR+".split(\" \");\n";
		QString l3 = "result =firstColumn.concat(otherColumns);";
		d.setScriptText(l1 + l2 + l3);
	}

    int rc = d.exec();
    if (rc != QDialog::Accepted) {
        return;
    }
    parsingScript = d.getScriptText();
    separatorEdit->setText(lastUsedSeparator);
}

void ImportAnnotationsFromCSVDialog::guessSeparator(bool silent) {
	QString url = checkInputGroup(silent);
	if (url.isEmpty()) {
		return;
	}
	QString text = readFileHeader(url, silent);
	CSVParsingConfig config;
	toParsingConfig(config);
	QString sep = ReadCSVAsAnnotationsTask::guessSeparatorString(text, config);
	if (sep.isEmpty()) {
		if (!silent) {
			QMessageBox::critical(this, L10N::errorTitle(), tr("Failed to guess separator sequence!"));
			readFileName->setFocus();
		}
		return;
	}
	separatorEdit->setText(sep);
	preview(silent);
}

void ImportAnnotationsFromCSVDialog::sl_separatorChanged(const QString& v) {
    if (v.isEmpty()) {
        separatorLabel->setText(tr("Column separator"));
    } else {
        separatorLabel->setText(tr("Column separator value: [%1], length: %2").arg(v).arg(v.length()));
        if (v.trimmed().length() == 0) {
            separatorsModeCheck->setChecked(true);
        }
    }
}

void ImportAnnotationsFromCSVDialog::sl_prefixToSkipChanged(const QString& v) {
    skipPrefixLabel->setEnabled(!v.isEmpty());
}

void ImportAnnotationsFromCSVDialog::sl_readFileClicked() {
    // show the dialog
    LastOpenDirHelper lod;
    
    lod.url = QFileDialog::getOpenFileName(this, tr("Select CSV file to read"), lod);
    if (lod.url.isEmpty()) {
        return;
    }
    readFileName->setText(lod.url);
    
	// guess separator & show preview
    guessSeparator(true);
}

QString ImportAnnotationsFromCSVDialog::checkInputGroup(bool silent) {
    QString inputFile = readFileName->text();
    if (inputFile.isEmpty()) {
		if (!silent) {
			QMessageBox::critical(this, L10N::errorTitle(), tr("Enter input CSV file name"));
			readFileName->setFocus();
		}
        return QString();
    }
    QFileInfo csvFile(inputFile);
    if (!csvFile.exists() || !csvFile.isFile()) {
		if (!silent) {
			QMessageBox::critical(this, L10N::errorTitle(), L10N::errorFileNotFound(inputFile));
			readFileName->setFocus();
		}
        return QString();
    }

    if (!csvFile.isReadable()) {
		if (!silent) {
			QMessageBox::critical(this, L10N::errorTitle(), L10N::errorOpeningFileRead(inputFile));
			readFileName->setFocus();
		}
        return QString();
    }
    
    if( columnSeparatorRadioButton->isChecked() && separatorEdit->text().isEmpty() ) {
        if (!silent) {
            QMessageBox::critical(this, L10N::errorTitle(), tr("Separator value is not set"));
            separatorEdit->setFocus();
        }
        return QString();
    } else if( scriptRadioButton->isChecked() && parsingScript.isEmpty() ){
        if(!silent) {
            QMessageBox::critical(this, L10N::errorTitle(), tr("Script not set"));
        }
        return QString();
    }
    return csvFile.canonicalFilePath();
}

QString ImportAnnotationsFromCSVDialog::checkOutputGroup() {
    QString outFile = saveGroupController->getSaveFileName();
    if (outFile.isEmpty()) {
        QMessageBox::critical(this, L10N::errorTitle(), tr("Enter output file name"));
        saveFileName->setFocus();
        return QString();
    }
    return outFile;
}

void ImportAnnotationsFromCSVDialog::sl_previewClicked() {
	preview(false);
}

void ImportAnnotationsFromCSVDialog::preview(bool silent) {
    QString fileName = checkInputGroup(silent);
    if ( fileName.isEmpty()) {
		return;
    }
	QString text = readFileHeader(fileName, silent);

	previewTable->clear();
    rawPreview->clear();
    
    CSVParsingConfig parseOptions;
	toParsingConfig(parseOptions);
	assert(!parseOptions.splitToken.isEmpty() || !parseOptions.parsingScript.isEmpty());
    
    rawPreview->setPlainText(text);
    
	int columnCount = 0;
	TaskStateInfo ti;
    QList<QStringList> lines = ReadCSVAsAnnotationsTask::parseLinesIntoTokens(text, parseOptions, columnCount, ti);
	if (ti.hasErrors()) {
		QMessageBox::critical(this, L10N::errorTitle(), ti.getError());
		return;
	}
    prepareColumnsConfig(columnCount);
    columnCount = qMax(columnCount, columnsConfig.size());
    
    previewTable->setRowCount(lines.size());
    previewTable->setColumnCount(columnCount);
    
    for (int column = 0; column < columnCount; column++) {
        QTableWidgetItem* headerItem = createHeaderItem(column);
        previewTable->setHorizontalHeaderItem(column, headerItem);
    }
    for (int row = 0; row < lines.size(); row++) {
        const QStringList& rowData = lines.at(row);
        for (int column = 0; column < rowData.size(); column ++) {
            QString token = rowData.at(column);
            QTableWidgetItem* item = new QTableWidgetItem(token);
            item->setFlags(Qt::ItemIsEnabled);
			previewTable->setItem(row, column, item);
        }
    }
}

void ImportAnnotationsFromCSVDialog::prepareColumnsConfig(int numColumns) {
    while (columnsConfig.size() < numColumns) {
        columnsConfig.append(ColumnConfig());
    }

    while (columnsConfig.size() > numColumns) {
        const ColumnConfig& config = columnsConfig.last();
        if (config.role == ColumnRole_Ignore) {
            columnsConfig.removeLast();
        }
    }
}

QTableWidgetItem* ImportAnnotationsFromCSVDialog::createHeaderItem(int column) const {
    assert(column >=0 && column < columnsConfig.size());
	QString text = getHeaderItemText(column);    
    return new QTableWidgetItem(text);
}

QString ImportAnnotationsFromCSVDialog::getHeaderItemText(int column) const {
    const ColumnConfig& config = columnsConfig.at(column);
    QString text = tr("[ignored]");
	switch (config.role) {
		case ColumnRole_Qualifier:
			text = tr("[qualifier %1]").arg(config.qualifierName);
			break;
        case ColumnRole_Name:
            text = tr("[name]");
            break;
        case ColumnRole_StartPos:
			if (config.startPositionOffset!=0) {
				text = tr("[start position with offset %1]").arg(config.startPositionOffset);
			} else {
				text = tr("[start position]");
			}
			break;
		case ColumnRole_EndPos:
			if (config.endPositionIsInclusive) {
				text = tr("[end position (inclusive)]");
			} else {
				text = tr("[end position]");
			}
			break;
		case ColumnRole_Length:
			text = tr("[length]");
			break;
		default:
			assert(config.role == ColumnRole_Ignore);
	}
	return text;
}

void ImportAnnotationsFromCSVDialog::sl_tableItemClicked(QTableWidgetItem* item) {
	int column = previewTable->column(item);
	configureColumn(column);
}

void ImportAnnotationsFromCSVDialog::sl_tableHeaderClicked(int column) {
	configureColumn(column);
}

void ImportAnnotationsFromCSVDialog::sl_separatorRadioToggled(bool checked) {
    separatorEdit->setEnabled(checked);
    guessButton->setEnabled(checked);
}

void ImportAnnotationsFromCSVDialog::sl_scriptRadioToggled(bool checked) {
    scriptButton->setEnabled(checked);
    if(checked && parsingScript.isEmpty()) {
        sl_scriptSeparatorClicked();
    }
}

void ImportAnnotationsFromCSVDialog::configureColumn(int column) {
	assert(column >= 0 && column < columnsConfig.size());

	const ColumnConfig& config = columnsConfig.at(column);
	CSVColumnConfigurationDialog d(this, config);
	int rc = d.exec(); // TODO: set dialog position close to the header item
	if (rc == QDialog::Accepted) {
		columnsConfig[column] = d.config;
	}
	previewTable->horizontalHeaderItem(column)->setText(getHeaderItemText(column));
}

} //namespace
