/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QMouseEvent>
#include <QtGui/QFileDialog>

#include "DotPlotWidget.h"
#include "DotPlotDialog.h"
#include "DotPlotTasks.h"

#include <util_repeat_finder/RepeatFinderTaskFactory.h>
#include <core_api/AppContext.h>
#include <util_gui/DialogUtils.h>

#include <gobjects/AnnotationTableObject.h>
#include <selection/DNASequenceSelection.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVSingleSequenceWidget.h>

#include <core_api/RepeatFinderTaskFactoryRegistry.h>
#include <util_repeat_finder/RepeatFinderSettings.h>
#include <util_repeat_finder/RepeatFinderTaskFactory.h>

namespace GB2 {

DotPlotWidget::DotPlotWidget(AnnotatedDNAView* dnaView)
    :ADVSplitWidget(dnaView),
    selecting(false),shifting(false),miniMapLooking(false),
    selectionX(NULL),selectionY(NULL),sequenceX(NULL),sequenceY(NULL),
    zoom(1.0f), shiftX(0), shiftY(0),
    minLen(100), identity(100),
    pixMapUpdateNeeded(true), deleteDotPlotFlag(false), dotPlotTask(NULL), pixMap(NULL), miniMap(NULL),
	nearestRepeat(NULL),
	sharedSeqX(NULL), sharedSeqY(NULL)
{
    dotPlotResultsListener = new DotPlotResultsListener();

	// border around view
    w = width() - 2*textSpace;
    h = height() - 2*textSpace;

	Q_ASSERT(dnaView);
    this->dnaView = dnaView;

	initActionsAndSignals();

	// background and foreground colors
    dotPlotBGColor = QColor(240, 240, 255);
    dotPlotDotColor = QColor(0, 0, 0);
	dotPlotNearestRepeatColor = QColor(240, 0, 0);
}

// init menu items, actions and connect signals
void DotPlotWidget::initActionsAndSignals() {

	showSettingsDialogAction = new QAction(tr("Parameters"), this);
	connect(showSettingsDialogAction, SIGNAL(triggered()), SLOT(sl_showSettingsDialog()));

	saveImageAction = new QAction(tr("Save as image"), this);
	connect(saveImageAction, SIGNAL(triggered()), SLOT(sl_showSaveImageDialog()));

	saveDotPlotAction = new QAction(tr("Save"), this);
	connect(saveDotPlotAction, SIGNAL(triggered()), SLOT(sl_showSaveFileDialog()));

	loadDotPlotAction = new QAction(tr("Load"), this);
	connect(loadDotPlotAction, SIGNAL(triggered()), SLOT(sl_showLoadFileDialog()));

	deleteDotPlotAction = new QAction(tr("Remove"), this);
	connect(deleteDotPlotAction, SIGNAL(triggered()), SLOT(sl_showDeleteDialog()));

	saveMenu = new QMenu(tr("Save/Load"));
	Q_ASSERT(saveMenu);

	saveMenu->addAction(saveImageAction);
	saveMenu->addAction(saveDotPlotAction);
	saveMenu->addAction(loadDotPlotAction);

	dotPlotMenu = new QMenu(tr("Dotplot"));
	dotPlotMenu->setIcon(QIcon(":dotplot/images/dotplot.png"));

	dotPlotMenu->addAction(showSettingsDialogAction);
	dotPlotMenu->addMenu(saveMenu);
	dotPlotMenu->addAction(deleteDotPlotAction);

	connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_taskFinished(Task*)));
}

// connect signals to know if user clicks on dotplot sequences
void DotPlotWidget::connectSequenceSelectionSignals() {

	if (!(sequenceX && sequenceY)) {
		return;
	}

	Q_ASSERT(dnaView);

	connect (dnaView, SIGNAL(si_sequenceWidgetRemoved(ADVSequenceWidget*)), SLOT(sl_sequenceWidgetRemoved(ADVSequenceWidget*)));

	foreach (ADVSequenceObjectContext* ctx, dnaView->getSequenceContexts()) {

		Q_ASSERT(ctx);

		if ((ctx == sequenceX)) {
			connect(ctx->getSequenceSelection(), 
				SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
				SLOT(sl_onXSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>& , const QList<LRegion>&)));
		}

		if ((ctx == sequenceY)) {
			connect(ctx->getSequenceSelection(), 
				SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
				SLOT(sl_onYSequenceSelectionChanged(LRegionsSelection*, const QList<LRegion>& , const QList<LRegion>&)));
		}
	}
}

DotPlotWidget::~DotPlotWidget() {

    delete showSettingsDialogAction;
    delete saveImageAction;
    delete saveDotPlotAction;
    delete loadDotPlotAction;
    delete deleteDotPlotAction;
    delete saveMenu;

    delete pixMap;

	if (dotPlotTask) {
		cancelRepeatFinderTask();
	}
	delete dotPlotResultsListener;
}

// called by DotPlotSplitter
void DotPlotWidget::buildPopupMenu(QMenu *m) const {

	Q_ASSERT(m);
    QPoint mapped(mapFromGlobal(QCursor::pos()));

	// build menu only if the mouse cursor is over this dotplot
    if (sequenceX && sequenceY && QRect(0, 0, width(), height()).contains(mapped)) {
		assert(!m->actions().isEmpty());

        QAction *b = *(m->actions().begin());
		m->insertMenu(b, dotPlotMenu);
    }
}

void DotPlotWidget::sl_taskFinished(Task *task) {

	Q_ASSERT(task);
    if ( task != dotPlotTask || task->getState() != Task::State_Finished) {
        return;
    }

	if (!dotPlotResultsListener->stateOk) {
		DotPlotDialogs::tooManyResults();
	}
	dotPlotTask = NULL;
	dotPlotResultsListener->setTask(NULL);

    if (deleteDotPlotFlag) {
        deleteDotPlotFlag = false;
        emit si_removeDotPlot();
        return;
    }

	// build dotplot task finished
    pixMapUpdateNeeded = true;
    update();
}

// tell repeat finder that dotPlotResultsListener will be deleted
// if dotPlotTask is not RF task, nothing happened
void DotPlotWidget::cancelRepeatFinderTask() {

	RepeatFinderTaskFactoryRegistry *tfr = AppContext::getRepeatFinderTaskFactoryRegistry();
	Q_ASSERT(tfr);
	RepeatFinderTaskFactory *factory = tfr->getFactory("");
	Q_ASSERT(factory);
	factory->setRFResultsListener(dotPlotTask, NULL);

	if (!dotPlotTask->isFinished()) {
		dotPlotTask->cancel();
	}
}

void DotPlotWidget::sl_sequenceWidgetRemoved(ADVSequenceWidget* w) {

	bool needed = false;
	foreach (ADVSequenceObjectContext *deleted, w->getSequenceContexts()) {
		if (deleted == sequenceX) {
			sequenceX = NULL;
			needed = true;
		}
		if (deleted == sequenceY) {
			sequenceY = NULL;
			needed = true;
		}
	}

	if (needed) {
		deleteDotPlotFlag = true;
		if (dotPlotTask) {
			cancelRepeatFinderTask();
		}
		else {
			addCloseDotPlotTask();
		}
	}
}


// click on the X sequence view
void DotPlotWidget::sl_onXSequenceSelectionChanged(LRegionsSelection* s, const QList<LRegion>& added, const QList<LRegion>& removed) {
	Q_UNUSED(added);
	Q_UNUSED(removed);

    selectionX = s;
    update();
}

// click on the Y sequence view
void DotPlotWidget::sl_onYSequenceSelectionChanged(LRegionsSelection* s, const QList<LRegion>& added, const QList<LRegion>& removed) {
	Q_UNUSED(added);
	Q_UNUSED(removed);

    selectionY = s;
    update();
}

// save dotplot as image
void DotPlotWidget::sl_showSaveImageDialog() {

	LastOpenDirHelper lod("Dotplot");
    lod.url = QFileDialog::getSaveFileName(NULL, tr("Save Dotplot image"), lod.dir, tr("Image Files (*.png *.jpg *.bmp)"));

    if (lod.url.length() <= 0) {
        return; // Cancel button pressed
    }

    QImage image(width(), height(), QImage::Format_RGB32);
    QPainter p(&image);

    drawAll(p);
    image.save(lod.url);
}

// save dotplot into a dotplot file, return true if successful
bool DotPlotWidget::sl_showSaveFileDialog() {

	LastOpenDirHelper lod("Dotplot");
    lod.url = QFileDialog::getSaveFileName(NULL, tr("Save Dotplot"), lod.dir, tr("Dotplot files (*.dpt)"));

    if (lod.url.length() <= 0) {
        return false; // Cancel button pressed
    }

	// check if file opens
    DotPlotDialogs::Errors err = SaveDotPlotTask::checkFile(lod.url);

    switch (err) {
        case DotPlotDialogs::ErrorOpen:
            DotPlotDialogs::fileOpenError(lod.url);
            return false;

        default:
            break;
    }

    TaskScheduler* ts = AppContext::getTaskScheduler();
    if (dotPlotTask) { // Check if there is already some dotPlotTask
        DotPlotDialogs::taskRunning();

        return false;
    }
	Q_ASSERT(dotPlotResultsListener);
	Q_ASSERT(sequenceX);
	Q_ASSERT(sequenceY);

    dotPlotTask = new SaveDotPlotTask(
			lod.url,
			dotPlotResultsListener->dotPlotList,
			sequenceX->getSequenceObject(),
			sequenceY->getSequenceObject(),
			minLen,
			identity
	);
    ts->registerTopLevelTask(dotPlotTask);

    return true;
}

// load dotplot from the dotplot file, return true if successful
bool DotPlotWidget::sl_showLoadFileDialog() {

	LastOpenDirHelper lod("Dotplot");
    lod.url = QFileDialog::getOpenFileName(NULL, tr("Load Dotplot"), lod.dir, tr("Dotplot files (*.dpt)"));

    if (lod.url.length() <= 0) {
        return false; // Cancel button pressed
    }

    if (dotPlotTask) { // Check if there is already some dotPlotTask
        DotPlotDialogs::taskRunning();

        return false;
    }

	Q_ASSERT(sequenceX);
	Q_ASSERT(sequenceY);

	Q_ASSERT(sequenceX->getSequenceObject());
	Q_ASSERT(sequenceY->getSequenceObject());

	// check if the file opens and names of the loading sequences same as names of current sequences
    DotPlotDialogs::Errors err = LoadDotPlotTask::checkFile(
			lod.url,
			sequenceX->getSequenceObject()->getGObjectName(),
			sequenceY->getSequenceObject()->getGObjectName()
	);

    switch (err) {
        case DotPlotDialogs::ErrorOpen:
            DotPlotDialogs::fileOpenError(lod.url);
            return false;

        case DotPlotDialogs::ErrorNames:
            if (DotPlotDialogs::loadDifferent() == QMessageBox::Yes) {
                break; // load dotplot anyway
            }
            else {
                return false;
            }

        default:
            break;
    }

	Q_ASSERT(dotPlotResultsListener);
	Q_ASSERT(dotPlotResultsListener->dotPlotList);

    dotPlotTask = new LoadDotPlotTask(
			lod.url,
			dotPlotResultsListener->dotPlotList,
			sequenceX->getSequenceObject(),
			sequenceY->getSequenceObject(),
			&minLen,
			&identity
	);

    TaskScheduler* ts = AppContext::getTaskScheduler();
    ts->registerTopLevelTask(dotPlotTask);

    pixMapUpdateNeeded = true;
    update();

    return true;
}

// creating new dotplot or changing settings
bool DotPlotWidget::sl_showSettingsDialog() {

    if (dotPlotTask) { // Check if there is already some dotPlotTask
        DotPlotDialogs::taskRunning();

        return false;
    }

	Q_ASSERT(dnaView);
    QList<ADVSequenceObjectContext *> sequences = dnaView->getSequenceContexts();

    if (sequences.size() <= 0) {
        return false;
    }

    DotPlotDialog d(this, sequences, minLen, identity, sequenceX, sequenceY);
    if (d.exec()) {
        setMinimumHeight(200);

		nearestRepeat = NULL;

		if (sequenceX != d.getXSeq() || (sequenceY != d.getYSeq())) {
			resetZooming();
		}

		sequenceX = d.getXSeq();
		sequenceY = d.getYSeq();

		Q_ASSERT(d.minLenBox);
		Q_ASSERT(d.identityBox);

        minLen = d.minLenBox->value();
        identity = d.identityBox->value();

        connectSequenceSelectionSignals();

		Q_ASSERT(dotPlotResultsListener);
		Q_ASSERT(dotPlotResultsListener->dotPlotList);
        dotPlotResultsListener->dotPlotList->clear();

		Q_ASSERT(sequenceX);
		Q_ASSERT(sequenceY);

		if ((sequenceX->getAlphabet()->getType() != sequenceY->getAlphabet()->getType()) || (sequenceX->getAlphabet()->getType() != DNAAlphabet_NUCL)){

            sequenceX = NULL;
            sequenceY = NULL;

            DotPlotDialogs::wrongAlphabetTypes();
            return false;
        }

		Q_ASSERT(sequenceX->getSequenceObject());
		Q_ASSERT(sequenceY->getSequenceObject());

		DNAAlphabet *al = sequenceX->getAlphabet();
		if ((al->getId() == BaseDNAAlphabetIds::NUCL_DNA_DEFAULT) || (al->getId() == BaseDNAAlphabetIds::NUCL_RNA_DEFAULT)) {
			al = sequenceY->getAlphabet();
		}

		sharedSeqX = sequenceX->getSequenceObject()->getSequence();
		sharedSeqY = sequenceY->getSequenceObject()->getSequence();

		RepeatFinderSettings c(
			dotPlotResultsListener,
//			sharedSeqX.data(),
			sequenceX->getSequenceObject()->getSequence().data(),
			sequenceX->getSequenceLen(),
//			sharedSeqY.data(),
			sequenceY->getSequenceObject()->getSequence().data(),
			sequenceY->getSequenceLen(),
			al,
			d.getMinLen(), d.getMismatches(),
			d.getAlgo()
		);

		RepeatFinderTaskFactoryRegistry *tfr = AppContext::getRepeatFinderTaskFactoryRegistry();
		Q_ASSERT(tfr);
		RepeatFinderTaskFactory *factory = tfr->getFactory("");
		Q_ASSERT(factory);
		dotPlotTask = factory->getTaskInstance(c);
		dotPlotResultsListener->setTask(dotPlotTask);

        TaskScheduler* ts = AppContext::getTaskScheduler();
        ts->registerTopLevelTask(dotPlotTask);
    }
    else {
        return false;
    }

    return true;
}

// ask user if he wants to save dotplot first
void DotPlotWidget::sl_showDeleteDialog() {

    int answer = DotPlotDialogs::saveDotPlot();
    bool saveDotPlot;

    switch (answer) {
        case QMessageBox::Cancel:
            return;

        case QMessageBox::Yes:
            saveDotPlot = sl_showSaveFileDialog();
            if (!saveDotPlot) { // cancel button pressed
                return;
            }
            break;

        default:
            break;
    }

    if (!deleteDotPlotFlag) {
		addCloseDotPlotTask();
    }
}

void DotPlotWidget::addCloseDotPlotTask() {

	deleteDotPlotFlag = true;

	Task *t = new Task("Closing dotplot", TaskFlags_NR_FOSCOE);
	if (!dotPlotTask) {
		dotPlotTask = t;
	}

	AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

// dotplot results updated, need to update picture
void DotPlotWidget::pixMapUpdate() {

    if (!pixMapUpdateNeeded || !sequenceX || !sequenceY || dotPlotTask) {
        return;
    }

	if ((sequenceX->getSequenceLen() <= 0) || (sequenceY->getSequenceLen() <= 0)) {
		return;
	}

    float ratioX = w/(float)sequenceX->getSequenceLen();
    float ratioY = h/(float)sequenceY->getSequenceLen();

    delete pixMap;
    pixMap = new QPixmap(w, h);

    QPainter pixp(pixMap);
    pixp.setPen( Qt::NoPen ); // do not draw outline
    pixp.setBrush(dotPlotBGColor);
    pixp.drawRect(0, 0, w, h);

    pixp.setPen(Qt::SolidPattern);
    pixp.setBrush(dotPlotDotColor);

    QLine line;

	Q_ASSERT(dotPlotResultsListener);
	Q_ASSERT(dotPlotResultsListener->dotPlotList);

	// draw to the dotplot picture results
    foreach(const DotPlotResults &r, *dotPlotResultsListener->dotPlotList) {

        if (!getLineToDraw(r, &line, ratioX, ratioY)) {
            continue;
        }

        pixp.drawLine(line);
    }

    pixMapUpdateNeeded = false;
}

// return true if the line intersects with the area to draw
bool DotPlotWidget::getLineToDraw(const DotPlotResults &r, QLine *line, float ratioX, float ratioY) const {

    float x1 = r.x*ratioX * zoom + shiftX;
    float y1 = r.y*ratioY * zoom + shiftY;
    float x2 = x1 + r.len*ratioX * zoom;
    float y2 = y1 + r.len*ratioY * zoom;

    if ((x2 < 0) || (y2 < 0) || (x1 > w) || (y1 > h)) {
        return false;
    }

    if (x1<0) {
        float y_0 = y1 - x1*(y2-y1)/(x2-x1);
        if ((y_0 >= 0) && (y_0 <= h)) {
            x1 = 0;
            y1 = y_0;
        }

    }

    if (x2>w) {
        float y_w = y1 + (w-x1)*(y2-y1)/(x2-x1);
        if ((y_w >= 0) && (y_w <= h)) {
            x2 = w;
            y2 = y_w;
        }

    }

    if (y1<0) {
        float x_0 = x1 - y1*(x2-x1)/(y2-y1);
        if ((x_0 >= 0) && (x_0 <= w)) {
            y1 = 0;
            x1 = x_0;
        }

    }

    if (y2>h) {
        float x_h = x1 + (h-y1)*(x2-x1)/(y2-y1);
        if ((x_h >= 0) && (x_h <= w)) {
            y2 = h;
            x2 = x_h;
        }

    }

    if ((x1 < 0) || (x2 < 0) || (y1 < 0) || (y2 < 0) || (x1 > w) || (y1 > h) || (x2 > w) || (y2 > h)) {
        return false;
    }

	Q_ASSERT(line);

    line->setLine(x1, y1, x2, y2);
    return true;
}

// draw everything
void DotPlotWidget::drawAll(QPainter& p) {

    if (sequenceX == NULL || sequenceY == NULL || w <= 0 || h <= 0) {
        return;
    }

    p.save();
    p.setRenderHint(QPainter::Antialiasing);

    p.setBrush(QBrush(palette().window().color()));
    p.drawRect(0, 0, width(), height());

	Q_ASSERT(sequenceX->getSequenceObject());
	Q_ASSERT(sequenceY->getSequenceObject());

    QString nameX = sequenceX->getSequenceObject()->getGObjectName();
    QString nameY = sequenceY->getSequenceObject()->getGObjectName();

    nameX += " (" + tr("min length")+" "+QString::number(minLen)+", "+tr("identity")+" "+QString::number(identity)+"%)";

    p.drawText(0, height() - textSpace, width(), textSpace, Qt::AlignCenter, nameX);

    p.save();
    p.rotate(90);
    p.translate(0, -width());
    p.drawText(0, 0, height()-textSpace, textSpace, Qt::AlignCenter, nameY);
    p.restore();

    p.translate(textSpace, textSpace);


    drawAxises(p);
    drawDots(p);
    drawSelection(p);
    drawMiniMap(p);
	drawNearestRepeat(p);

    p.translate(-textSpace, -textSpace);
    drawRulers(p);

    p.restore();
}

void DotPlotWidget::drawNearestRepeat(QPainter& p) const {

	if (!nearestRepeat) {
		return;
	}
	p.save();
	p.setPen(dotPlotNearestRepeatColor);

	float ratioX = w/(float)sequenceX->getSequenceLen();
	float ratioY = h/(float)sequenceY->getSequenceLen();

	QLine line;
	if (getLineToDraw(*nearestRepeat, &line, ratioX, ratioY)) {

		p.drawLine(line);
	}

	p.restore();
}

void DotPlotWidget::drawMiniMap(QPainter& p) const {

    if (miniMap) {
        miniMap->draw(p, shiftX, shiftY, zoom);
    }
}

// update dotplot picture if needed and draw it
void DotPlotWidget::drawDots(QPainter& p) {

    pixMapUpdate();

    if (pixMap) {
        p.drawPixmap(0, 0, w, h, *pixMap);
    }
}

void DotPlotWidget::drawAxises(QPainter& p) const {

    QPoint zeroPoint(0, 0);
    QPoint lowLeftPoint(0, h);
    QPoint topRightPoint(w, 0);

    p.drawLine(zeroPoint, lowLeftPoint);
    p.drawLine(zeroPoint, topRightPoint);
}

// get cutted text to draw rulers
QString DotPlotWidget::getRoundedText(QPainter &p, int num, int size) const {

    QRectF rect;
    QString curStr = QString::number(num);

    rect = p.boundingRect(0, 0, size, 100, Qt::AlignLeft | Qt::AlignTop, curStr);
    if (rect.width() < size)
        return curStr;

    curStr = QString::number(num/(float)1000, 'f', 1) + QString("K");
    rect = p.boundingRect(0, 0, size, 100, Qt::AlignLeft | Qt::AlignTop, curStr);
    if (rect.width() < size)
        return curStr;

    curStr = QString::number(num/(float)1000000, 'f', 1) + QString("M");
    rect = p.boundingRect(0, 0, size, 100, Qt::AlignLeft | Qt::AlignTop, curStr);
    if (rect.width() < size)
        return curStr;

    return "";
}

void DotPlotWidget::drawRulers(QPainter& p) const {

    static const int partsStart = 20;
    static const int lineLen = 5;

    int ok_count = 0;
    int parts = partsStart*2;

    while (ok_count <= 1) {
        if (ok_count == 0) {
            parts/=2;
        }
        if (!parts) {
            return;
        }

        int xPart = w/parts;

		// assume that it's enough space for all labels
        ok_count++;
        QString prevTextToDraw = "";

        for (int i=0; i<parts; i++) {
            QPointF coord(i*xPart, 0);
            QPointF inner = sequenceCoords(unshiftedUnzoomed(coord));

            QString textToDraw = getRoundedText(p, inner.x(), xPart - 4);

            if (ok_count == 2) {
                p.drawLine(textSpace + i*xPart, textSpace - lineLen/2, textSpace + i*xPart, textSpace + lineLen/2);
                p.drawText(textSpace/2 + i*xPart, textSpace/2, textToDraw);
            }

			// stop dividing parts if there is not enought space or
			// the previous label equals this label
            if ((textToDraw == "") || (textToDraw == prevTextToDraw)) {
                ok_count--;
                break;
            }
            prevTextToDraw = textToDraw;
        }
    }

    p.save();
    p.rotate(-90);
    p.translate(-height(), 0);

    ok_count = 0;
    parts = partsStart*2;

    while (ok_count < 2) {
        if (ok_count == 0) {
            parts/=2;

            if (!parts) {
                return;
            }
        }

		// same as for X axis
        int yPart = h/parts;

        ok_count++;
        QString prevTextToDraw = "";

        for (int i=0; i<parts; i++) {
            QPointF coord(0, i*yPart);
            QPointF inner = sequenceCoords(unshiftedUnzoomed(coord));

            QString textToDraw = getRoundedText(p, inner.y(), yPart - 4);

            if (ok_count == 2) {
                p.drawLine(textSpace + h - i*yPart, textSpace - lineLen/2, textSpace + h - i*yPart, textSpace + lineLen/2);
                if (textToDraw!="0") {
                    p.drawText(textSpace + h - 25 - i*yPart, textSpace/2, textToDraw);
                }
            }

            if ((textToDraw == "") || (textToDraw == prevTextToDraw)) {
                ok_count--;
                break;
            }
            prevTextToDraw = textToDraw;
        }
    }

    p.restore();
}

// need to draw inside area not containing borders
void DotPlotWidget::drawRectCorrect(QPainter &p, QRectF r) const {

    if ((r.right() < 0) || (r.left() > w) || (r.bottom() < 0) || (r.top() > h)) {
        return;
    }

    if (r.left() < 0) {
        r.setLeft(0);
    }

    if (r.top() < 0) {
        r.setTop(0);
    }

    if (r.right() > w) {
        r.setRight(w);
    }

    if (r.bottom() > h) {
        r.setBottom(h);
    }

    p.drawRect(r);
}

// part of the sequence is selected, show it
void DotPlotWidget::drawSelection(QPainter &p) const {

    if (!sequenceX || !sequenceY) {
        return;
    }

    if (!(selectionX || selectionY)) {
        return;
    }

    p.save();

    QPen pen;
    pen.setStyle(Qt::DotLine);
    pen.setColor(QColor(0, 125, 227, 200));

    p.setPen(pen);
    p.setBrush(QBrush(QColor(200, 200, 200, 100)));

    float xLeft, xLen, yBottom, yLen;
    int xSeqLen = sequenceX->getSequenceLen();
    int ySeqLen = sequenceY->getSequenceLen();

	Q_ASSERT(xSeqLen);
	Q_ASSERT(ySeqLen);

	// for each selected part on the sequence X, highlight selected part on the sequence Y
    if (selectionX) {
        foreach(const LRegion &rx, selectionX->getSelectedRegions()) {

            xLeft = rx.startPos/(float)xSeqLen * w*zoom;
            xLen = rx.len/(float)xSeqLen * w*zoom;

            if (!selectionY || selectionY->getSelectedRegions().size() == 0) {
                yBottom = 0;
                yLen = 1.0f * h*zoom;

                drawRectCorrect(p, QRectF(xLeft + shiftX, yBottom + shiftY, xLen, yLen));
            }
            else {
                foreach(const LRegion &ry, selectionY->getSelectedRegions()) {
                    yBottom = ry.startPos/(float)ySeqLen * h*zoom;
                    yLen = ry.len/(float)ySeqLen * h*zoom;

                    drawRectCorrect(p, QRectF(xLeft + shiftX, yBottom + shiftY, xLen, yLen));
                }
            }
        }
    }

	// user selected only part of the Y sequence
    if ((!selectionX || selectionX->getSelectedRegions().size() == 0) && (selectionY && selectionY->getSelectedRegions().size() != 0)) {
        xLeft = 0;
        xLen = 1.0f * w*zoom;

        foreach(const LRegion &ry, selectionY->getSelectedRegions()) {
            yBottom = ry.startPos/(float)ySeqLen * h*zoom;
            yLen = ry.len/(float)ySeqLen * h*zoom;

            drawRectCorrect(p, QRectF(xLeft + shiftX, yBottom + shiftY, xLen, yLen));
        }
    }

    p.restore();
}

// shifted camera or changed zoom
void DotPlotWidget::checkShift() {

    if (shiftX > 0) {
        shiftX = 0;
    }

    if (shiftY > 0) {
        shiftY = 0;
    }

    if (shiftX < (width()-2*textSpace)*(1-zoom)) {
        shiftX = (width()-2*textSpace)*(1-zoom);
    }

    if (shiftY < (height()-2*textSpace)*(1-zoom)) {
        shiftY = (height()-2*textSpace)*(1-zoom);
    }
}

// new zoom should convert mouse cursor to the same dotplot point as the old zoom
void DotPlotWidget::calcZooming(float oldzoom, float newzoom, const QPoint &cursor) {

    if (dotPlotTask || (w<=0) || (h<=0)) {
        return;
    }

	if (!(sequenceX && sequenceY)) {
		return;
	}

	// cursor coords excluding dotplot border
    QPoint inner = toInnerCoords(cursor.x(), cursor.y());

	float minSeqLen;
	if (sequenceX->getSequenceLen()/(float)w < sequenceY->getSequenceLen()/(float)h) {
		minSeqLen = sequenceX->getSequenceLen()/(float)w;
	}
	else {
		minSeqLen = sequenceY->getSequenceLen()/(float)h;
	}
    minSeqLen*=10;

	// limit maximum zoom
    if (newzoom > minSeqLen) {
        newzoom = minSeqLen;
    }
	// dotplot has no zooming and the user tries zoom out
    if (newzoom < 1.0f) {
        newzoom = 1.0f;
    }

    float xi = (inner.x() - shiftX)/oldzoom;
    float yi = (inner.y() - shiftY)/oldzoom;

    shiftX = inner.x() - xi*newzoom;
    shiftY = inner.y() - yi*newzoom;

    if (zoom != newzoom) {
        pixMapUpdateNeeded = true;
        update();
    }
    zoom = newzoom;
    checkShift();
}

// translate visible coords to the sequence coords
QPoint DotPlotWidget::sequenceCoords(const QPointF &c) const {

	Q_ASSERT(sequenceX);
	Q_ASSERT(sequenceY);

    int xLen = sequenceX->getSequenceLen();
    int yLen = sequenceY->getSequenceLen();

	Q_ASSERT(w>0);
	Q_ASSERT(h>0);

    int innerX = (c.x() * xLen)/w;
    int innerY = (c.y() * yLen)/h;

    return QPoint(innerX, innerY);
}

// select sequences using sequence coords
void DotPlotWidget::sequencesCoordsSelection(const QPointF &start, const QPointF &end) {

    float startX = start.x();
    float startY = start.y();
    float endX = end.x();
    float endY = end.y();

    if (sequenceX == sequenceY) { // choose the X coords selection
        endY = endX;
    }

    if (endX < startX) {
        float tmp = endX;
        endX = startX;
        startX = tmp;
    }
    if (endY < startY) {
        float tmp = endY;
        endY = startY;
        startY = tmp;
    }

	Q_ASSERT(dnaView);
    foreach (ADVSequenceWidget *w, dnaView->getSequenceWidgets()) {

		Q_ASSERT(w);
        foreach (ADVSequenceObjectContext *s, w->getSequenceContexts()) {
			Q_ASSERT(s);

            if ( ((int)(endX-startX) > 0) && (s == sequenceX)) {
                s->getSequenceSelection()->clear();
                s->getSequenceSelection()->addRegion(LRegion(startX, endX-startX));

                w->centerPosition(startX);
            }

            if ( ((int)(endY-startY) > 0) && (s == sequenceY)) {
                s->getSequenceSelection()->clear();
                s->getSequenceSelection()->addRegion(LRegion(startY, endY-startY));

                w->centerPosition(startY);
            }
        }
    }

    update();
}

// select sequences with mouse
void DotPlotWidget::sequencesMouseSelection(const QPointF &zoomedA, const QPointF &zoomedB) {

    if (!(sequenceX || sequenceY)) {
        return;
    }

    if (zoomedA == zoomedB) {
        selectionX = NULL;
        selectionY = NULL;

        return;
    }

    QPointF a(unshiftedUnzoomed(zoomedA));
    QPointF b(unshiftedUnzoomed(zoomedB));

    QPointF start = sequenceCoords(a);
    QPointF end = sequenceCoords(b);

    sequencesCoordsSelection(start, end);
}

// get mouse coords, select the nearest found repeat
void DotPlotWidget::selectNearestRepeat(const QPointF &p) {

    QPoint seqCoords = sequenceCoords(unshiftedUnzoomed(p));

	nearestRepeat = findNearestRepeat(seqCoords);
    if (!nearestRepeat) {
        return;
    }

	// There is only one sequence view, can't select two places there
	if (sequenceX != sequenceY) {
		sequencesCoordsSelection(
			QPoint(nearestRepeat->x, nearestRepeat->y),
			QPoint(nearestRepeat->x + nearestRepeat->len, nearestRepeat->y + nearestRepeat->len)
		);
	}

	foreach (ADVSequenceWidget *w, dnaView->getSequenceWidgets()) {
		Q_ASSERT(w);
		foreach (ADVSequenceObjectContext *s, w->getSequenceContexts()) {
			Q_ASSERT(s);

			if (s == sequenceX) {
				w->centerPosition(nearestRepeat->x);
			}
		}
	}
}

// get sequence coords, return sequence coords of the nearest repeat
const DotPlotResults* DotPlotWidget::findNearestRepeat(const QPoint &p) const {

    const DotPlotResults* need = NULL;
    float minD = 0;

    float x = p.x();
    float y = p.y();

	Q_ASSERT(sequenceX);
	Q_ASSERT(sequenceY);

	if ((sequenceX->getSequenceLen() <= 0) || (sequenceY->getSequenceLen() <= 0)) {
		return NULL;
	}

    float ratioX = w/(float)sequenceX->getSequenceLen();
    float ratioY = h/(float)sequenceY->getSequenceLen();

    ratioX*=ratioX;
    ratioY*=ratioY;

    bool first = true;

	Q_ASSERT (dotPlotResultsListener);
    foreach (const DotPlotResults &r, *dotPlotResultsListener->dotPlotList) {

        float halfLen = r.len/(float)2;
        float midX = r.x + halfLen;
        float midY = r.y + halfLen;

		// square of the distance between two points. ratioX and ratioY are squared
        float d = (x-midX)*(x-midX)*ratioX + (y-midY)*(y-midY)*ratioY;

        if (d < minD || first) {
            minD = d;
            need = &r;
        }

        first = false;
    }
    return need;
}

// get mouse coords, return coords in the area without border
QPoint DotPlotWidget::toInnerCoords(int x, int y) const {

    x = x - textSpace;
    y = y - textSpace;

    if (x > w) {
        x = w;
    }
    if (y > h) {
        y = h;
    }

    if (x < 0) {
        x = 0;
    }
    if (y < 0) {
        y = 0;
    }

    return QPoint(x, y);
}

void DotPlotWidget::paintEvent(QPaintEvent *e) {

    QWidget::paintEvent(e);

    QPainter p(this);
    drawAll(p);
}

void DotPlotWidget::resizeEvent(QResizeEvent *e) {

	Q_ASSERT(e);

    if (e->oldSize() == e->size()) {
        return;
    }

    int oldw = e->oldSize().width() - 2*textSpace;
    int oldh = e->oldSize().height() - 2*textSpace;

    w = e->size().width() - 2*textSpace;
    h = e->size().height() - 2*textSpace;

	// update shift when resizing
    if (pixMap && (oldw > 0) && (oldh > 0)) {
        shiftX *= w/(float)(oldw);
        shiftY *= h/(float)(oldh);
    }

    delete miniMap;
    miniMap = new DotPlotMiniMap(w, h, 10);

    pixMapUpdateNeeded = true;
}

// zoom in/zoom out
void DotPlotWidget::wheelEvent(QWheelEvent *e) {

	Q_ASSERT(e);
	if (dotPlotTask) {
		return;
	}

    QPointF oldCoords = toInnerCoords(e->pos().x(), e->pos().y());

    float oldzoom = zoom;
    float newzoom = zoom*(1 + e->delta()/(float)1000);

    calcZooming(oldzoom, newzoom, e->pos());
}

void DotPlotWidget::resetZooming() {

    calcZooming(zoom, 1.0f, QPoint(w/2, h/2));
}

// user clicked on the minimap
void DotPlotWidget::miniMapShift() {

	Q_ASSERT(miniMap);

    QPointF fromMiniMap = miniMap->fromMiniMap(clickedSecond, zoom);
    shiftX = -fromMiniMap.x();
    shiftY = -fromMiniMap.y();
    checkShift();
    pixMapUpdateNeeded = true;
    update();
}

void DotPlotWidget::mousePressEvent(QMouseEvent *e) {

	Q_ASSERT(e);

    QWidget::mousePressEvent(e);

	if (dotPlotTask) {
		return;
	}

    clickedFirst = toInnerCoords(e->pos().x(), e->pos().y());
    clickedSecond = clickedFirst;

	// selecting sequences or using minimap
    if (e->button() == Qt::LeftButton) {

        if (miniMap && miniMap->getBoundary().contains(clickedFirst)) { // click on the miniMap
            miniMapLooking = true;
            miniMapShift();

            return;
        }

        selecting = true;
    }

	// shifting dotplot view
    if (e->button() == Qt::MidButton) {
        shifting = true;
    }
}

// return real coords on the dotplot
QPointF DotPlotWidget::unshiftedUnzoomed(const QPointF &p) const {

	Q_ASSERT(zoom>0);

    return QPointF((p.x() - shiftX)/zoom, (p.y() - shiftY)/zoom);
}

void DotPlotWidget::mouseMoveEvent(QMouseEvent *e) {

	Q_ASSERT(e);

    QWidget::mouseMoveEvent(e);

	if (dotPlotTask) {
		return;
	}

    clickedSecond = toInnerCoords(e->pos().x(), e->pos().y());

    if (miniMapLooking) {
        miniMapShift();

        return;
    }

    if (selecting) {
        if ((clickedFirst.x() != clickedSecond.x()) && (clickedFirst.y() != clickedSecond.y())) {
            sequencesMouseSelection(clickedFirst, clickedSecond);
        }
    }

    if (shifting) {
        shiftX += (clickedSecond.x() - clickedFirst.x());
        shiftY += (clickedSecond.y() - clickedFirst.y());

        clickedFirst = toInnerCoords(e->pos().x(), e->pos().y());
        checkShift();
        pixMapUpdateNeeded = true;
        update();
    }
}

void DotPlotWidget::mouseReleaseEvent(QMouseEvent *e) {

	Q_ASSERT(e);

	if (dotPlotTask) {
		return;
	}

    QWidget::mouseReleaseEvent(e);

    if (e->button() == Qt::LeftButton) {
        selecting = false;
        miniMapLooking = false;

		// mouse click
        if (clickedFirst == clickedSecond) {
            selectNearestRepeat(clickedFirst);
        }
    }

    if (e->button() == Qt::MidButton) {
        shifting = false;
    }

    update();
}


} // namespace
