/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>
#include <core_api/Log.h>
#include <util_gui/DialogUtils.h>

#include "GenomeAlignerWorker.h"


/* TRANSLATOR GB2::LocalWorkflow::GenomeAlignerWorker */

namespace GB2 {
namespace LocalWorkflow {

const QString GenomeAlignerWorkerFactory::ACTOR_ID("genome.assembly");
const QString MISMATCHES_ATTR("mismatches");
const QString REFSEQ_URL_ATTR("ref-seq-url");
const QString REVERSE_ATTR("reverse");

static LogCategory log(ULOG_CAT_WD);

void GenomeAlignerWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    Descriptor ind(CoreLibConstants::IN_PORT_ID, GenomeAlignerWorker::tr("Short read sequences"), GenomeAlignerWorker::tr("Short reads to be aligned."));
    Descriptor oud(CoreLibConstants::OUT_PORT_ID, GenomeAlignerWorker::tr("Short reads alignment"), GenomeAlignerWorker::tr("Result of alignment."));
    p << new PortDescriptor(ind, BioDataTypes::DNA_SEQUENCE_TYPE(), true /*input*/);
    p << new PortDescriptor(oud, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), false /*input*/, true /*multi*/);
    Descriptor refseq(REFSEQ_URL_ATTR, GenomeAlignerWorker::tr("Reference"), 
         GenomeAlignerWorker::tr("Reference sequence url. The short reads will be aligned to this genome."));
    Descriptor mismatches(MISMATCHES_ATTR, GenomeAlignerWorker::tr("Mismatches"), 
        GenomeAlignerWorker::tr("Number of mismatches allowed while aligning reads."));
    Descriptor reverse(REVERSE_ATTR, GenomeAlignerWorker::tr("Align reversed"), 
        GenomeAlignerWorker::tr("Set this option to align both direct and reverse reads."));

    a << new Attribute(refseq, CoreDataTypes::STRING_TYPE(), true /*required*/, QString());
    a << new Attribute(mismatches, CoreDataTypes::NUM_TYPE(), false, 0);
    a << new Attribute(reverse, CoreDataTypes::BOOL_TYPE(), false/*required*/, true);    

    Descriptor desc(ACTOR_ID, GenomeAlignerWorker::tr("UGENE genome aligner"), 
        GenomeAlignerWorker::tr("Unique UGENE algorithm for aligning short reads to reference genome"));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;    

    delegates[REFSEQ_URL_ATTR] = new URLDelegate(DialogUtils::prepareDocumentsFileFilter(true), QString(), true);
    {
        QVariantMap m; m["minimum"] = 0; m["maximum"] = 3;
        delegates[MISMATCHES_ATTR] = new SpinBoxDelegate(m);
    }

    proto->setEditor(new DelegateEditor(delegates));
    proto->setPrompter(new GenomeAlignerPrompter());
    proto->setIconPath(":core/images/align.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_ASSEMBLY(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new GenomeAlignerWorkerFactory());
}

QString GenomeAlignerPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(CoreLibConstants::IN_PORT_ID);
    QString producerName = producer ? tr(" from <u>%1</u>").arg(producer->getLabel()) : "";
    QString refSeqUrl = getParameter(REFSEQ_URL_ATTR).toString();
    QString refSeq = (refSeqUrl.isEmpty() ? "" : QString("<u>%1</u>").arg(GUrl(refSeqUrl).fileName()) );

    QString doc = tr("Align short reads %1 to the reference genome %2 and send it to output.")
        .arg(producerName).arg(refSeq);

    return doc;
}

void GenomeAlignerWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    settings.refSeqUrl = actor->getParameter(REFSEQ_URL_ATTR)->getAttributeValue<QString>();
    int nMismatches = actor->getParameter(MISMATCHES_ATTR)->getAttributeValue<int>();
	settings.setCustomValue(GenomeAlignerTask::OPTION_MISMATCHES, nMismatches);
	bool alignReverse = actor->getParameter(REVERSE_ATTR)->getAttributeValue<bool>();
    settings.setCustomValue(GenomeAlignerTask::OPTION_ALIGN_REVERSED, alignReverse);
}

bool GenomeAlignerWorker::isReady() {
    return (input && input->hasMessage());
}

Task* GenomeAlignerWorker::tick() {
    while (!input->isEnded()) {
       DNASequence read = input->get().getData().value<DNASequence>();
       log.trace(QString("Loaded short read %1").arg(read.getName())); 
       settings.shortReads.append(read);
    }
    
    if( settings.shortReads.isEmpty() ) {
         log.error( tr("Short reads list is empty.") );
         return NULL;
    }

    Task* t = new GenomeAlignerTask(settings);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void GenomeAlignerWorker::sl_taskFinished() {
    GenomeAlignerTask* t = qobject_cast<GenomeAlignerTask*>(sender());
    if (t->getState() != Task::State_Finished) {
        return;
    }
    
    QVariant v = qVariantFromValue<MAlignment>(t->getResult());
    output->put(Message(BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.trace(tr("Genome aligner finished. Result name is %1").arg(t->getResult().getName()));
}

bool GenomeAlignerWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
